(in-package :cl-sdm-tests)

(in-suite cl-sdm-test-suite)

(test trimming-whitespace
  (let ((str "  hello, world!
        ")
        (expected "hello, world!"))
    (setf str (concatenate 'string str (string #\Page)))
    (is-true (string= (sdm:trim-whitespace str) expected)))

  (let ((str "  hello, world!
        ")
        (expected "hello, world!"))
    (setf str (concatenate 'string str (string #\Page) (string #\Zero_Width_Space)))
    (is-false (string= (sdm:trim-whitespace str) expected))
    (is-true (string= (sdm:trim-whitespace! str) expected)))

  (let ((str "hello, world!")
        (expected "hello, world!"))
    (is-true (string= (sdm:trim-whitespace str) expected))
    (is-true (string= (sdm:trim-whitespace! str) expected))))

(test strings+
  (let ((str "hello")
        (chars '(#\, #\Space #\w #\o #\r #\l #\d #\!))
        (nums '(69 42))
        (path #P"/path/to/file")
        (other '(foo bar baz)))

    (is-true (sdm:strings+ str #\, #\Space "world!") "hello, world!")
    (is-true `(sdm:strings+ str ,@chars) "hello, world!")
    (is-true (sdm:strings+ str #\- (elt nums 0) #\- (elt nums 1)) "hello-69-42")
    (is-true (sdm:strings+ path str) "/path/to/filehello")
    (is-true (sdm:strings+ path other) "/path/to/file(foo bar baz)")))

(test string/char-comparisons
  (is-true (sdm:caseless-string= "hello" "HELLO"))
  (is-true (sdm:caseless-string= "hello" "hello"))
  (is-true (sdm:caseless-string= "hello" "hElLo"))

  (is-false (sdm:caseless-string= "hello" "WORLD"))
  (is-false (sdm:caseless-string= "hello" "world"))
  (is-false (sdm:caseless-string= "hello" "WoRlD"))

  (is-true (sdm:caseless-string= "hello" "ell" :start1 1 :end1 4))
  (is-true (sdm:caseless-string= "hello" "ELL" :start1 1 :end1 4))

  (is-true (sdm:caseless-string= "hello" "why hello there" :start2 4 :end2 9))
  (is-true (sdm:caseless-string= "hello" "WHY HELLO THERE" :start2 4 :end2 9))

  (is-true (sdm:caseless-char= #\a #\A))
  (is-true (sdm:caseless-char= #\a #\a))
  (is-true (sdm:caseless-char= #\space #\space))
  (is-false (sdm:caseless-char= #\a #\Z))
  (is-false (sdm:caseless-char= #\a #\z)))

(test split-string
  (let ((str "hello there world how are you")
        (expected-list '("hello" "there" "world" "how" "are" "you"))
        (expected-space-list '("hello " "there " "world " "how " "are " "you")))

    (is-true (equalp (sdm:split-string str #\Space :as-list t) expected-list))
    (is-true (equalp (sdm:split-string str #\Space :as-list t :dont-drop t) expected-space-list))
    (is-true (equalp (sdm:split-string str #\Space :as-list t :limit 3) '("hello" "there" "world how are you")))
    (is-true (equalp (sdm:split-string "hello     world" #\Space :as-list t :remove-empty t) '("hello" "world")))
    (is-true (equalp (sdm:split-string "hello     world" #\Space :as-list t) '("hello" "" "" "" "" "world")))))

(test string-replace
  (let ((str "hello lol"))
    (is-true (string= (sdm:string-replace str "lol" "world") "hello world"))
    (is-false (string= (sdm:string-replace str "nah" "world") "hello world"))))

(test string-?-with
  (let ((str "hello world"))
    (is-true (sdm:string-starts-with str "hell"))
    (is-false (sdm:string-starts-with str "HELL"))
    (is-true (sdm:string-starts-with str "HELL" :test #'sdm:caseless-char=))

    (is-true (sdm:string-ends-with str "orld"))
    (is-false (sdm:string-ends-with str "ORLD"))
    (is-true (sdm:string-ends-with str "ORLD" :test #'sdm:caseless-char=))))

(test string-case
  (let ((str "hello"))
    (sdm:string-case str
      ("hello" (pass))
      ("world" (fail "STRING-CASE should not have matched ~s with \"world\"" str))
      (otherwise (fail "STRING-CASE should not have reached OTHERWISE clause.")))

    (sdm:string-case str
      ("nope" (fail "STRING-CASE should not have matched ~s with \"nope\"" str))
      ("world" (fail "STRING-CASE should not have matched ~s with \"world\"" str))
      (otherwise (pass)))

    (sdm:string-case str
      ("HELLO" (fail "STRING-CASE should not have matched ~s with \"HELLO\"" str))
      (otherwise (pass)))

    (sdm:string-case (string-upcase str)
      ("HELLO" (pass))
      (otherwise (fail "STRING-CASE should not have reached OTHERWISE clause in casless comparison"))))

  (let ((strs '("foo" "bar" "baz")))
    (loop for i from 0 below (length strs) do
      (sdm:string-case (nth i strs)
        ("foo" (is-true (= i 0)))
        ("bar" (is-true (= i 1)))
        ("baz" (is-true (= i 2)))))))

(test empty-strings
  (is-true (sdm:empty-string-p ""))
  (is-true (sdm:empty-string-p "   "))
  (is-false (sdm:empty-string-p "a"))
  (is-false (sdm:empty-string-p " a  "))

  (is-true (sdm:empty-string-p! ""))
  (is-true (sdm:empty-string-p! "   "))
  (is-false (sdm:empty-string-p! "a"))
  (is-false (sdm:empty-string-p! " a  ")))

(test char-in-range
  (is-true (sdm:char-in-range-p #\r
             ((#\a . #\z)
              (#\0 . #\9))
             #\Space))

  (is-true (sdm:char-in-range-p #\3
             ((#\a . #\z)
              (#\0 . #\9))))

  (is-true (sdm:char-in-range-p #\Space
             ((#\a . #\z)
              (#\0 . #\9))
             #\Space))

  (is-false (sdm:char-in-range-p #\R
              (("a" . "z")
               (#\0 . #\9))
              #\Space))

  (is-true (sdm:char-in-range-p #\R
              (("A" . "Z")
               (#\0 . #\9))
             #\Space)))

(test strings-join
  (is-true (string= (sdm:strings-join #\- "hello" "there" "world") "hello-there-world"))
  (is-false (string= (sdm:strings-join #\- "hello" "there" "world") "hello-there-world-"))
  (is-false (string= (sdm:strings-join #\- "hello" "there" "world") "-hello-there-world-"))
  (is-false (string= (sdm:strings-join #\- "hello" "there" "world") "-hello-there-world"))

  (is-true (string= (sdm:strings-join "--" "hello" "there" "world") "hello--there--world"))
  (is-false (string= (sdm:strings-join "--" "hello" "there" "world") "hello--there--world--"))
  (is-false (string= (sdm:strings-join "--" "hello" "there" "world") "--hello--there--world--"))
  (is-false (string= (sdm:strings-join "--" "hello" "there" "world") "--hello--there--world"))

  (is-true (string= (sdm:strings-join! #\- "hello" "there" "world") "hello-there-world-"))
  (is-false (string= (sdm:strings-join! #\- "hello" "there" "world") "hello-there-world"))
  (is-false (string= (sdm:strings-join! #\- "hello" "there" "world") "-hello-there-world-"))
  (is-false (string= (sdm:strings-join! #\- "hello" "there" "world") "-hello-there-world"))

  (is-true (string= (sdm:strings-join! "--" "hello" "there" "world") "hello--there--world--"))
  (is-false (string= (sdm:strings-join! "--" "hello" "there" "world") "hello--there--world"))
  (is-false (string= (sdm:strings-join! "--" "hello" "there" "world") "--hello--there--world--"))
  (is-false (string= (sdm:strings-join! "--" "hello" "there" "world") "--hello--there--world")))

(test strings-join-vector
  (is-true (string= (sdm:strings-join-vector #\- #("hello" "there" "world")) "hello-there-world"))
  (is-false (string= (sdm:strings-join-vector #\- #("hello" "there" "world")) "hello-there-world-"))
  (is-false (string= (sdm:strings-join-vector #\- #("hello" "there" "world")) "-hello-there-world-"))
  (is-false (string= (sdm:strings-join-vector #\- #("hello" "there" "world")) "-hello-there-world"))

  (is-true (string= (sdm:strings-join-vector "--" #("hello" "there" "world")) "hello--there--world"))
  (is-false (string= (sdm:strings-join-vector "--" #("hello" "there" "world")) "hello--there--world--"))
  (is-false (string= (sdm:strings-join-vector "--" #("hello" "there" "world")) "--hello--there--world--"))
  (is-false (string= (sdm:strings-join-vector "--" #("hello" "there" "world")) "--hello--there--world"))

  (is-true (string= (sdm:strings-join-vector! #\- #("hello" "there" "world")) "hello-there-world-"))
  (is-false (string= (sdm:strings-join-vector! #\- #("hello" "there" "world")) "hello-there-world"))
  (is-false (string= (sdm:strings-join-vector! #\- #("hello" "there" "world")) "-hello-there-world-"))
  (is-false (string= (sdm:strings-join-vector! #\- #("hello" "there" "world")) "-hello-there-world"))

  (is-true (string= (sdm:strings-join-vector! "--" #("hello" "there" "world")) "hello--there--world--"))
  (is-false (string= (sdm:strings-join-vector! "--" #("hello" "there" "world")) "hello--there--world"))
  (is-false (string= (sdm:strings-join-vector! "--" #("hello" "there" "world")) "--hello--there--world--"))
  (is-false (string= (sdm:strings-join-vector! "--" #("hello" "there" "world")) "--hello--there--world")))

(test ascii-char-p
  (is-true (sdm:ascii-char-p #\a))
  (is-true (sdm:ascii-char-p (code-char 127)))
  (is-false (sdm:ascii-char-p (code-char 128)))
  (is-false (sdm:ascii-char-p (elt "道" 0))))

(test ascii-control-char-p
  (is-false (sdm:ascii-control-char-p #\a))
  (is-false (sdm:ascii-control-char-p #\Latin_Small_Letter_U_With_Grave))
  (is-true (sdm:ascii-control-char-p #\DC2))
  (is-true (sdm:ascii-control-char-p #\Rubout))
  (is-false (sdm:ascii-control-char-p (code-char #x80)))
  (is-false (sdm:ascii-control-char-p (elt "道" 0))))

(test ascii-lower-case-char-p
  (is-true (sdm:ascii-lower-case-char-p #\a))
  (is-false (sdm:ascii-lower-case-char-p #\A))
  (is-false (sdm:ascii-lower-case-char-p #\1))
  (is-false (sdm:ascii-lower-case-char-p (elt "道" 0))))

(test ascii-upper-case-char-p
  (is-false (sdm:ascii-upper-case-char-p #\a))
  (is-true (sdm:ascii-upper-case-char-p #\A))
  (is-false (sdm:ascii-upper-case-char-p #\1))
  (is-false (sdm:ascii-upper-case-char-p (elt "道" 0))))
