(in-package :cl-sdm-tests)

(in-suite cl-sdm-test-suite)

(defmacro with-sized-stream ((stream wrapped limit init-string &optional sync-close) &body forms)
  `(let ((,wrapped (make-instance 'sdm:memory-stream)))
     (write-string ,init-string ,wrapped)
     (file-position ,wrapped 0)

     (let ((,stream (sdm:make-sized-stream ,wrapped ,limit ,sync-close)))
       ,@forms)))

(test sized-stream/doesnt-read-past-limit/char-by-char
  (with-sized-stream (sized stream 5 "abcdefg")
    (macrolet
        ((expect-char (ch)
           (sdm:with-gensyms (result)
             `(let ((,result (read-char sized nil nil)))
                (is-true (eql ,result ,ch) "Expected to read ~s, not ~s" ,ch ,result)))))
      (expect-char #\a)
      (expect-char #\b)
      (expect-char #\c)
      (is-true (= (sdm:sized-stream-read-remaining sized) 2))
      (expect-char #\d)
      (expect-char #\e)
      (is-true (= (sdm:sized-stream-read-remaining sized) 0))
      (is-true (null (read-char sized nil nil)))
      (is-true (= (sdm:sized-stream-read-remaining sized) 0))
      (is-true (null (read-char sized nil nil))))))

(test sized-stream/doesnt-read-past-limit/correct-size
  (with-sized-stream (sized stream 5 "1234567")
    (let ((buf (sdm:new-array 5 sdm:t/uint8)))
      (is-true (= (read-sequence buf sized) 5))
      (is-true (equalp buf (sdm:string->bytes "12345" :as-vector t)))
      (is-true (= (read-sequence buf sized) 0))
      (is-true (equalp buf (sdm:string->bytes "12345" :as-vector t))))))

(test sized-stream/reads-sequence-partially
  (with-sized-stream (sized stream 5 "1234567")
    (let ((buf (sdm:new-array 10 sdm:t/uint8)))
      (is-true (= (read-sequence buf sized) 5))
      (is-true (equalp buf #(49 50 51 52 53 0 0 0 0 0))))))

(test sized-stream/can-extend-size
  (with-sized-stream (sized stream 5 "1234567890")
    (let ((buf (sdm:new-array 10 sdm:t/uint8)))
      (is-true (= (read-sequence buf sized) 5))
      (is-true (equalp buf #(49 50 51 52 53 0 0 0 0 0)))
      (setf (sdm:sized-stream-read-remaining sized) 5)
      (is-true (= (read-sequence buf sized) 5))
      (is-true (equalp buf #(54 55 56 57 48 0 0 0 0 0))))))

(test sized-stream/cannot-write
  (with-sized-stream (sized stream 5 "a")
    (signals (error)
      (write-byte 69 sized))))

(test sized-stream/close
  (with-sized-stream (sized stream 5 "abcdefg")
    (is-true (eql (read-char sized) #\a))
    (is-true (eql (read-char sized) #\b))
    (close sized)
    (is-true (sdm:sized-stream-closed-p sized))
    (is-false (open-stream-p sized))
    (is-true (open-stream-p stream))
    (signals (sdm:sized-stream-closed-error)
      (read-char sized))))

(test sized-stream/sync-close
  (with-sized-stream (sized stream 5 "abcdefg" t)
    (close sized)
    (is-true (sdm:sized-stream-closed-p sized))
    (is-false (open-stream-p sized))
    (is-false (open-stream-p stream))))

(test sized-stream/read-byte
  (with-sized-stream (sized stream 3 "abcdefg")
    (is-true (eql (read-byte sized nil nil) (char-code #\a)))
    (is-true (eql (read-byte sized nil nil) (char-code #\b)))
    (is-true (eql (read-byte sized nil nil) (char-code #\c)))
    (is-true (null (read-byte sized nil nil)))))

(test sized-stream/read-line
  (with-sized-stream (sized stream 9 (format nil "foo~%bar~%baz"))
    (macrolet
        ((expect-str (str)
           (sdm:with-gensyms (result)
             `(let ((,result (read-line sized nil nil)))
                (is-true (equalp ,result ,str) "Expected to read ~s, not ~s" ,str ,result)))))
      (expect-str "foo")
      (expect-str "bar")
      (expect-str "b")
      (is-true (null (read-byte sized nil nil))))))
