(in-package :cl-sdm-tests)
(in-suite cl-sdm-test-suite)

(defmacro with-rsconf-data ((result-var parsed-var data &body failure-reason-args) &body forms)
  `(let ((,parsed-var ,data)
         (,result-var nil))
     ,(if failure-reason-args
          `(handler-case
               (setf ,result-var (sdm-rsconf:parse ,parsed-var))
             (error ()
               (fail ,@failure-reason-args)))
          `(handler-case
               (setf ,result-var (sdm-rsconf:parse ,parsed-var))
             (error (err)
               (fail "Failed with error: ~a~%~w" err ,(list 'quote forms)))))
     ,@forms))




(test rsconf/basic-parsing
  (with-rsconf-data (result data "key1: 69
key2:\"test\"")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 2))
    (is-true (eql (gethash "key1" result) 69))
    (is-true (equalp (gethash "key2" result) "test")))

  (with-rsconf-data (result data "key1: 42, key2: \"on-same-line\"")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 2))
    (is-true (eql (gethash "key1" result) 42))
    (is-true (equalp (gethash "key2" result) "on-same-line"))))




(test rsconf/parse-toplevel-braces
  (with-rsconf-data (result data "{
  key1: 69
  key2: \"test\"
}")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 2))
    (is-true (eql (gethash "key1" result) 69))
    (is-true (equalp (gethash "key2" result) "test")))

  (with-rsconf-data (result data  "{ key1: \"on same line\" }")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (equalp (gethash "key1" result) "on same line"))))




(test rsconf/parse-toplevel-array
  (with-rsconf-data (result data "[
  69
  \"test\"
]")
    (is-true (vectorp result))
    (is-true (= (length result) 2))
    (is-true (eql (elt result 0) 69))
    (is-true (equalp (elt result 1) "test")))

  (with-rsconf-data (result data  "[ 42, \"on same line\" ]")
    (is-true (vectorp result))
    (is-true (= (length result) 2))
    (is-true (eql (elt result 0) 42))
    (is-true (equalp (elt result 1) "on same line"))))




(test rsconf/parse-comments
  (with-rsconf-data (result data "; comment
key: 69")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (eql (gethash "key" result) 69)))

  (with-rsconf-data (result data "; comment
key: 42 ;; comment 2")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (eql (gethash "key" result) 42)))

  (with-rsconf-data (result data "
key: 36
;; comment 3")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (eql (gethash "key" result) 36))))




(test rsconf/parse-string-types
  (with-rsconf-data (result data "key: \"test\"")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (stringp (gethash "key" result)))
    (is-true (string= (gethash "key" result) "test")))

  (let ((str "multi-line

data"))
    (with-rsconf-data (result data (format nil "key: ~s" str))
      (is-true (hash-table-p result))
      (is-true (= (hash-table-count result) 1))
      (is-true (stringp (gethash "key" result)))
      (is-true (string= (gethash "key" result) str)))))




(test rsconf/parse-bool-types
  (with-rsconf-data (result data "key: true")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (eql (gethash "key" result) :true)))

  (with-rsconf-data (result data "key: false")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (eql (gethash "key" result) :false)))

  (signals (sdm-rsconf:rsconf-parse-error "'ture' did not signal an error")
    (sdm-rsconf:parse "key: ture"))

  (signals (sdm-rsconf:rsconf-parse-error "'flase' did not signal an error")
    (sdm-rsconf:parse "key: flase"))

  (signals (sdm-rsconf:rsconf-parse-error "'tlol' did not signal an error")
    (sdm-rsconf:parse "key: tlol")))




(test rsconf/parse-null-type
  (with-rsconf-data (result data "key: nil")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (eql (gethash "key" result) :null)))

  (signals (sdm-rsconf:rsconf-parse-error "'nli' did not signal an error")
    (sdm-rsconf:parse "key: nli"))

  (signals (sdm-rsconf:rsconf-parse-error "'null' did not signal an error")
    (sdm-rsconf:parse "key: null"))

  (signals (sdm-rsconf:rsconf-parse-error "'nlol' did not signal an error")
    (sdm-rsconf:parse "key: nlol")))




(test rsconf/parse-numeric-types
  (with-rsconf-data (result data "key: 69")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (integerp (gethash "key" result)))
    (is-true (= (gethash "key" result) 69)
             "Result of '69' was ~a, not 69" (gethash "key" result)))

  (with-rsconf-data (result data "key: #x45")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (integerp (gethash "key" result)))
    (is-true (= (gethash "key" result) 69)
             "Result of '#x45' was ~a, not 69" (gethash "key" result)))

  (with-rsconf-data (result data "key: #o105")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (integerp (gethash "key" result)))
    (is-true (= (gethash "key" result) 69)
             "Result of '#o105' was ~a, not 69" (gethash "key" result)))

  (with-rsconf-data (result data "key: #b1000101")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (integerp (gethash "key" result)))
    (is-true (= (gethash "key" result) 69)
             "Result of '#b1000101' was ~a, not 69" (gethash "key" result)))

  (with-rsconf-data (result data "key: +69")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (integerp (gethash "key" result)))
    (is-true (= (gethash "key" result) 69)
             "Result of '+69' was ~a, not 69" (gethash "key" result)))

  (with-rsconf-data (result data "key: -69")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (integerp (gethash "key" result)))
    (is-true (= (gethash "key" result) -69)
             "Result of '-69' was ~a, not -69" (gethash "key" result)))

  (with-rsconf-data (result data "key: 69.42")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.42d0)
             "Result of '69.42' was ~a, not 69.42d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: 69.")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.0d0)
             "Result of '69.' was ~a, not 69.0d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: +69.42")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.42d0)
             "Result of '+69.42' was ~a, not 69.42d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: -69.42")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) -69.42d0)
             "Result of '-69.42' was ~a, not -69.42d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: 6942e-2")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.42d0)
             "Result of '6942e-2' was ~a, not 69.42d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: 6942d-2")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.42d0)
             "Result of '6942d-2' was ~a, not 69.42d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: 6936.e-2")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.36d0)
             "Result of '6936.e-2' was ~a, not 69.36d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: 6936.d-2")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.36d0)
             "Result of '6936.d-2' was ~a, not 69.36d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: 6936.5e-2")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.365d0)
             "Result of '6936.5e-2' was ~a, not 69.365d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: 6936.5d-2")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.365d0)
             "Result of '6936.5d-2' was ~a, not 69.365d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: 6.942e+1")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.42d0)
             "Result of '6.942e+1' was ~a, not 69.42d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: 6.942d+1")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.42d0)
             "Result of '6.942d+1' was ~a, not 69.42d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: 6.942e1")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.42d0)
             "Result of '6.942e1' was ~a, not 69.42d0" (gethash "key" result)))

  (with-rsconf-data (result data "key: 6.942d1")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (typep (gethash "key" result) 'double-float))
    (is-true (= (gethash "key" result) 69.42d0)
             "Result of '6.942d1' was ~a, not 69.42d0" (gethash "key" result)))

  (signals (sdm-rsconf:rsconf-parse-error "'59.69.79' did not signal an error")
    (sdm-rsconf:parse "key: 59.69.79"))

  (signals (sdm-rsconf:rsconf-parse-error "'59e69e79' did not signal an error")
    (sdm-rsconf:parse "key: 59e69e79"))

  (signals (sdm-rsconf:rsconf-parse-error "'59d69e79' did not signal an error")
    (sdm-rsconf:parse "key: 59d69e79"))

  (signals (sdm-rsconf:rsconf-parse-error "'59e69d79' did not signal an error")
    (sdm-rsconf:parse "key: 59e69d79"))

  (signals (sdm-rsconf:rsconf-parse-error "'6+9' did not signal an error")
    (sdm-rsconf:parse "key: 6+9"))

  (signals (sdm-rsconf:rsconf-parse-error "'+6-9' did not signal an error")
    (sdm-rsconf:parse "key: +6-9"))

  (signals (sdm-rsconf:rsconf-parse-error "'0x45' did not signal an error")
    (sdm-rsconf:parse "key: 0x45"))

  (signals (sdm-rsconf:rsconf-parse-error "'0x105' did not signal an error")
    (sdm-rsconf:parse "key: 0x105"))

  (signals (sdm-rsconf:rsconf-parse-error "'0b1000101' did not signal an error")
    (sdm-rsconf:parse "key: 0b1000101")))




(test rsconf/parse-should-fail
  (signals (sdm-rsconf:rsconf-parse-error "A single comma did not signal an error")
    (sdm-rsconf:parse ","))

  (signals (sdm-rsconf:rsconf-parse-error "Multiple commas did not signal an error")
    (sdm-rsconf:parse ",,,,"))

  (signals (sdm-rsconf:rsconf-parse-error "Multiple commas after data did not signal an error")
    (sdm-rsconf:parse "key: 69,,"))

  (signals (sdm-rsconf:rsconf-parse-error "Unquoted string did not signal an error")
    (sdm-rsconf:parse "key: lol"))

  (signals (sdm-rsconf:rsconf-parse-error "Quote in unquoted key name did not signal an error")
    (sdm-rsconf:parse "k\"ey: lol"))

  (signals (sdm-rsconf:rsconf-parse-error "Colon in unquoted key name did not signal an error")
    (sdm-rsconf:parse "k:ey: lol"))

  (signals (sdm-rsconf:rsconf-parse-error "Open brace in unquoted key name did not signal an error")
    (sdm-rsconf:parse "k{ey: lol"))

  (signals (sdm-rsconf:rsconf-parse-error "Close brace in unquoted key name did not signal an error")
    (sdm-rsconf:parse "k}ey: lol"))

  (signals (sdm-rsconf:rsconf-parse-error "Open bracket in unquoted key name did not signal an error")
    (sdm-rsconf:parse "k[ey: lol"))

  (signals (sdm-rsconf:rsconf-parse-error "Close bracket in unquoted key name did not signal an error")
    (sdm-rsconf:parse "k]ey: lol")))




(test rsconf/whitespace-ignored
  (with-rsconf-data (result data "   key1: 69
key2: \"test\"     "
                            "Basic spaces did not get ignored.")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 2))
    (is-true (eql (gethash "key1" result) 69))
    (is-true (equalp (gethash "key2" result) "test")))

  (dolist (ws (list #\Newline #\Tab #\Zero_Width_Space #\No-Break_Space #\Figure_Space
                    #-ccl #\Ogham_Space_Mark
                    #-ccl #\En_Quad
                    #-ccl #\Em_Quad
                    #-ccl #\En_Space
                    #-ccl #\Em_Space
                    #-ccl #\Three-Per-Em_Space
                    #-ccl #\Four-Per-Em_Space
                    #-ccl #\Six-Per-Em_Space
                    #-ccl #\Punctuation_Space
                    #-ccl #\Thin_Space
                    #-ccl #\Hair_Space
                    #-ccl #\Narrow_No-Break_Space
                    #-ccl #\Medium_Mathematical_Space
                    #-ccl #\Ideographic_Space))
    (with-rsconf-data (result data (format nil "~akey: true" ws)
                        "The character ~s before the key name was not skipped correctly" ws)
      (is-true (hash-table-p result))
      (is-true (= (hash-table-count result) 1))
      (is-true (eql (gethash "key" result) :true))))

  (dolist (ws (list #\Tab #\Zero_Width_Space #\No-Break_Space #\Figure_Space
                    #-ccl #\Ogham_Space_Mark
                    #-ccl #\En_Quad
                    #-ccl #\Em_Quad
                    #-ccl #\En_Space
                    #-ccl #\Em_Space
                    #-ccl #\Three-Per-Em_Space
                    #-ccl #\Four-Per-Em_Space
                    #-ccl #\Six-Per-Em_Space
                    #-ccl #\Punctuation_Space
                    #-ccl #\Thin_Space
                    #-ccl #\Hair_Space
                    #-ccl #\Narrow_No-Break_Space
                    #-ccl #\Medium_Mathematical_Space
                    #-ccl #\Ideographic_Space))
    (with-rsconf-data (result data (format nil "key~a: true" ws)
                        "The character ~s after the key name was not skipped correctly" ws)
      (is-true (hash-table-p result))
      (is-true (= (hash-table-count result) 1))
      (is-true (eql (gethash "key" result) :true))))

  (dolist (ws (list #\Tab #\Zero_Width_Space #\No-Break_Space #\Figure_Space
                    #-ccl #\Ogham_Space_Mark
                    #-ccl #\En_Quad
                    #-ccl #\Em_Quad
                    #-ccl #\En_Space
                    #-ccl #\Em_Space
                    #-ccl #\Three-Per-Em_Space
                    #-ccl #\Four-Per-Em_Space
                    #-ccl #\Six-Per-Em_Space
                    #-ccl #\Punctuation_Space
                    #-ccl #\Thin_Space
                    #-ccl #\Hair_Space
                    #-ccl #\Narrow_No-Break_Space
                    #-ccl #\Medium_Mathematical_Space
                    #-ccl #\Ideographic_Space))
    (with-rsconf-data (result data (format nil "key: true~a," ws)
                        "The character ~s after the value was not skipped correctly" ws)
      (is-true (hash-table-p result))
      (is-true (= (hash-table-count result) 1))
      (is-true (eql (gethash "key" result) :true))))

  (dolist (ws (list #\Tab #\Zero_Width_Space #\No-Break_Space #\Figure_Space
                    #-ccl #\Ogham_Space_Mark
                    #-ccl #\En_Quad
                    #-ccl #\Em_Quad
                    #-ccl #\En_Space
                    #-ccl #\Em_Space
                    #-ccl #\Three-Per-Em_Space
                    #-ccl #\Four-Per-Em_Space
                    #-ccl #\Six-Per-Em_Space
                    #-ccl #\Punctuation_Space
                    #-ccl #\Thin_Space
                    #-ccl #\Hair_Space
                    #-ccl #\Narrow_No-Break_Space
                    #-ccl #\Medium_Mathematical_Space
                    #-ccl #\Ideographic_Space))
    (with-rsconf-data (result data (format nil "key: true~a,~:*~a key2: false" ws)
                        "The character ~s when a comma is present was not skipped correctly" ws)
      (is-true (hash-table-p result))
      (is-true (= (hash-table-count result) 2))
      (is-true (eql (gethash "key" result) :true))
      (is-true (eql (gethash "key2" result) :false))))

  (signals (sdm-rsconf:rsconf-parse-error "A newline between the key name and colon did not signal an error")
    (sdm-rsconf:parse (format nil "key~a: true" #\Newline))))




(test rsconf/return-and-page-handled
  (dolist (ws '(#\Return #\Page))
    (signals (sdm-rsconf:rsconf-parse-error "A ~s character before a key name did not signal an error" ws)
      (sdm-rsconf:parse (format nil "~akey: true" ws)))

    (signals (sdm-rsconf:rsconf-parse-error "A ~s character after a key name did not signal an error" ws)
      (sdm-rsconf:parse (format nil "key~a: true" ws)))

    (signals (sdm-rsconf:rsconf-parse-error "A ~s character before a value did not signal an error" ws)
      (sdm-rsconf:parse (format nil "key:~atrue" ws)))

    (signals (sdm-rsconf:rsconf-parse-error "A ~s character after a value did not signal an error" ws)
      (sdm-rsconf:parse (format nil "key: true~a" ws)))

    (finishes
      (sdm-rsconf:parse (format nil "key: \"test~a\"" ws)))

    (finishes
      (sdm-rsconf:parse (format nil "\"key~a\": true" ws)))))



(test rsconf/nested-objects-and-arrays
  (with-rsconf-data (result data "obj: { key: true }")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (hash-table-p (gethash "obj" result)))
    (let ((obj (gethash "obj" result)))
      (is-true (= (hash-table-count obj) 1))
      (is-true (eql (gethash "key" obj) :true))))

  (with-rsconf-data (result data "obj: { key: [1, 2, 3] }")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (hash-table-p (gethash "obj" result)))
    (let ((obj (gethash "obj" result)))
      (is-true (vectorp (gethash "key" obj)))
      (let ((val (gethash "key" obj)))
        (is-true (= (length val) 3))
        (is-true (= (elt val 0) 1))
        (is-true (= (elt val 1) 2))
        (is-true (= (elt val 2) 3)))))

  (with-rsconf-data (result data "obj: [ 1, 2, 3]")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (vectorp (gethash "obj" result)))
    (let ((obj (gethash "obj" result)))
      (is-true (= (length obj) 3))
      (is-true (= (elt obj 0) 1))
      (is-true (= (elt obj 1) 2))
      (is-true (= (elt obj 2) 3))))


  (with-rsconf-data (result data "obj: [ true, [1, 2, 3] ]")
    (is-true (hash-table-p result))
    (is-true (= (hash-table-count result) 1))
    (is-true (vectorp (gethash "obj" result)))
    (let ((obj (gethash "obj" result)))
      (is-true (= (length obj) 2))
      (is-true (eql (elt obj 0) :true))
      (is-true (vectorp (elt obj 1)))
      (let ((val (elt obj 1)))
        (is-true (= (length val) 3))
        (is-true (= (elt val 0) 1))
        (is-true (= (elt val 1) 2))
        (is-true (= (elt val 2) 3))))))



(test rsconf/handles-bom
  ;; These values were taken from https://en.wikipedia.org/wiki/Byte_order_mark
  (dolist (enc-bytes '((#xEF #xBB #xBF) ;; UTF-8
                       (#xFE #xFF) ;; UTF-16 BE
                       (#xFF #xFE) ;; UTF-16 LE
                       (#x00 #x00 #xFE #xFF) ;; UTF-32 BE
                       (#xFF #xFE #x00 #x00) ;; UTF-32 LE
                       (#x2B #x2F #x76) ;; UTF-7
                       (#xF7 #x64 #x4C) ;; UTF-1
                       (#xDD #x73 #x66 #x73) ;; UTF-EBCDIC
                       (#x0E #xFE #xFF) ;; SCSU
                       (#xFB #xEE #x28) ;; BOCU-1
                       (#x84 #x31 #x95 #x33))) ;; GB18030
    (with-input-from-string (in (sdm:with-memory-stream (out :return :string)
                                  (dolist (byte enc-bytes)
                                    (write-byte byte out))
                                  (write-string "key: false" out)))
      (signals (sdm-rsconf:rsconf-bom-error)
        (sdm-rsconf:parse in)))))

(test rsconf/handles-escapes
  (macrolet
      ((test-str (str1 str2)
         (sdm:with-gensyms (raw-rsconf table)
           `(let* ((,raw-rsconf (format nil "foo: \"~a\"" ,str1))
                   (,table (sdm-rsconf:parse ,raw-rsconf)))
              (is-true (string= (gethash "foo" ,table) ,str2)
                       "The string ~s, stored as '~a', did not equal ~s after accounting for escaped, got ~s instead"
                       ,str1 ,raw-rsconf ,str2 (gethash "foo" ,table))))))
    (test-str "foo\\abar" "fooabar")
    (test-str "foo\\nbar" "foonbar")
    (test-str "foo\\\\bar" "foo\\bar")
    (test-str "foo\\\"bar" "foo\"bar")
    (test-str "foo\\u{9053}bar" "foo道bar")
    (test-str "foo\\\"" "foo\"")
    (test-str "foo\\\\" (sdm:strings+ "foo" #\\))
    (test-str "foo\\\\\\\\" (sdm:strings+ "foo" #\\ #\\))))
