(in-package :cl-sdm-tests)

(in-suite cl-sdm-test-suite)

(test num-8bit-bytes-needed
  (is-true (= (sdm:num-8bit-bytes-needed 69) 1))
  (is-true (= (sdm:num-8bit-bytes-needed 255) 1))
  (is-true (= (sdm:num-8bit-bytes-needed 256) 2))
  (is-true (= (sdm:num-8bit-bytes-needed 2569) 2))
  (is-true (= (sdm:num-8bit-bytes-needed 32768) 2))
  (is-true (= (sdm:num-8bit-bytes-needed 65535) 2))
  (is-true (= (sdm:num-8bit-bytes-needed 65536) 3))
  (is-true (= (sdm:num-8bit-bytes-needed 123456789123456789) 8))
  (is-true (= (sdm:num-8bit-bytes-needed 123456789123456789123456789123456789) 15)))

(test twos-complement-conversions
  (is-true (= (sdm:int->uint/2c -1 8) 255))
  (is-true (= (sdm:int->uint/2c 9 8) 9))

  (is-true (= (sdm:uint->int/2c 255 8)   -1) "(sdm:uint->int/2c 255 8) didn't return -1")
  (is-true (= (sdm:uint->int/2c   9 8)    9) "(sdm:uint->int/2c   9 8) didn't return 9")
  (is-true (= (sdm:uint->int/2c 127 8)  127) "(sdm:uint->int/2c 127 8) didn't return 127")
  (is-true (= (sdm:uint->int/2c 128 8) -128) "(sdm:uint->int/2c 128 8) didn't return -128")

  ;; Check some unusual bit widths
  (is-true (= (sdm:uint->int/2c  31 6)  31) "(sdm:uint->int/2c 31 6) didn't return 31")
  (is-true (= (sdm:uint->int/2c  32 6) -32) "(sdm:uint->int/2c 32 6) didn't return -32")
  (is-true (= (sdm:uint->int/2c  63 6)  -1) "(sdm:uint->int/2c 63 6) didn't return -1")
  (is-true (= (sdm:int->uint/2c  31 6)  31) "(sdm:int->uint/2c 31 6) didn't return 31")
  (is-true (= (sdm:int->uint/2c -32 6)  32) "(sdm:int->uint/2c -32 6) didn't return 32")
  (is-true (= (sdm:int->uint/2c  -1 6)  63) "(sdm:int->uint/2c -1 6) didn't return 63")

  ;; Check some unusual bit widths. but without the compiler macro
  (let ((bit-width 6))
    (is-true (= (sdm:uint->int/2c  31 bit-width)  31) "(sdm:uint->int/2c 31 bit-width) didn't return 31")
    (is-true (= (sdm:uint->int/2c  32 bit-width) -32) "(sdm:uint->int/2c 32 bit-width) didn't return -32")
    (is-true (= (sdm:uint->int/2c  63 bit-width)  -1) "(sdm:uint->int/2c 63 bit-width) didn't return -1")
    (is-true (= (sdm:int->uint/2c  31 bit-width)  31) "(sdm:int->uint/2c 31 bit-width) didn't return 31")
    (is-true (= (sdm:int->uint/2c -32 bit-width)  32) "(sdm:int->uint/2c -32 bit-width) didn't return 32")
    (is-true (= (sdm:int->uint/2c  -1 bit-width)  63) "(sdm:int->uint/2c -1 bit-width) didn't return 63")))

(test signed->unsigned
  (is-true (= (sdm:signed->unsigned -1 8) 255))
  (is-true (= (sdm:signed->unsigned 9 8) 9))
  (is-true (= (sdm:signed->unsigned 255 8) 255)))

(test coersions
  (let ((num #b1010101011001100010101011111111111111111101010100011001101010101))
    (is-true (= (sdm:coerce-to-uint8 num) #b01010101))
    (is-true (= (sdm:coerce-to-uint16 num) #b0011001101010101))
    (is-true (= (sdm:coerce-to-uint32 num) #b11111111101010100011001101010101))
    (is-true (= (sdm:coerce-to-uint64 num) #b1010101011001100010101011111111111111111101010100011001101010101))

    (is-true (= (sdm:coerce-to-int8 #b10000001) -127))
    (is-true (= (sdm:coerce-to-int16 #b1000000000000001) -32767))
    (is-true (= (sdm:coerce-to-int32 #b10000000000000000000000000000001) -2147483647))
    (is-true (= (sdm:coerce-to-int64 #b1000000000000000000000000000000000000000000000000000000000000001)
                -9223372036854775807))

    (is-true (= (sdm:coerce-to-int8 129) -127))
    (is-true (= (sdm:coerce-to-int8 653789) -35))

    (is-true (= (sdm:coerce-to-bit-size 129 8) -127))
    (is-true (= (sdm:coerce-to-bit-size 129 8 t) 129))

    ;; Same thing, but without the compiler macro
    (let ((bit-size 8))
      (is-true (= (sdm:coerce-to-bit-size 129 bit-size) -127))
      (is-true (= (sdm:coerce-to-bit-size 129 bit-size t) 129)))

    ;; Additional tests
    (dotimes (i (1- (expt 2 5)))
      (is-true (= (sdm:coerce-to-bit-size i 5)
                  (sdm:coerce-to-bit-size i (1+ 4)))
               "Coercing ~a to a bit size of 5 didn't work" i))

    ;; Remi: Encountered this in CL-RemiAudio's FLAC decoder, 19 April 2025
    (sdm:with-memory-stream (in :buffer (sdm:new-array-with sdm:t/uint8 '(#b10011001 0)))
      (let ((reader (sdm:make-bit-reader in)))
        (sdm:bit-reader-read reader 2)
        (is-true (sdm:coerce-to-bit-size (print (sdm:bit-reader-read reader 5)) 5) 12)))))

(test ieee-float-conversions
  (dolist (value '(1.0d0 69.0d0 69.42d0 36.69d0))
    (is-true (= (sdm:uint64->dfloat (sdm:dfloat->uint64 value)) value)))

  (dolist (value '(1.0 69.0 69.42 36.69))
    (is-true (= (sdm:uint32->sfloat (sdm:sfloat->uint32 value)) value))))

(test ieee-float-conversions/special-values
  (is-true (eql (sdm:uint64->dfloat 9218868437227405312) :positive-infinity))
  (is-true (eql (sdm:uint64->dfloat 18442240474082181120) :negative-infinity))
  (is-true (eql (sdm:uint64->dfloat 18444492273895866368) :not-a-number))
  (is-true (eql (sdm:uint64->dfloat 0) 0.0d0))
  (is-true (eql (sdm:uint64->dfloat 9223372036854775808) -0.0d0))

  (is-true (eql (sdm:uint32->sfloat 2139095040) :positive-infinity))
  (is-true (eql (sdm:uint32->sfloat 4286578688) :negative-infinity))
  (is-true (eql (sdm:uint32->sfloat 4290772992) :not-a-number))
  (is-true (eql (sdm:uint32->sfloat 0) 0.0))
  (is-true (eql (sdm:uint32->sfloat 2147483648) -0.0)))

(test clamp
  (for-all ((num (gen-integer)))
    (cond
      ((< num -69)
       (is-true (= (sdm:clamp num -69 42) -69)))
      ((> num 42)
       (is-true (= (sdm:clamp num -69 42) 42)))

      (t (is-true (= (sdm:clamp num -69 42) num))))))

(test fast-sin
  (let ((expected #(0.00000000000000000000d0 0.00999983333416693161d0
                    0.01999866669333520697d0 0.02999550020250281446d0 0.03998933418665104322d0
                    0.04997916927071109627d0 0.05996400647950079349d0 0.06994284733762118345d0
                    0.07991469396930328539d0 0.08987854919819479593d0 0.09983341664707684471d0
                    0.10977830083750091117d0 0.11971220728933583410d0 0.12963414262021491496d0
                    0.13954311464487342831d0 0.14943813247436621428d0 0.15931820661515572191d0
                    0.16918234906806065010d0 0.17902957342705502763d0 0.18885889497790778191d0
                    0.19866933079665366479d0 0.20845989984788462768d0 0.21822962308285254185d0
                    0.22797752353737338327d0 0.23770262642952277865d0 0.24740395925711333724d0
                    0.25708055189494427584d0 0.26673143669181298421d0 0.27635564856727928840d0
                    0.28595222510817291939d0 0.29552020666483386213d0 0.30505863644707681415d0
                    0.31456656061986904005d0 0.32404302839871340636d0 0.33348709214472610496d0
                    0.34289780745939940632d0 0.35227423327904128225d0 0.36161543196888057405d0
                    0.37092046941682998984d0 0.38018841512689632856d0 0.38941834231222971496d0
                    0.39860932798780113195d0 0.40776045306270036761d0 0.41687080243204449603d0
                    0.42593946506848734401d0 0.43496553411332183892d0 0.44394810696716469067d0
                    0.45288628538021608039d0 0.46177917554208375295d0 0.47062588817116302042d0
                    0.47942553860356468265d0 0.48817724688158031787d0 0.49688013784167761600d0
                    0.50553334120201576241d0 0.51413599164947287878d0 0.52268722892617702769d0
                    0.53118619791553056686d0 0.53963204872772196907d0 0.54802393678471306160d0
                    0.55636102290469702236d0 0.56464247338601536441d0 0.57286746009052724737d0
                    0.58103516052642301126d0 0.58914475793047227370d0 0.59719544134969959615d0
                    0.60518640572247905940d0 0.61311685195903975476d0 0.62098598702137397520d0
                    0.62879302400254044603d0 0.63653718220535382422d0 0.64421768722045413824d0
                    0.65183377100374695345d0 0.65938467195320671355d0 0.66686963498503681880d0
                    0.67428791160917644909d0 0.68163876000414946876d0 0.68892144509124586538d0
                    0.69613523860802861698d0 0.70327941918115932651d0 0.71035327239853462977d0
                    0.71735609088072727069d0 0.72428717435172285111d0 0.73114582970894681502d0
                    0.73793137109257433970d0 0.74464311995411514022d0 0.75128040512426796926d0
                    0.75784256288003715163d0 0.76432893701110349305d0 0.77073887888544545532d0
                    0.77707174751420149406d0 0.78332690961576800781d0 0.78950373967912745865d0
                    0.79560162002639822632d0 0.80161994087460275438d0 0.80755810039664466249d0
                    0.81341550478149104997d0 0.81919156829355299632d0 0.82488571333125915164d0
                    0.83049737048481497848d0 0.83602597859314331519d0 0.84147098480000037668d0
                    0.84683184460926086423d0 0.85210802193936685534d0 0.85729898917693625471d0
                    0.86240422722952370105d0 0.86742322557752860046d0 0.87235548232524851020d0
                    0.87720050425106677050d0 0.88195780685677571675d0 0.88662691441602525799d0
                    0.89120736002189648861d0 0.89569868563359089642d0 0.90010044212223561111d0
                    0.90441218931579514528d0 0.90863349604308885077d0 0.91276394017690842819d0
                    0.91680310867622905047d0 0.92075059762751521042d0 0.92460601228511096750d0
                    0.92836896711071370536d0 0.93203908581192951299d0 0.93561600137990064141d0
                    0.93909935612600736743d0 0.94248880171763549374d0 0.94578399921301048447d0
                    0.94898461909509035372d0 0.95209034130451719591d0 0.95510085527162291719d0
                    0.95801585994748517106d0 0.96083506383403172180d0 0.96355818501318990510d0
                    0.96618495117507607883d0 0.96871509964522717340d0 0.97114837741086545986d0
                    0.97348454114619908939d0 0.97572335723675240882d0 0.97786460180272649634d0
                    0.97990806072138281202d0 0.98185352964845529211d0 0.98370081403858034008d0
                    0.98544972916474860014d0 0.98710010013677451646d0 0.98865176191878023637d0
                    0.99010455934569552294d0 0.99145834713876912492d0 0.99271298992009049389d0
                    0.99386836222612273684d0 0.99492434852024080882d0 0.99588084320427816554d0
                    0.99673775062907898903d0 0.99749498510405276708d0 0.99815247090573488986d0
                    0.99871014228534737978d0 0.99916794347536208587d0 0.99952582869506456653d0
                    0.99978376215511832736d0 0.99994171806112897016d0 0.99999968061620558846d0)))
    (loop for i double-float from 0.0d0 to (/ pi 2.0d0) by 0.01d0
          for idx from 0
          for expect = (elt expected idx)
          for actual = (sdm:fast-sin i)
          do (is-true (close-to expect actual 1.0e-9)
                      "Expected (close-to ~f ~f 1.0e-9) for fast-sin of ~f" expect actual i))))

(test fast-cos
  (let ((expected #(1.00000000000000000000d0 0.99995000041703507865d0
                    0.99980000666805379783d0 0.99955003375229745100d0 0.99920010666683445333d0
                    0.99875026040406156280d0 0.99820053994820423515d0 0.99755100027081833325d0
                    0.99680170632529363495d0 0.99595273304035869533d0 0.99500416531258784048d0
                    0.99395609799791240224d0 0.99280863590213508374d0 0.99156189377045000910d0
                    0.99021599627596879056d0 0.98877107800725350106d0 0.98722728345485821766d0
                    0.98558476699688102318d0 0.98384369288352613303d0 0.98200423522067981086d0
                    0.98006657795250040621d0 0.97803091484302462355d0 0.97589744945679157695d0
                    0.97366639513848696197d0 0.97133797499160967615d0 0.96891242185616177629d0
                    0.96638997828536510326d0 0.96377089652140723874d0 0.96105543847021712711d0
                    0.95824387567527591347d0 0.95533648929046310894d0 0.95233357005194152478d0
                    0.94923541824908475029d0 0.94604234369444883956d0 0.94275466569279142703d0
                    0.93937271300914215733d0 0.93589682383592665005d0 0.93232734575914799624d0
                    0.92866463572362834000d0 0.92490905999731576337d0 0.92106099413465680747d0
                    0.91712082293904240160d0 0.91308894042432808824d0 0.90896574977543209695d0
                    0.90475166330801803927d0 0.90044710242726355709d0 0.89605249758571980934d0
                    0.89156828824026679303d0 0.88699492280816760736d0 0.88233285862222698981d0
                    0.87758256188505845419d0 0.87274450762246358337d0 0.86781917963593002696d0
                    0.86280707045425075741d0 0.85770868128427213417d0 0.85252452196077199709d0
                    0.84725511089547700383d0 0.84190097502522065476d0 0.83646264975924999874d0
                    0.83094067892568401756d0 0.82533561471713090540d0 0.81964801763546824009d0
                    0.81387845643579304067d0 0.80802750806954515284d0 0.80209575762681240008d0
                    0.79608379827782116411d0 0.78999223121361861200d0 0.78382166558595356420d0
                    0.77757271844636077773d0 0.77124601468445574959d0 0.76484218696544503668d0
                    0.75836187566685930861d0 0.75180572881451590561d0 0.74517440201771512065d0
                    0.73846855840367964241d0 0.73168886855124126711d0 0.72483601042378298374d0
                    0.71791066930144220581d0 0.71091353771258325445d0 0.70384531536454386647d0
                    0.69670670907366494262d0 0.68949843269460853179d0 0.68222120704897326604d0
                    0.67487575985321190952d0 0.66746282564586034702d0 0.65998314571408411844d0
                    0.65243746801955060377d0 0.64482654712363274285d0 0.63715114411195483690d0
                    0.62941202651828387360d0 0.62160996824777825509d0 0.61374574949959770365d0
                    0.60582015668888478110d0 0.59783398236812512749d0 0.58978802514789419043d0
                    0.58168308961699710657d0 0.57351998626201217135d0 0.56529953138624455811d0
                    0.55702254702809772535d0 0.54868986087887239300d0 0.54030230619999941588d0
                    0.53186072173971710164d0 0.52336595164919841316d0 0.51481884539813937973d0
                    0.50622025768981493421d0 0.49757104837561150212d0 0.48887208236904611258d0
                    0.48012422955927736012d0 0.47132836472412054096d0 0.46248536744257362496d0
                    0.45359612200686227901d0 0.44466151733401482193d0 0.43568244687697399442d0
                    0.42665980853525531380d0 0.41759450456516022943d0 0.40848744148955495881d0
                    0.39933953000722044457d0 0.39015168490178631089d0 0.38092482495025559164d0
                    0.37165987283112922324d0 0.36235775503214129323d0 0.35301940175761181706d0
                    0.34364574683542892242d0 0.33423772762366632527d0 0.32479628491684919833d0
                    0.31532236285187398295d0 0.30581690881359502310d0 0.29628087334008479381d0
                    0.28671521002757738295d0 0.27712087543510643961d0 0.26749882898884513871d0
                    0.25785003288615870876d0 0.24817545199937773859d0 0.23847605377930392034d0
                    0.22875280815845544513d0 0.21900668745406304261d0 0.20923866627082587932d0
                    0.19944972140343675271d0 0.18964083173888601763d0 0.17981297815855468158d0
                    0.16996714344010477316d0 0.16010431215917853010d0 0.15022547059091451160d0
                    0.14033160661129095992d0 0.13042370959830518196d0 0.12050277033300171858d0
                    0.11056978090035296525d0 0.10062573459000945242d0 0.09067162579692200630d0
                    0.08070844992185266520d0 0.07073720327177612788d0 0.06075888296019016366d0
                    0.05077448680733676056d0 0.04078501324034855546d0 0.03079146119332820675d0
                    0.02079483000737192278d0 0.01079611933054358541d0 0.00079632901781367949d0)))
    (loop for i double-float from 0.0d0 to (/ pi 2.0d0) by 0.01d0
          for idx from 0
          for expect = (elt expected idx)
          for actual = (sdm:fast-cos i)
          do (is-true (close-to expect actual 1.0e-9)
                      "Expected (close-to ~f ~f 1.0e-9) for fast-cos of ~f" expect actual i))))

(test fast-tan
  (let ((expected #(0.00000000000000000000d0 0.01000033334474337288d0
                    0.02000266707815280889d0 0.03000900319050028159d0 0.04002134687796335411d0
                    0.05004170815232131947d0 0.06007210345876833751d0 0.07011455730459084656d0
                    0.08017110390149266574d0 0.09024378882439533511d0 0.10033467068959606161d0
                    0.11044582285522841492d0 0.12057933514704298472d0 0.13073731561260834577d0
                    0.14092189230712301451d0 0.15113521511413149923d0 0.16137945760454891486d0
                    0.17165681893752027243d0 0.18196952580677469569d0 0.19231983443627687080d0
                    0.20271003262913653287d0 0.21314244187390257723d0 0.22361941951255173699d0
                    0.23414336097467497444d0 0.24471670208257689794d0 0.25534192143222644855d0
                    0.26602154285524226562d0 0.27675813796735482253d0 0.28755432880906911519d0
                    0.29841279058455372208d0 0.30933625450510515709d0 0.32032751074389015367d0
                    0.33138941150904216437d0 0.34252487424259997484d0 0.35373688495321431446d0
                    0.36502850169102518674d0 0.37640285817362945098d0 0.38786316757261413146d0
                    0.39941272647073700064d0 0.41105491900049195841d0 0.42279322117550610649d0
                    0.43463120542698963078d0 0.44657254535829610242d0 0.45862102073155985860d0
                    0.47078052270136777624d0 0.48305505931149261833d0 0.49544876127188275383d0
                    0.50796588803436804049d0 0.52061083418691533886d0 0.53338813618775737790d0
                    0.54630247946234156142d0 0.55935870588779379453d0 0.57256182169150260375d0
                    0.58591700579248739977d0 0.59942961861645205346d0 0.61310521141784002364d0
                    0.62694953614483028836d0 0.64096855588603773857d0 0.65516845594074435422d0
                    0.66955565555778595588d0 0.68413682039178769223d0 0.69891887572927957617d0
                    0.71390902054136962107d0 0.72911474242410845292d0 0.74454383349248054458d0
                    0.76020440729911731292d0 0.77610491685436988618d0 0.79225417383033902574d0
                    0.80866136903783825129d0 0.82533609427212573095d0 0.84228836563056763698d0
                    0.85952864841325893330d0 0.87706788372602673132d0 0.89491751691422694037d0
                    0.91308952796534104124d0 0.93159646402862950065d0 0.95045147421102116336d0
                    0.96966834682007907453d0 0.98926154923728926160d0)))
    (loop for i double-float from 0.0d0 to (/ pi 4.0d0) by 0.01d0
          for idx from 0
          for expect = (elt expected idx)
          for actual = (sdm:fast-tan i)
          do (is-true (close-to expect actual 1.0e-9)
                      "Expected (close-to ~f ~f 1.0e-9) for fast-tan of ~f" expect actual i))))

(test fast-inv-sin
  (let ((expected #(0.00006752679489663294d0 0.01004404508484557645d0
                    0.02002453988619867786d0 0.03000979415159710229d0 0.04000060329084864641d0
                    0.04999777566913410354d0 0.06000213312922109132d0 0.07001451153910243086d0
                    0.08003576136657475359d0 0.09006674828238803165d0 0.10010835379370019993d0
                    0.11016147590971336889d0 0.12022702984149225003d0 0.13030594873812395562d0
                    0.14039918446153332177d0 0.15050770840245042415d0 0.16063251234021502789d0
                    0.17077460934931143477d0 0.18093503475575856143d0 0.19111484714672655194d0
                    0.20131512943702789542d0 0.21153698999641790124d0 0.22178156384198288720d0
                    0.23205001390023927144d0 0.24234353234396088794d0 0.25266334200919460251d0
                    0.26301069789838682489d0 0.27338688877607442151d0 0.28379323886417462397d0
                    0.29423110964453735861d0 0.30470190177714151503d0 0.31520705714309538159d0
                    0.32574806102246167683d0 0.33632644441790948697d0 0.34694378653625523867d0
                    0.35760171744115343273d0 0.36830192089153679369d0 0.37904613738190029437d0
                    0.38983616740218196561d0 0.40067387493688588762d0 0.41156119122518264319d0
                    0.42250011880610816029d0 0.43349273587564574051d0 0.44454120098550586881d0
                    0.45564775811683988493d0 0.46681474216600649108d0 0.47804458488393164473d0
                    0.48933982131563436013d0 0.50070309679224545008d0 0.51213717453443585725d0
                    0.52364494393373250958d0 0.53522942958690689075d0 0.54689380116866970205d0
                    0.55864138423953391133d0 0.57047567209920635456d0 0.58240033881158537188d0
                    0.59441925354578950547d0 0.60653649639915829805d0 0.61875637589344212319d0
                    0.63108344836523910804d0 0.64352253950707660746d0 0.65607876835754541656d0
                    0.66875757408906821855d0 0.68156474600202598957d0 0.69450645720641701608d0
                    0.70758930255989949121d0 0.72082034153769558582d0 0.73420714684021815000d0
                    0.74775785970455788121d0 0.76148125308420744339d0 0.77538680410805616017d0
                    0.78948477753867751794d0 0.80378632233968327903d0 0.81830358395722480047d0
                    0.83304983555522837957d0 0.84803963226364731387d0 0.86328899356772437201d0
                    0.87881562037325222647d0 0.89463915515485714636d0 0.91078149611363867777d0
                    0.92726717970297489746d0 0.94412385062120407486d0 0.96138284501187465381d0
                    0.97907992207047334077d0 0.99725619296577761030d0 1.01595931624835977836d0
                    1.03524505953400081104d0 1.05517937464893529764d0 1.07584120886418510565d0
                    1.09732639875372828264d0 1.11975320421721336039d0 1.14327041496398784659d0
                    1.16806966114128107925d0 1.19440494481832559792d0 1.22262535832035457162d0
                    1.25323383468755977788d0 1.28700284126785891026d0 1.32523373574058545010d0
                    1.37046603400695832242d0 1.42926175389097132040d0)))
    (loop for i double-float from 0.0d0 to 1.0d0 by 0.01d0
          for idx from 0
          for expect = (elt expected idx)
          for actual = (sdm:fast-inv-sin i)
          do (is-true (close-to expect actual 1.0e-9)
                      "Expected (close-to ~f ~f 1.0e-9) for fast-inv-sin of ~f" expect actual i))))

(test fast-inv-cos
  (let ((expected #(1.57072879999999992506d0 1.56075228171005098154d0
                    1.55077178690869788014d0 1.54078653264329945571d0 1.53079572350404791159d0
                    1.52079855112576245446d0 1.51079419366567546668d0 1.50078181525579412714d0
                    1.49076056542832180440d0 1.48072957851250852634d0 1.47068797300119635807d0
                    1.46063485088518318911d0 1.45056929695340430797d0 1.44049037805677260238d0
                    1.43039714233336323623d0 1.42028861839244613385d0 1.41016381445468153011d0
                    1.40002171744558512323d0 1.38986129203913799657d0 1.37968147964817000606d0
                    1.36948119735786866258d0 1.35925933679847865676d0 1.34901476295291367080d0
                    1.33874631289465728656d0 1.32845279445093567006d0 1.31813298478570195549d0
                    1.30778562889650973311d0 1.29740943801882213648d0 1.28700308793072193403d0
                    1.27656521715035919939d0 1.26609442501775504297d0 1.25558926965180117641d0
                    1.24504826577243488117d0 1.23446988237698707103d0 1.22385254025864131933d0
                    1.21319460935374312527d0 1.20249440590335976431d0 1.19175018941299626363d0
                    1.18096015939271459239d0 1.17012245185801067038d0 1.15923513556971391480d0
                    1.14829620798878839771d0 1.13730359091925081749d0 1.12625512580939068918d0
                    1.11514856867805667306d0 1.10398158462889006692d0 1.09275174191096491327d0
                    1.08145650547926219787d0 1.07009323000265110792d0 1.05865915226046070075d0
                    1.04715138286116404842d0 1.03556689720798966725d0 1.02390252562622685595d0
                    1.01215494255536264667d0 1.00032065469569020344d0 0.98839598798331118612d0
                    0.97637707324910705253d0 0.96425983039573825994d0 0.95203995090145443481d0
                    0.93971287842965744996d0 0.92727378728781995054d0 0.91471755843735114144d0
                    0.90203875270582833945d0 0.88923158079287056843d0 0.87628986958847954192d0
                    0.86320702423499706679d0 0.84997598525720097218d0 0.83658917995467840800d0
                    0.82303846709033867679d0 0.80931507371068911461d0 0.79540952268684039783d0
                    0.78131154925621904006d0 0.76701000445521327897d0 0.75249274283767175753d0
                    0.73774649123966817843d0 0.72275669453124924413d0 0.70750733322717218599d0
                    0.69198070642164433153d0 0.67615717164003941164d0 0.66001483068125788023d0
                    0.64352914709192166054d0 0.62667247617369248314d0 0.60941348178302190419d0
                    0.59171640472442321723d0 0.57354013382911894769d0 0.55483701054653677964d0
                    0.53555126726089574696d0 0.51561695214596137138d0 0.49495511793071150786d0
                    0.47346992804116821985d0 0.45104312257768319760d0 0.42752591183090860039d0
                    0.40272666565361542323d0 0.37639138197657090457d0 0.34817096847454187536d0
                    0.31756249210733678012d0 0.28379348552703770325d0 0.24556259105431105239d0
                    0.20033029278793829109d0 0.14153457290392534862d0)))
    (loop for i double-float from 0.0d0 to 1.0d0 by 0.01d0
          for idx from 0
          for expect = (elt expected idx)
          for actual = (sdm:fast-inv-cos i)
          do (is-true (close-to expect actual 1.0e-9)
                      "Expected (close-to ~f ~f 1.0e-9) for fast-inv-cos of ~f" expect actual i))))

(test fast-inv-tan
  (let ((expected #(-0.78539816340000001560d0 -0.78037306671719752771d0
                    -0.77529748665897668669d0 -0.77017091257094505430d0 -0.76499283950558694123d0
                    -0.75976276679548038206d0 -0.75448019713104286232d0 -0.74914463603321512775d0
                    -0.74375559163197202395d0 -0.73831257467886779455d0 -0.73281509873636863084d0
                    -0.72726268049888165290d0 -0.72165484021046044649d0 -0.71599110215245553857d0
                    -0.71027099518112324361d0 -0.70449405330065606456d0 -0.69865981626142570349d0
                    -0.69276783017663368103d0 -0.68681764815318269157d0 -0.68080883093455357713d0
                    -0.67474094755491564879d0 -0.66861357600470172624d0 -0.66242630390854417843d0
                    -0.65617872921685327370d0 -0.64987046091249367485d0 -0.64350111973402601695d0
                    -0.63707033891687436800d0 -0.63057776495359263436d0 -0.62402305837414995260d0
                    -0.61740589454688710180d0 -0.61072596450048954875d0 -0.60398297576703585676d0
                    -0.59717665324588897757d0 -0.59030674008792061169d0 -0.58337299859931057711d0
                    -0.57637521116392465004d0 -0.56931318118306750709d0 -0.56218673403121388610d0
                    -0.55499571802615732530d0 -0.54774000541186274038d0 -0.54041949335217487338d0
                    -0.53303410493341751852d0 -0.52558379017380585374d0 -0.51806852703749295408d0
                    -0.51048832245097863769d0 -0.50284321331951542522d0 -0.49513326754105374539d0
                    -0.48735858501518525276d0 -0.47951929864444498008d0 -0.47161557532524617242d0
                    -0.46364761692562606044d0 -0.45561566124688346324d0 -0.44751998296609607486d0
                    -0.43936089455640220391d0 -0.43113874718184036494d0 -0.42285393156344103227d0
                    -0.41450687881317543892d0 -0.40609806123227992769d0 -0.39762799307039459196d0
                    -0.38909723124188821952d0 -0.38050637599567904701d0 -0.37185607153482180820d0
                    -0.36314700658209786432d0 -0.35437991488783737726d0 -0.34555557567620842629d0
                    -0.33667481402623966602d0 -0.32773850118389241670d0 -0.31874755480157740406d0
                    -0.30970293910161172946d0 -0.30060566496023860550d0 -0.29145678990898915561d0
                    -0.28225741805034504361d0 -0.27300869988487214135d0 -0.26371183204722836724d0
                    -0.25436805694871394978d0 -0.24497866232431844691d0 -0.23554498068253196252d0
                    -0.22606838865652270631d0 -0.21655030625564100633d0 -0.20699219601658469525d0
                    -0.19739556205395608623d0 -0.18776194901034778950d0 -0.17809294090651683895d0
                    -0.16839015989263722561d0 -0.15865526490205830723d0 -0.14888995020943840419d0
                    -0.13909594389556534377d0 -0.12927500622161605670d0 -0.11942892791604167724d0
                    -0.10955952837769085406d0 -0.09966865379919882806d0 -0.08975817521506899987d0
                    -0.07982998647925587665d0 -0.06988600217741937348d0 -0.05992815547935879950d0
                    -0.04995839593744639401d0 -0.03997868723716416217d0 -0.02999100490610111594d0
                    -0.01999733398798865525d0 -0.00999966668853857765d0 0.00000000000000075287d0
                    0.00999966668854008339d0 0.01999733398799016099d0 0.02999100490610262168d0
                    0.03997868723716566791d0 0.04995839593744789975d0 0.05992815547936030524d0
                    0.06988600217742085841d0 0.07982998647925736158d0 0.08975817521507049868d0
                    0.09966865379920031298d0 0.10955952837769232511d0 0.11942892791604314828d0
                    0.12927500622161752775d0 0.13909594389556678706d0 0.14888995020943987524d0
                    0.15865526490205977828d0 0.16839015989263869666d0 0.17809294090651828224d0
                    0.18776194901034920504d0 0.19739556205395752952d0 0.20699219601658611078d0
                    0.21655030625564242186d0 0.22606838865652414960d0 0.23554498068253337806d0
                    0.24497866232431986244d0 0.25436805694871533756d0 0.26371183204722975502d0
                    0.27300869988487352913d0 0.28225741805034643139d0 0.29145678990899054339d0
                    0.30060566496023999328d0 0.30970293910161306172d0 0.31874755480157873633d0
                    0.32773850118389374897d0 0.33667481402624099829d0 0.34555557567620975856d0
                    0.35437991488783870953d0 0.36314700658209919659d0 0.37185607153482308496d0
                    0.38050637599568032376d0 0.38909723124188944077d0 0.39762799307039586871d0
                    0.40609806123228114894d0 0.41450687881317671568d0 0.42285393156344230903d0
                    0.43113874718184164170d0 0.43936089455640342516d0 0.44751998296609729611d0
                    0.45561566124688468449d0 0.46364761692562717066d0 0.47161557532524733816d0
                    0.47951929864444609031d0 0.48735858501518636299d0 0.49513326754105480010d0
                    0.50284321331951642442d0 0.51048832245097974791d0 0.51806852703749406430d0
                    0.52558379017380696396d0 0.53303410493341862875d0 0.54041949335217587258d0
                    0.54774000541186385060d0 0.55499571802615832450d0 0.56218673403121488530d0
                    0.56931318118306850629d0 0.57637521116392564924d0 0.58337299859931157631d0
                    0.59030674008792172192d0 0.59717665324588997677d0 0.60398297576703685596d0
                    0.61072596450049043693d0 0.61740589454688798998d0 0.62402305837415084078d0
                    0.63057776495359352253d0 0.63707033891687536720d0 0.64350111973402690513d0
                    0.64987046091249456303d0 0.65617872921685416188d0 0.66242630390854506661d0
                    0.66861357600470261442d0 0.67474094755491653697d0 0.68080883093455435429d0
                    0.68681764815318357975d0 0.69276783017663456921d0 0.69865981626142648064d0
                    0.70449405330065684172d0 0.71027099518112402077d0 0.71599110215245631572d0
                    0.72165484021046122365d0 0.72726268049888243006d0 0.73281509873636940799d0
                    0.73831257467886868273d0 0.74375559163197280110d0 0.74914463603321590490d0
                    0.75448019713104375050d0 0.75976276679548104820d0 0.76499283950558760736d0
                    0.77017091257094572043d0 0.77529748665897746385d0 0.78037306671719830486d0)))
    (loop for i double-float from -1.0d0 to 1.0d0 by 0.01d0
          for idx from 0
          for expect = (elt expected idx)
          for actual = (sdm:fast-inv-tan i)
          do (is-true (close-to expect actual 1.0e-9)
                      "Expected (close-to ~f ~f 1.0e-9) for fast-inv-tan of ~f" expect actual i))))

(test faster-sin
  (let ((expected #(0.00000000000000000000d0 0.00999983395076100082d0
                    0.01999867162435199977d0 0.02999551683492299864d0 0.03998937357926399955d0
                    0.04997924612812500167d0 0.05996413911753600834d0 0.06994305764012700988d0
                    0.07991500733644800458d0 0.08987899448628899890d0 0.09983402609999998678d0
                    0.10977911000981098455d0 0.11971325496115198250d0 0.12963547070397299366d0
                    0.13954476808406399146d0 0.14944015913437500020d0 0.15932065716633600494d0
                    0.16918527686117701414d0 0.17903303436124803905d0 0.18886294736133904526d0
                    0.19867403520000004313d0 0.20846531895086103936d0 0.21823582151395204409d0
                    0.22798456770702305030d0 0.23771058435686406862d0 0.24741290039062505124d0
                    0.25709054692713606549d0 0.26674255736822705121d0 0.27636796749004810536d0
                    0.28596581553438910062d0 0.29553514230000010876d0 0.30507499123391107476d0
                    0.31458440852275210142d0 0.32406244318407312344d0 0.33350814715766413654d0
                    0.34292057539687514911d0 0.35229878595993613377d0 0.36164184010127714552d0
                    0.37094880236284816233d0 0.38021874066543914772d0 0.38945072640000016895d0
                    0.39864383451896118205d0 0.40779714362755214996d0 0.41690973607512321619d0
                    0.42598069804646421233d0 0.43500911965312522023d0 0.44399409502473619060d0
                    0.45293472240032722764d0 0.46183010421964820758d0 0.47067934721448922986d0
                    0.47948156250000018019d0 0.48823586566601123726d0 0.49694137686835221368d0
                    0.50559722092017322925d0 0.51420252738326421849d0 0.52275643065937515974d0
                    0.53125807008153624889d0 0.53970659000537724026d0 0.54810113990044828647d0
                    0.55644087444153922384d0 0.56472495360000019016d0 0.57295254273506124321d0
                    0.58112281268515231236d0 0.58923493985922326122d0 0.59728810632806428327d0
                    0.60528149991562529841d0 0.61321431429033634863d0 0.62108574905642732755d0
                    0.62889500984524826510d0 0.63664130840658927912d0 0.64432386270000030493d0
                    0.65194189698611126893d0 0.65949464191795226231d0 0.66698133463227338158d0
                    0.67440121884086423609d0 0.68175354492187534294d0 0.68903757001113630221d0
                    0.69625255809347730285d0 0.70339778009404829717d0 0.71047251396963939651d0
                    0.71747604480000037874d0 0.72440766487916130600d0 0.73126667380675236529d0
                    0.73805237857932337597d0 0.74476409368166429648d0 0.75140114117812528693d0
                    0.75796285080393632771d0 0.76444856005652739306d0 0.77085761428684840268d0
                    0.77718936679068928441d0 0.78344317890000036986d0 0.78961842007421134682d0
                    0.79571446799155232110d0 0.80173070864037332406d0 0.80766653641046437517d0
                    0.81352135418437532266d0 0.81929457342873635017d0 0.82498561428557737329d0
                    0.83059390566364832420d0 0.83611888532973932531d0 0.84156000000000030781d0
                    0.84691670543126140736d0 0.85218846651235236056d0 0.85737475735542334476d0
                    0.86247506138726426350d0 0.86748887144062536425d0 0.87241568984553630184d0
                    0.87725502852062731129d0 0.88200640906444838230d0 0.88666936284678932179d0
                    0.89124343110000037171d0 0.89572816501031127245d0 0.90012312580915232463d0
                    0.90442788486447334151d0 0.90864202377206437777d0 0.91276513444687523702d0
                    0.91679681921433631153d0 0.92073669090167731266d0 0.92458437292924833262d0
                    0.92833949940183935201d0 0.93200171520000019143d0 0.93557067607136124021d0
                    0.93904604872195229781d0 0.94242751090752319154d0 0.94571475152486439431d0
                    0.94890747070312531086d0 0.95200537989513622783d0 0.95500820196872737711d0
                    0.95791567129804833236d0 0.96072753385488929290d0 0.96344354730000025810d0
                    0.96606348107441120110d0 0.96858711649075224237d0 0.97101424682457326831d0
                    0.97334467740566421590d0 0.97557822570937524631d0 0.97771472144793625247d0
                    0.97975400666177714371d0 0.98169593581084824141d0 0.98354037586593912046d0
                    0.98528720640000022613d0 0.98693631967946116035d0 0.98848762075555207662d0
                    0.98994102755562307649d0 0.99129647097446416115d0 0.99255389496562507201d0
                    0.99371325663273590845d0 0.99477452632082719042d0 0.99573768770764792269d0
                    0.99660273789498898811d0 0.99736968750000010164d0 0.99803856074651098318d0
                    0.99860939555635208631d0 0.99908224364067321766d0 0.99945717059126393256d0
                    0.99973425597187515201d0 0.99991359340953600565d0 0.99999529068587700298d0)))
    (loop for i double-float from 0.0d0 to (/ pi 2.0d0) by 0.01d0
          for idx from 0
          for expect = (elt expected idx)
          for actual = (sdm:faster-sin i)
          do (is-true (close-to expect actual 1.0e-9)
                      "Expected (close-to ~f ~f 1.0e-9) for faster-sin of ~f" expect actual i))))

(test faster-cos
  (let ((expected #(1.00000000000000000000d0 0.99995033037049996238d0
                    0.99980132592799997315d0 0.99955300001049995906d0 0.99920537484799998307d0
                    0.99875848156250002230d0 0.99821236016799996804d0 0.99756705957049995881d0
                    0.99682263756800004728d0 0.99597916085049997825d0 0.99503670499999996579d0
                    0.99399535449050002711d0 0.99285520268799998256d0 0.99161635185050001073d0
                    0.99027891312799998236d0 0.98884300656250001538d0 0.98730876108800003088d0
                    0.98567631453049997514d0 0.98394581360800004166d0 0.98211741393050000504d0
                    0.98019127999999999812d0 0.97816758521049995689d0 0.97604651184799995356d0
                    0.97382825109049997447d0 0.97151300300800003118d0 0.96910097656249993836d0
                    0.96659238960799997997d0 0.96398746889050002107d0 0.96128645004799995188d0
                    0.95848957761050002091d0 0.95559710499999994671d0 0.95260929453049991711d0
                    0.94952641740799992309d0 0.94634875373049998082d0 0.94307659248799990959d0
                    0.93971023156249999797d0 0.93624997772800000462d0 0.93269614665049993540d0
                    0.92904906288799993241d0 0.92530905989049994087d0 0.92147647999999993118d0
                    0.91755167445049989894d0 0.91353500336799986492d0 0.90942683577049987509d0
                    0.90522754956799988957d0 0.90093753156249989367d0 0.89655717744799989788d0
                    0.89208689181049982686d0 0.88752708812799985250d0 0.88287818877049994981d0
                    0.87814062499999989697d0 0.87331483697049994142d0 0.86840127372799991168d0
                    0.86340039321049988352d0 0.85831266224799984688d0 0.85313855656249981685d0
                    0.84787856076799983374d0 0.84253316837049985200d0 0.83710288176799985127d0
                    0.83158821225049983639d0 0.82598967999999983736d0 0.82030781409049979835d0
                    0.81454315248799979976d0 0.80869624205049983612d0 0.80276763852799981613d0
                    0.79675790656249978472d0 0.79066761968799981197d0 0.78449736033049977113d0
                    0.77824771980799978266d0 0.77191929833049977017d0 0.76551270499999979346d0
                    0.75902855781049982653d0 0.75246748364799964648d0 0.74583011829049983277d0
                    0.73911710640799976879d0 0.73232910156249975131d0 0.72546676620799965818d0
                    0.71853077169049961448d0 0.71152179824799977048d0 0.70444053501049963550d0
                    0.69728767999999963223d0 0.69006394013049965341d0 0.68277003120799961700d0
                    0.67540667793049968815d0 0.66797461388799961313d0 0.66047458156249960748d0
                    0.65290733232799968988d0 0.64527362645049968215d0 0.63757423308799965334d0
                    0.62980993029049958665d0 0.62198150499999949048d0 0.61408975305049962046d0
                    0.60613547916799959125d0 0.59811949697049948682d0 0.59004262896799952731d0
                    0.58190570656249951398d0 0.57370957004799949530d0 0.56545506861049954495d0
                    0.55714306032799953972d0 0.54877441217049949262d0 0.54034999999999944187d0
                    0.53187070857049945083d0 0.52333743152799949705d0 0.51475107141049947224d0
                    0.50611253964799940430d0 0.49742275656249934634d0 0.48868265136799937665d0
                    0.47989316217049937663d0 0.47105523596799936392d0 0.46216982865049927032d0
                    0.45323790499999927484d0 0.44426043869049935964d0 0.43523841228799931002d0
                    0.42617281725049938057d0 0.41706465392799929592d0 0.40791493156249936103d0
                    0.39872466828799935090d0 0.38949489113049928779d0 0.38022663600799933015d0
                    0.37092094773049921752d0 0.36157887999999926976d0 0.35220149541049927677d0
                    0.34278986544799916469d0 0.33334507049049921790d0 0.32386819980799930185d0
                    0.31436035156249919620d0 0.30482263280799914984d0 0.29525615949049921483d0
                    0.28566205644799924634d0 0.27604145741049912477d0 0.26639550499999908872d0
                    0.25672535073049917997d0 0.24703215500799913240d0 0.23731708713049903814d0
                    0.22758132528799912553d0 0.21782605656249909298d0 0.20805247692799910819d0
                    0.19826179125049903096d0 0.18845521328799907934d0 0.17863396569049894147d0
                    0.16879927999999910782d0 0.15895239665049909483d0 0.14909456496799899927d0
                    0.13922704317049905409d0 0.12935109836799896232d0 0.11946800656249900729d0
                    0.10957905264799894240d0 0.09968553041049887931d0 0.08978874252799884381d0
                    0.07989000057049899794d0 0.06999062499999886278d0 0.06009194517049887274d0
                    0.05019529932799882133d0 0.04030203461049897129d0 0.03041350704799894444d0
                    0.02053108156249894289d0 0.01065613196799897189d0 0.00079004097049895083d0)))
    (loop for i double-float from 0.0d0 to (/ pi 2.0d0) by 0.01d0
          for idx from 0
          for expect = (elt expected idx)
          for actual = (sdm:faster-cos i)
          do (is-true (close-to expect actual 1.0e-9)
                      "Expected (close-to ~f ~f 1.0e-9) for faster-cos of ~f" expect actual i))))

(test faster-tan
  (let ((expected #(0.00000000000000000000d0 0.01000031757033000013d0
                    0.02000254105055999931d0 0.03000857879019000127d0 0.04002034401791999818d0
                    0.05003975728124999967d0 0.06006874888607999924d0 0.07010926133630999979d0
                    0.08016325177343999742d0 0.09023269441617000342d0 0.10031958299999997608d0
                    0.11042593321683000229d0 0.12055378515455998645d0 0.13070520573668997777d0
                    0.14088229116191999801d0 0.15108716934375002183d0 0.16132200235008001266d0
                    0.17158898884280998631d0 0.18189036651744000528d0 0.19222841454267006189d0
                    0.20260545600000004485d0 0.21302386032333006649d0 0.22348604573856009625d0
                    0.23399448170319006590d0 0.24455169134592005808d0 0.25516025390625007851d0
                    0.26582280717408007797d0 0.27654204992931008533d0 0.28732074438144006301d0
                    0.29816171860917006775d0 0.30906786900000010609d0 0.32004216268983015636d0
                    0.33108764000256013516d0 0.34220741688969014138d0 0.35340468736992014520d0
                    0.36468272596875017655d0 0.37604489015808018060d0 0.38749462279581020629d0
                    0.39903545456544020631d0 0.41067100641567022556d0 0.42240499200000025670d0
                    0.43424122011633026208d0 0.44618359714656030679d0 0.45823612949619024759d0
                    0.47040292603392025450d0 0.48268820053125027769d0 0.49509627410208029152d0
                    0.50763157764231026103d0 0.52029865426944033047d0 0.53310216176217029016d0
                    0.54604687500000026468d0 0.55913768840283040173d0 0.57237961837056028358d0
                    0.58577780572269033676d0 0.59933751813792035534d0 0.61306415259375035554d0
                    0.62696323780608043119d0 0.64104043666881038721d0 0.65530154869344048318d0
                    0.66975251244867040068d0 0.68439940800000043097d0 0.69924845934933044234d0
                    0.71430603687456051265d0 0.72957865976919056283d0 0.74507299848192054537d0
                    0.76079587715625052180d0 0.77675427607008062925d0 0.79295533407531071379d0
                    0.80940635103744063006d0 0.82611479027517065177d0 0.84308828100000066108d0
                    0.86033462075583067019d0 0.87786177785856078781d0 0.89567789383569085260d0
                    0.91379128586592073269d0 0.93221044921875084732d0 0.95094405969408080104d0
                    0.97000097606181090448d0 0.98939024250144091965d0 1.00912109104167102558d0
                    1.02920294400000100765d0 1.04964541642233100127d0 1.07045831852256112526d0
                    1.09165165812219111530d0 1.11323564308992128957d0 1.13522068378125129406d0
                    1.15761739547808129025d0 1.18043660082831136648d0 1.20368933228544117142d0
                    1.22738683454817132379d0 1.25154056700000149149d0 1.27616220614883157936d0
                    1.30126364806656158457d0 1.32685701082869167422d0 1.35295463695392159664d0
                    1.37956909584375186917d0 1.40671318622208163518d0 1.43439993857481185024d0
                    1.46264261758944180691d0 1.49145472459467187676d0 1.52085000000000203357d0
                    1.55084242573533193088d0 1.58144622769056208966d0 1.61267587815519219951d0
                    1.64454609825792230637d0 1.67707186040625222390d0 1.71026839072608249914d0
                    1.74415117150131249169d0 1.77873594361344267156d0 1.81403870898117269839d0
                    1.85007573300000260907d0 1.88686354698183289536d0 1.92441895059456302697d0
                    1.96275901430169308348d0 2.00190108180192360976d0 2.04186277246875347302d0
                    2.08266198379008304897d0 2.12431689380781385523d0 2.16684596355744352181d0
                    2.21026793950767341812d0 2.25460185600000340145d0 2.29986703768833367079d0
                    2.34608310197856440027d0 2.39326996146819448441d0 2.44144782638592428015d0
                    2.49063720703125435207d0 2.54085891621408510588d0 2.59213407169431464538d0
                    2.64448409862144506732d0 2.69793073197417498932d0 2.75249601900000495647d0
                    2.80820232165483574249d0 2.86507231904256531863d0 2.92312900985469559245d0
                    2.98239571480992538000d0 3.04289607909375625638d0 3.10465407479808552793d0
                    3.16769400336081652370d0 3.23204049800544668258d0 3.29771852618067651264d0
                    3.36475339200000700401d0 3.43317073868133704195d0 3.50299655098656703700d0
                    3.57425715766119767025d0 3.64697923387392730632d0 3.72118980365625828810d0
                    3.79691624234208857658d0 3.87418627900731848968d0 3.95302799890944944750d0
                    4.03346984592717916485d0 4.11554062500001016645d0 4.19926950456783920629d0
                    4.28468601901056977965d0 4.37182007108769976611d0 4.46070193437792994473d0
                    4.55136225571876007479d0 4.64383205764609119370d0 4.73814274083382080960d0)))
    (loop for i double-float from 0.0d0 to (/ pi 4.0d0) by 0.01d0
          for idx from 0
          for expect = (elt expected idx)
          for actual = (sdm:faster-tan i)
          do (is-true (close-to expect actual 1.0e-9)
                      "Expected (close-to ~f ~f 1.0e-9) for faster-tan of ~f" expect actual i))))

(test faster-inv-tan
  (let ((expected #(-0.78540959999999993002d0 -0.78037406796821640054d0
                    -0.77529171622921366769d0 -0.77016127716554549920d0 -0.76498159158898226817d0
                    -0.75975159849789142275d0 -0.75447032556866977515d0 -0.74913688035031045231d0
                    -0.74375044213191776610d0 -0.73831025445369735660d0 -0.73281561823266383460d0
                    -0.72726588547500592430d0 -0.72166045354775087972d0 -0.71599875998305329006d0
                    -0.71028027778911528856d0 -0.70450451124241997292d0 -0.69867099213662064994d0
                    -0.69277927646409176443d0 -0.68682894150679407730d0 -0.68081958331374858862d0
                    -0.67475081454305274598d0 -0.66862226264699653466d0 -0.66243356837945688298d0
                    -0.65618438460536154810d0 -0.64987437539261927455d0 -0.64350321536750776374d0
                    -0.63707058931510462951d0 -0.63057619200692371919d0 -0.62401972823849971572d0
                    -0.61740091306022493178d0 -0.61071947218530553414d0 -0.60397514255925710547d0
                    -0.59716767307589901925d0 -0.59029682542534911427d0 -0.58336237506004684494d0
                    -0.57636411226535522179d0 -0.56930184332180500117d0 -0.56217539174655350998d0
                    -0.55498459960212798148d0 -0.54772932886101288741d0 -0.54040946281512936000d0
                    -0.53302490751972597671d0 -0.52557559326167280211d0 -0.51806147604260999184d0
                    -0.51048253906785490130d0 -0.50283879423242072715d0 -0.49513028359593602712d0
                    -0.48735708083868511942d0 -0.47951929269141385648d0 -0.47161706033196171273d0
                    -0.46365056074218718996d0 -0.45562000801905577774d0 -0.44752565463415405755d0
                    -0.43936779263627467884d0 -0.43114675479210173847d0 -0.42286291566038936018d0
                    -0.41451669259539508650d0 -0.40610854667568113685d0 -0.39763898355474641688d0
                    -0.38910855423029183653d0 -0.38051785572925395096d0 -0.37186753170606695607d0
                    -0.36315827295193170432d0 -0.35439081781317838260d0 -0.34556595251711441641d0
                    -0.33668451140404170863d0 -0.32774737706441575691d0 -0.31875548038039980270d0
                    -0.30970980047133672697d0 -0.30061136454292913811d0 -0.29146124764017272701d0
                    -0.28226057230433931888d0 -0.27301050813454574850d0 -0.26371227125468194030d0
                    -0.25436712368669595064d0 -0.24497637263145383146d0 -0.23554136965860464348d0
                    -0.22606350980708370124d0 -0.21654423059808464158d0 -0.20698501096252000520d0
                    -0.19738737008517051597d0 -0.18775286616789810257d0 -0.17808309511446390738d0
                    -0.16837968913965076223d0 -0.15864431530554129468d0 -0.14887867398794626950d0
                    -0.13908449727611427171d0 -0.12926354730898137468d0 -0.11941761455134188019d0
                    -0.10954851601343312817d0 -0.09965809341753435746d0 -0.08974821131527710438d0
                    -0.07982075515945609434d0 -0.06987762933421187705d0 -0.05992075514753291560d0
                    -0.04995206879009229245d0 -0.03997351926449492499d0 -0.02998706628906410224d0
                    -0.01999467818034148664d0 -0.00999832971851249627d0 0.00000000000000075277d0
                    0.00999832971851400201d0 0.01999467818034299238d0 0.02998706628906560798d0
                    0.03997351926449642379d0 0.04995206879009379125d0 0.05992075514753441440d0
                    0.06987762933421336198d0 0.07982075515945757926d0 0.08974821131527858931d0
                    0.09965809341753584238d0 0.10954851601343461309d0 0.11941761455134336511d0
                    0.12926354730898284573d0 0.13908449727611574276d0 0.14887867398794774054d0
                    0.15864431530554276573d0 0.16837968913965223328d0 0.17808309511446535067d0
                    0.18775286616789954586d0 0.19738737008517195926d0 0.20698501096252144849d0
                    0.21654423059808605712d0 0.22606350980708511678d0 0.23554136965860605901d0
                    0.24497637263145524700d0 0.25436712368669739393d0 0.26371227125468332808d0
                    0.27301050813454708077d0 0.28226057230434070666d0 0.29146124764017411479d0
                    0.30061136454293047038d0 0.30970980047133805924d0 0.31875548038040119048d0
                    0.32774737706441714469d0 0.33668451140404298538d0 0.34556595251711574868d0
                    0.35439081781317971487d0 0.36315827295193303659d0 0.37186753170606828833d0
                    0.38051785572925528323d0 0.38910855423029311329d0 0.39763898355474769364d0
                    0.40610854667568241361d0 0.41451669259539636325d0 0.42286291566039063694d0
                    0.43114675479210295972d0 0.43936779263627595560d0 0.44752565463415527880d0
                    0.45562000801905705449d0 0.46365056074218830018d0 0.47161706033196287846d0
                    0.47951929269141496670d0 0.48735708083868622964d0 0.49513028359593719285d0
                    0.50283879423242183737d0 0.51048253906785601153d0 0.51806147604261110207d0
                    0.52557559326167391234d0 0.53302490751972697591d0 0.54040946281513047023d0
                    0.54772932886101399763d0 0.55498459960212898068d0 0.56217539174655450918d0
                    0.56930184332180600038d0 0.57636411226535622099d0 0.58336237506004795517d0
                    0.59029682542535011347d0 0.59716767307590001845d0 0.60397514255925810467d0
                    0.61071947218530653334d0 0.61740091306022593098d0 0.62401972823850071492d0
                    0.63057619200692471839d0 0.63707058931510551769d0 0.64350321536750865192d0
                    0.64987437539262016273d0 0.65618438460536243628d0 0.66243356837945766014d0
                    0.66862226264699742284d0 0.67475081454305363415d0 0.68081958331374947679d0
                    0.68682894150679496548d0 0.69277927646409265261d0 0.69867099213662142709d0
                    0.70450451124242075007d0 0.71028027778911617673d0 0.71599875998305406721d0
                    0.72166045354775176790d0 0.72726588547500681248d0 0.73281561823266461175d0
                    0.73831025445369813376d0 0.74375044213191843223d0 0.74913688035031122947d0
                    0.75447032556867055231d0 0.75975159849789231092d0 0.76498159158898293430d0
                    0.77016127716554616534d0 0.77529171622921433382d0 0.78037406796821728872d0)))
    (loop for i double-float from -1.0d0 to 1.0d0 by 0.01d0
          for idx from 0
          for expect = (elt expected idx)
          for actual = (sdm:faster-inv-tan i)
          do (is-true (close-to expect actual 1.0e-9)
                      "Expected (close-to ~f ~f 1.0e-9) for faster-inv-tan of ~f" expect actual i))))

#|
*** NOTE: Fast Sin/Cos/Tan/InvSin/InvCos/InvTan values taken from this:

#include <stdio.h>
#include <math.h>

#define PI 3.141592653589793
#define HALF_PI 1.5707963267948966

#define MAX_SIN (PI / 2)
#define MAX_COS (PI / 2)
#define MAX_TAN (PI / 4)
#define MAX_INV_SIN 1.0
#define MAX_INV_COS 1.0
#define MAX_INV_TAN 1.0

double FastSin0 (double fAngle)
{
    double fASqr = fAngle*fAngle;
    double fResult = 7.61e-03;
    fResult *= fASqr;
    fResult -= 1.6605e-01;
    fResult *= fASqr;
    fResult += 1.0;
    fResult *= fAngle;
    return fResult;
}

double FastSin1 (double fAngle)
{
    double fASqr = fAngle*fAngle;
    double fResult = -2.39e-08;
    fResult *= fASqr;
    fResult += 2.7526e-06;
    fResult *= fASqr;
    fResult -= 1.98409e-04;
    fResult *= fASqr;
    fResult += 8.3333315e-03;
    fResult *= fASqr;
    fResult -= 1.666666664e-01;
    fResult *= fASqr;
    fResult += 1.0;
    fResult *= fAngle;
    return fResult;
}

double FastCos0 (double fAngle)
{
    double fASqr = fAngle*fAngle;
    double fResult = 3.705e-02;
    fResult *= fASqr;
    fResult -= 4.967e-01;
    fResult *= fASqr;
    fResult += 1.0;
    return fResult;
}

double FastCos1 (double fAngle)
{
    double fASqr = fAngle*fAngle;
    double fResult = -2.605e-07;
    fResult *= fASqr;
    fResult += 2.47609e-05;
    fResult *= fASqr;
    fResult -= 1.3888397e-03;
    fResult *= fASqr;
    fResult += 4.16666418e-02;
    fResult *= fASqr;
    fResult -= 4.999999963e-01;
    fResult *= fASqr;
    fResult += 1.0;
    return fResult;
}

double FastTan0 (double fAngle)
{
    double fASqr = fAngle*fAngle;
    double fResult = 2.033e-01;
    fResult *= fASqr;
    fResult += 3.1755e-01;
    fResult *= fASqr;
    fResult += 1.0;
    fResult *= fAngle;
    return fResult;
}

double FastTan1 (double fAngle)
{
    double fASqr = fAngle*fAngle;
    double fResult = 9.5168091e-03;
    fResult *= fASqr;
    fResult += 2.900525e-03;
    fResult *= fASqr;
    fResult += 2.45650893e-02;
    fResult *= fASqr;
    fResult += 5.33740603e-02;
    fResult *= fASqr;
    fResult += 1.333923995e-01;
    fResult *= fASqr;
    fResult += 3.333314036e-01;
    fResult *= fASqr;
    fResult += 1.0;
    fResult *= fAngle;
    return fResult;
}

double FastInvSin (double fValue)
{
    double fRoot = sqrt(1.0f-fValue);
    double fResult = -0.0187293;
    fResult *= fValue;
    fResult += 0.0742610;
    fResult *= fValue;
    fResult -= 0.2121144;
    fResult *= fValue;
    fResult += 1.5707288;
    fResult = HALF_PI - fRoot*fResult;
    return fResult;
}

double FastInvCos (double fValue)
{
    double fRoot = sqrt(1.0f-fValue);
    double fResult = -0.0187293;
    fResult *= fValue;
    fResult += 0.0742610;
    fResult *= fValue;
    fResult -= 0.2121144;
    fResult *= fValue;
    fResult += 1.5707288;
    fResult *= fRoot;
    return fResult;
}

double FastInvTan0 (double fValue)
{
    double fVSqr = fValue*fValue;
    double fResult = 0.0208351;
    fResult *= fVSqr;
    fResult -= 0.085133;
    fResult *= fVSqr;
    fResult += 0.180141;
    fResult *= fVSqr;
    fResult -= 0.3302995;
    fResult *= fVSqr;
    fResult += 0.999866;
    fResult *= fValue;
    return fResult;
}

double FastInvTan1 (double fValue)
{
    double fVSqr = fValue*fValue;
    double fResult = 0.0028662257;
    fResult *= fVSqr;
    fResult -= 0.0161657367;
    fResult *= fVSqr;
    fResult += 0.0429096138;
    fResult *= fVSqr;
    fResult -= 0.0752896400;
    fResult *= fVSqr;
    fResult += 0.1065626393;
    fResult *= fVSqr;
    fResult -= 0.1420889944;
    fResult *= fVSqr;
    fResult += 0.1999355085;
    fResult *= fVSqr;
    fResult -= 0.3333314528;
    fResult *= fVSqr;
    fResult += 1.0;
    fResult *= fValue;
    return fResult;
}

int main(int argc, char** argv) {
    printf("Fast\n");

    printf("Sin\n(");
    for(double i = 0.0; i < MAX_SIN; i += 0.01) {
        printf("%0.20fd0", FastSin1(i));
        if(i + 0.01 >= MAX_SIN) {
            printf(")\n");
        } else {
            printf(" ");
        }
    }

    printf("\nCos\n(");
    for(double i = 0.0; i < MAX_COS; i += 0.01) {
        printf("%0.20fd0", FastCos1(i));
        if(i + 0.01 >= MAX_COS) {
            printf(")\n");
        } else {
            printf(" ");
        }
    }

    printf("\nTan\n(");
    for(double i = 0.0; i < MAX_TAN; i += 0.01) {
        printf("%0.20fd0", FastTan1(i));
        if(i + 0.01 >= MAX_TAN) {
            printf(")\n");
        } else {
            printf(" ");
        }
    }

    printf("\nInvSin\n(");
    for(double i = 0.0; i < MAX_INV_SIN; i += 0.01) {
        printf("%0.20fd0", FastInvSin(i));
        if(i + 0.01 >= MAX_INV_SIN) {
            printf(")\n");
        } else {
            printf(" ");
        }
    }

    printf("\nInvCos\n(");
    for(double i = 0.0; i < MAX_INV_COS; i += 0.01) {
        printf("%0.20fd0", FastInvCos(i));
        if(i + 0.01 >= MAX_INV_COS) {
            printf(")\n");
        } else {
            printf(" ");
        }
    }

    printf("\nInvTan\n(");
    for(double i = -1.0; i < MAX_INV_TAN; i += 0.01) {
        printf("%0.20fd0", FastInvTan1(i));
        if(i + 0.01 >= MAX_INV_TAN) {
            printf(")\n");
        } else {
            printf(" ");
        }
    }



    printf("\nFaster\n");
    printf("Sin\n(");
    for(double i = 0.0; i < PI / 2; i += 0.01) {
        printf("%0.20fd0", FastSin0(i));
        if(i + 0.01 >= PI / 2) {
            printf(")\n");
        } else {
            printf(" ");
        }
    }

    printf("\nCos\n(");
    for(double i = 0.0; i < PI / 2; i += 0.01) {
        printf("%0.20fd0", FastCos0(i));
        if(i + 0.01 >= PI / 2) {
            printf(")\n");
        } else {
            printf(" ");
        }
    }

    printf("\nTan\n(");
    for(double i = 0.0; i < PI / 2; i += 0.01) {
        printf("%0.20fd0", FastTan0(i));
        if(i + 0.01 >= PI / 2) {
            printf(")\n");
        } else {
            printf(" ");
        }
    }

    printf("\nInvTan\n(");
    for(double i = -1.0; i < MAX_INV_TAN; i += 0.01) {
        printf("%0.20fd0", FastInvTan0(i));
        if(i + 0.01 >= MAX_INV_TAN) {
            printf(")\n");
        } else {
            printf(" ");
        }
    }

    return 0;
}
|#
