(in-package :cl-sdm-tests)
(in-suite cl-sdm-test-suite)

(test memory-utf-8-stream/read-char/utf-8
  (let ((stream (make-instance 'sdm:memory-utf-8-stream
                               :buffer (coerce (babel:string-to-octets "aikido合気道")
                                               '(vector (unsigned-byte 8))))))
    (is-true (eql (read-char stream) #\a))
    (is-true (eql (read-char stream) #\i))
    (is-true (eql (read-char stream) #\k))
    (is-true (eql (read-char stream) #\i))
    (is-true (eql (read-char stream) #\d))
    (is-true (eql (read-char stream) #\o))
    (is-true (eql (read-char stream) #\合))
    (is-true (eql (read-char stream) #\気))
    (is-true (eql (read-char stream) #\道))))

(test memory-utf-8-stream/read-line/utf-8
  (let* ((str (format nil "aikido~%合気道~%hello~%world"))
         (stream (make-instance 'sdm:memory-utf-8-stream
                                :buffer (coerce (babel:string-to-octets str)
                                                '(vector (unsigned-byte 8))))))
    (macrolet
        ((expect-str (str)
           (sdm:with-gensyms (val)
             `(let ((,val (read-line stream)))
                (is-true (equalp ,val ,str) "Expected to read ~s, not ~s" ,str ,val)))))
      (expect-str "aikido")
      (expect-str "合気道")
      (expect-str "hello")
      (expect-str "world"))

    ;; ECL does not handle READ-LINE correctly.
    #-ecl
    (is-true (null (read-line stream nil nil)))
    #-ecl
    (signals (end-of-file)
      (read-line stream))))

(test memory-utf-8-stream/write-string/utf-8
  (let ((stream (make-instance 'sdm:memory-utf-8-stream))
        (str "hello 合気道"))
    (is-true (string= (write-string str stream) str))
    (is-true (= (file-position stream) (babel:string-size-in-octets str)))
    (is-true (open-stream-p stream))
    (is-true (equalp (sdm:memory-stream-buffer stream)
                     (babel:string-to-octets str)))))

(test memory-utf-8-stream/write-char/utf-8
  (let ((stream (make-instance 'sdm:memory-utf-8-stream)))
    (is-true (char= (write-char #\合 stream) #\合))
    (is-true (char= (write-char #\気 stream) #\気))
    (is-true (char= (write-char #\道 stream) #\道))
    (is-true (= (file-position stream)
                (babel:string-size-in-octets "合気道")))))

(test memory-utf-8-stream/unread-char
  (macrolet
      ((expect-char (ch)
         (sdm:with-gensyms (val)
           `(let ((,val (read-char stream)))
              (is-true (char= ,val ,ch) "Expected to read character '~a', not '~a'" ,ch ,val))))

       (expect-line (limit str)
         (sdm:with-gensyms (val)
           `(let ((,val (sdm:read-line* stream ,limit)))
              (is-true (string= ,val ,str) "Expected to read string '~a', not '~a'" ,str ,val))))

       (expect-pos (pos)
         (sdm:with-gensyms (val)
           `(let ((,val (file-position stream)))
              (is-true (= ,val ,pos) "Expected to be at position ~a, not ~a" ,pos ,val)))))

    (let ((stream (make-instance 'sdm:memory-utf-8-stream))
          (str "abc合気道"))
      (write-string str stream)
      (file-position stream 0)
      (expect-char #\a)
      (expect-char #\b)
      (expect-pos 2)
      (unread-char #\b stream)
      (expect-pos 1)

      (expect-char #\b)
      (expect-char #\c)
      (expect-char #\合)
      (expect-char #\気)
      (expect-pos 9)
      (unread-char #\気 stream)
      (expect-pos 6)
      (expect-char #\気)
      (expect-char #\道))

    ;; Test that the INCOMPLETE-CHAR-BYTES slot and the extra buffers are reset
    ;; correctly.
    (let ((stream (make-instance 'sdm:memory-utf-8-stream))
          (str (format nil "abc~%合気道~%test")))
      (write-string str stream)
      (file-position stream 0)
      (expect-line 69 "abc")
      (expect-line 8 "合気")
      (unread-char #\気 stream)
      (expect-pos 7)
      (expect-char #\気)
      (expect-pos 10)
      (expect-line 69 "道")
      (expect-line 69 "test"))))

(test memory-utf-8-stream/unread-char/wont-do-twice
  (let ((stream (make-instance 'sdm:memory-utf-8-stream))
        (str "abc合気道"))
    (write-string str stream)
    (file-position stream 0)

    (finishes (read-char stream))
    (finishes (unread-char #\a stream))
    (signals (sdm:memory-stream-error "Failed to signal an error when doing two unreads in a row")
      (unread-char #\a stream))))
