(in-package :cl-sdm-tests)
(in-suite cl-sdm-test-suite)

(test memory-stream/creates-empty-stream
  (let ((stream (make-instance 'sdm:memory-stream)))
    (is-true (open-stream-p stream))
    (is-true (input-stream-p stream))
    (is-true (output-stream-p stream))
    (is-true (subtypep (stream-element-type stream)
                       '(unsigned-byte 8)))
    (is-true (not (sdm:memory-stream-read-only-p stream)))
    (is-true (zerop (file-position stream)))))

(test memory-stream/almost-no-ops
  (let ((stream (make-instance 'sdm:memory-stream)))
    (finishes (finish-output stream))
    (finishes (clear-output stream))
    (finishes (force-output stream))))

(test memory-stream/creates-with-buffer
  (let* ((buf (sdm:new-array 69 sdm:t/uint8))
         (stream (make-instance 'sdm:memory-stream :buffer buf)))
    (is-true (typep (sdm:memory-stream-buffer! stream)
                    'sdm:t/memory-stream-buffer))
    (is-true (eq (sdm:memory-stream-buffer! stream) buf))
    (is-true (= (length (sdm:memory-stream-buffer! stream)) 69))
    (is-true (= (sdm:memory-stream-size stream)
                (length (sdm:memory-stream-buffer! stream))))
    (is-true (every #'zerop (sdm:memory-stream-buffer! stream)))
    (is-true (open-stream-p stream))
    (is-true (not (sdm:memory-stream-read-only-p stream)))
    (is-true (zerop (file-position stream)))))

(test memory-stream/creates-with-initial-size
  (let ((stream (make-instance 'sdm:memory-stream :initial-size 69)))
    (is-true (typep (sdm:memory-stream-buffer! stream)
                    'sdm:t/memory-stream-buffer))
    (is-true (= (length (sdm:memory-stream-buffer! stream)) 69))
    (is-true (= (sdm:memory-stream-size stream)
                (length (sdm:memory-stream-buffer! stream))))
    (is-true (every #'zerop (sdm:memory-stream-buffer! stream)))
    (is-true (open-stream-p stream))
    (is-true (not (sdm:memory-stream-read-only-p stream)))
    (is-true (zerop (file-position stream)))))

(test memory-stream/cant-use-buffer-and-initial-size
  (signals (error)
    (make-instance 'sdm:memory-stream
                   :initial-size 69
                   :buffer (sdm:new-array-with sdm:t/uint8 #(1 2 3 4 5)))))

(test memory-stream/get-buffer
  (let* ((buf (sdm:new-array 69 sdm:t/uint8))
         (stream (make-instance 'sdm:memory-stream :buffer buf)))
    (is-true (typep (sdm:memory-stream-buffer stream)
                    'sdm:t/memory-stream-buffer))
    (is-true (eq (sdm:memory-stream-buffer! stream) buf))
    (is-false (eq (sdm:memory-stream-buffer stream) buf))
    (is-true (equalp (sdm:memory-stream-buffer! stream)
                     (sdm:memory-stream-buffer stream)))))

(test memory-stream/read-byte
  (let ((stream (make-instance 'sdm:memory-stream
                               :buffer (sdm:new-array-with sdm:t/uint8
                                                           #(1 2 3 4 5)))))
    (is-true (eq (read-byte stream) 1))
    (is-true (eq (read-byte stream) 2))
    (is-true (= (file-position stream) 2))
    (is-true (eq (read-byte stream) 3))
    (is-true (eq (read-byte stream) 4))
    (is-true (eq (read-byte stream) 5))
    (is-true (= (file-position stream) 5))
    (is-true (null (read-byte stream nil nil)))
    (signals (end-of-file)
      (read-byte stream))))

(test memory-stream/set-file-position
  (let ((stream (make-instance 'sdm:memory-stream
                               :buffer (sdm:new-array-with sdm:t/uint8
                                                           #(1 2 3 4 5)))))
    (is-true (eq (read-byte stream) 1))
    (is-true (eq (read-byte stream) 2))
    (is-true (= (file-position stream) 2))

    (finishes (file-position stream 0))
    (is-true (= (file-position stream) 0))

    (is-true (eq (read-byte stream) 1))
    (is-true (eq (read-byte stream) 2))
    (is-true (= (file-position stream) 2))))

(test memory-stream/read-sequence
  (let ((stream (make-instance 'sdm:memory-stream
                               :buffer (sdm:new-array-with sdm:t/uint8
                                                           #(1 2 3 4 5 6 7 8 9 0))))
        (buf (sdm:new-array 4 sdm:t/uint8)))
    (is-true (= (read-sequence buf stream) (length buf)))
    (is-true (equalp buf #(1 2 3 4)))
    (is-true (file-position stream) 4)

    (is-true (= (read-sequence buf stream) (length buf)))
    (is-true (equalp buf #(5 6 7 8)))
    (is-true (file-position stream) 8)

    (is-true (= (read-sequence buf stream) 2))
    (is-true (equalp buf #(9 0 7 8)))
    (is-true (file-position stream) 10)))

(test memory-stream/read-char/basic
  (let ((stream (make-instance 'sdm:memory-stream
                               :buffer (coerce (sdm:string->bytes "hello world")
                                               '(vector (unsigned-byte 8))))))
    (is-true (eql (read-char stream) #\h))
    (is-true (eql (read-char stream) #\e))
    (is-true (eql (read-char stream) #\l))
    (is-true (eql (read-char stream) #\l))
    (is-true (eql (read-char stream) #\o))
    (is-true (eql (read-char stream) #\Space))
    (is-true (eql (read-char stream) #\w))
    (is-true (eql (read-char stream) #\o))
    (is-true (eql (read-char stream) #\r))
    (is-true (eql (read-char stream) #\l))
    (is-true (eql (read-char stream) #\d))
    (is-true (null (read-char stream nil nil)))
    (signals (end-of-file)
      (read-char stream))))

(test memory-stream/read-char/full-ascii-range
  (let ((stream (make-instance 'sdm:memory-stream
                               :buffer (loop with ret = (sdm:new-vector sdm:t/uint8)
                                             for code from 0 below 128
                                             do (vector-push-extend code ret)
                                             finally (return ret)))))
    (loop for code from 0 below 128 do
      (is-true (eql (read-char stream) (code-char code))))))

(test memory-stream/read-line
  (let* ((str (format nil "hello~%world~%multi~%line"))
         (stream (make-instance 'sdm:memory-stream
                               :buffer (coerce (babel:string-to-octets str)
                                               '(vector (unsigned-byte 8))))))
    (macrolet
        ((expect-str (str)
           (sdm:with-gensyms (val)
             `(let ((,val (read-line stream)))
                (is-true (equalp ,val ,str) "Expected the string ~s, not ~s" ,str ,val)))))
      (expect-str "hello")
      (expect-str "world")
      (expect-str "multi")
      (expect-str "line"))

    ;; ECL does not handle READ-LINE correctly.
    #-ecl
    (is-true (null (read-line stream nil nil)))
    #-ecl ;; ECL does not handle READ-LINE correctly.
    (signals (end-of-file)
      (read-line stream))))

(test memory-stream/write-byte
  (let ((stream (make-instance 'sdm:memory-stream)))
    (is-true (= (write-byte 1 stream) 1))
    (is-true (= (write-byte 2 stream) 2))
    (is-true (= (write-byte 3 stream) 3))
    (is-true (= (file-position stream) 3))
    (is-true (open-stream-p stream))
    (is-true (equalp (sdm:memory-stream-buffer stream) #(1 2 3)))
    (is-true (equalp (sdm:memory-stream-buffer! stream) #(1 2 3)))))

(test memory-stream/write-char/full-ascii-range
  (let ((stream (make-instance 'sdm:memory-stream)))
    (loop for code from 0 below 128
          for char = (code-char code)
          do (is-true (char= (write-char char stream) char))
             (is-true (= (file-position stream) (1+ code))))
    (let ((result (sdm:memory-stream-buffer stream)))
      (loop for byte across result
            for code from 0
            do (is-true (= byte code))))))

(test memory-stream/write-sequence
  (let ((stream (make-instance 'sdm:memory-stream))
        (buf (sdm:new-array-with sdm:t/uint8 #(1 2 3 4 5))))
    (is-true (eq (write-sequence buf stream) buf))
    (is-true (= (file-position stream) 5))
    (is-true (open-stream-p stream))
    (is-true (equalp (sdm:memory-stream-buffer stream) buf))))

(test memory-stream/write-string
  (let ((stream (make-instance 'sdm:memory-stream))
        (str "hello world"))
    (is-true (string= (write-string str stream) str))
    (is-true (= (file-position stream) (babel:string-size-in-octets str)))
    (is-true (open-stream-p stream))
    (is-true (equalp (sdm:memory-stream-buffer stream)
                     (sdm:string->bytes str :as-vector t)))))

(test memory-stream/unread-char
  (let ((stream (make-instance 'sdm:memory-stream))
        (str "abc123"))
    (write-string str stream)
    (file-position stream 0)

    (macrolet
        ((expect-char (ch)
           (sdm:with-gensyms (val)
             `(let ((,val (read-char stream)))
                (is-true (char= ,val ,ch) "Expected to read character '~a', not '~a'" ,ch ,val))))

         (expect-pos (pos)
           (sdm:with-gensyms (val)
             `(let ((,val (file-position stream)))
                (is-true (= ,val ,pos) "Expected to be at position ~a, not ~a" ,pos ,val)))))
      (expect-char #\a)
      (expect-char #\b)
      (expect-pos 2)
      (unread-char #\b stream)
      (expect-pos 1)

      (expect-char #\b)
      (expect-char #\c)
      (expect-char #\1)
      (expect-char #\2)
      (expect-pos 5)
      (unread-char #\2 stream)
      (expect-pos 4)
      (expect-char #\2)
      (expect-char #\3))))

(test memory-stream/unread-char/wont-do-twice
  (let ((stream (make-instance 'sdm:memory-stream))
        (str "abc123"))
    (write-string str stream)
    (file-position stream 0)

    (finishes (read-char stream))
    (finishes (unread-char #\a stream))
    (signals (sdm:memory-stream-error "Failed to signal an error when doing two unreads in a row")
      (unread-char #\a stream))))

(test memory-stream/read-line*
  (let ((stream (make-instance 'sdm:memory-stream))
        (str "This is a
test string
with multiple stuff
and lines"))
    (write-string str stream)
    (file-position stream 0)

    (macrolet
        ((expect-read (size expected)
           (sdm:with-gensyms (line)
             `(let ((,line (sdm:read-line* stream ,size)))
                (is-true (string= ,line ,expected) "Expected ~s, not ~s" ,expected ,line))))

         (expect-pos (pos)
           (sdm:with-gensyms (val)
           `(let ((,val (file-position stream)))
              (is-true (= ,val ,pos) "Expected to be at position ~a, not ~a" ,pos ,val)))))

      (expect-read 69 "This is a")
      (expect-pos 10)
      (expect-read 69 "test string")
      (expect-pos 22)

      (expect-read 14 "with multiple ")
      (expect-pos 36)

      (expect-read 5 "stuff")
      (expect-pos 41)
      (expect-read 69 "")
      (expect-pos 42)
      (expect-read 2 "an")
      (expect-pos 44)
      (expect-read 69 "d lines")
      (expect-pos 51)
      (expect-read 1 "")
      (expect-read 999 "")
      (expect-pos 51))))
