(in-package :cl-sdm-tests)

(in-suite cl-sdm-test-suite)

(test stream-excursions
  (let ((seq (make-array 9 :element-type '(unsigned-byte 8) :initial-contents '(1 2 3 4 5 6 7 8 9))))
    (flex:with-input-from-sequence (in seq)
      (sdm:with-stream-excursion (in 5)
        (is-true (= (read-byte in) 6)))
      (is-true (= (read-byte in) 1)))))

(test stream-conversions
  (let* ((str "Hello, world!")
         (seq (coerce (map 'vector #'(lambda (x) (char-code x)) str)
                      '(vector (unsigned-byte 8)))))
    (is-true (string= (sdm:bytes->string seq) str))
    (is-true (equalp (sdm:string->bytes str :as-vector t) seq)))

  (let* ((num 3735928559)
         (seq (coerce (reverse #(#xDE #xAD #xBE #xEF)) '(vector (unsigned-byte 8))))
         (list (coerce seq 'list)))
    (is-true (= (sdm:bytes->uint seq) num))
    (is-true (equalp (sdm:uint->byte-vector num) seq))
    (is-true (equalp (sdm:uint->bytes num) list))))

(test read-bytes
  (let ((bytes (sdm:new-array-with sdm:t/uint8 #(1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16))))
    ;;
    ;; Non-constant byte size tests
    ;;

    ;; Test with a non-constant number of bytes.
    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (loop for byte-size in '(2 3)
            for data = (sdm:read-bytes in byte-size)
            do (ecase byte-size
                 (2
                  (is-true (equalp data #(1 2))
                           "Didn't read the first two bytes correctly (non-constant size)"))
                 (3
                  (is-true (equalp data #(3 4 5))
                           "Didn't read the next three bytes correctly (non-constant size)")))))

    ;; Test with a non-constant number of bytes, and a small buffer size
    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (loop for byte-size in '(5 7)
            for data = (sdm:read-bytes in byte-size :buffer-size 2)
            do (ecase byte-size
                 (5
                  (is-true (equalp data #(1 2 3 4 5))
                           "Didn't read the first five bytes correctly (non-constant size, small buffer)"))
                 (7
                  (is-true (equalp data #(6 7 8 9 10 11 12))
                           "Didn't read the next seven bytes correctly (non-constant size, small buffer)")))))

    ;; Test with a non-constant number of bytes, and return a LIST
    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (loop for byte-size in '(2 3)
            for data = (sdm:read-bytes in byte-size :as-list t)
            do (ecase byte-size
                 (2
                  (is-true (listp data) "Didn't read a list of two bytes (non-constant size), read a ~a"
                           (type-of data))
                  (is-true (equalp data '(1 2))
                           "Didn't read the first two bytes correctly (non-constant size, as list)"))
                 (3
                  (is-true (listp data) "Didn't read a list of three bytes (non-constant size), read a ~a"
                           (type-of data))
                  (is-true (equalp data '(3 4 5))
                           "Didn't read the next three bytes correctly (non-constant size, as list)")))))

    ;; Test with a non-constant number of bytes, a small buffer size, and returning a list.
    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (loop for byte-size in '(5 7)
            for data = (sdm:read-bytes in byte-size :buffer-size 2 :as-list t)
            do (ecase byte-size
                 (5
                  (is-true (listp data) "Didn't read a list of five bytes (non-constant size, small buffer)")
                  (is-true (equalp data '(1 2 3 4 5))
                           "Didn't read the first five bytes correctly (non-constant size, small buffer, as list)"))
                 (7
                  (is-true (listp data) "Didn't read a list of seven bytes (non-constant size, small buffer)")
                  (is-true (equalp data '(6 7 8 9 10 11 12))
                           "Didn't read the next seven bytes correctly (non-constant size, small buffer, as list)")))))

    ;;
    ;; Constant byte size tests
    ;;

    ;; Test with a constant number of bytes.
    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (let ((data (sdm:read-bytes in 2)))
        (is-true (equalp data #(1 2)) "Didn't read the first two bytes correctly (constant size)")

        (setf data (sdm:read-bytes in 3))
        (is-true (equalp data #(3 4 5)) "Didn't read the next three bytes correctly (constant size)")))

    ;; Test with a constant number of bytes, and a small buffer size
    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (let ((data (sdm:read-bytes in 5 :buffer-size 2)))
        (is-true (equalp data #(1 2 3 4 5))
                 "Didn't read the first five bytes correctly (constant size, small buffer)")

        (setf data (sdm:read-bytes in 7 :buffer-size 2))
        (is-true (equalp data #(6 7 8 9 10 11 12))
                 "Didn't read the next seven bytes correctly (constant size, small buffer)")))

    ;; Test with a constant number of bytes, and return a LIST
    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (let ((data (sdm:read-bytes in 2 :as-list t)))
        (is-true (listp data) "Didn't read a list of two bytes (constant size), read a ~a" (type-of data))
        (is-true (equalp data '(1 2)) "Didn't read the first two bytes correctly (constant size, as list)")

        (setf data (sdm:read-bytes in 3 :as-list t))
        (is-true (listp data) "Didn't read a list of three bytes (constant size), read a ~a" (type-of data))
        (is-true (equalp data '(3 4 5)) "Didn't read the next three bytes correctly (constant size, as list)")))

    ;; Test with a constant number of bytes, a small buffer size, and returning a list.
    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (let ((data (sdm:read-bytes in 5 :buffer-size 2 :as-list t)))
        (is-true (listp data) "Didn't read a list of five bytes (constant size, small buffer), read a ~a"
                 (type-of data))
        (is-true (equalp data '(1 2 3 4 5))
                 "Didn't read the first five bytes correctly (constant size, small buffer, as list)")

        (setf data (sdm:read-bytes in 7 :buffer-size 2 :as-list t))
        (is-true (listp data) "Didn't read a list of seven bytes (constant size, small buffer), read a ~a"
                 (type-of data))
        (is-true (equalp data '(6 7 8 9 10 11 12))
                 "Didn't read the next seven bytes correctly (constant size, small buffer, as list)")))))

;;;
;;; Reading of integers.
;;;
;;; These arrays of bytes, and their values, were taken from a QOA audio file,
;;; then modified so their signed and unsigned would be different.  This was
;;; done using Okteta.
;;;

(test reads-correctly/read-uint8
  (let ((bytes (sdm:new-array-with sdm:t/uint8 #(#x69 #xF6 #xF5 #x54))))
    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (let ((result (sdm:read-uint8 in)))
        (is-true (= result #x69) "Didn't read #x69, read #x~2,'0x instead" result)))))

(test reads-correctly/read-uint16
  (let ((bytes (sdm:new-array-with sdm:t/uint8 #(#x69 #xF6 #xF5 #x54))))
    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (let ((result (sdm:read-uint16 in)))
        (is-true (= result 63081) "Didn't read 63081, read ~a instead" result)))

    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (let ((result (sdm:read-uint16/be in)))
      (is-true (= result 27126) "Didn't read 63081, read ~a instead" result)))))

(test reads-correctly/read-uint32
  (let ((bytes (sdm:new-array-with sdm:t/uint8 #(#x69 #xF6 #xF5 #xFF))))
    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (let ((result (sdm:read-uint32 in)))
        (is-true (= result 4294309481) "Didn't read 4294309481, read ~a instead" result)))

    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (let ((result (sdm:read-uint32/be in)))
        (is-true (= result 1777792511) "Didn't read 1777792511, read ~a instead" result)))))

(test reads-correctly/read-uint64
  (let ((bytes (sdm:new-array-with sdm:t/uint8 #(#xF6 #xF6 #xF5 #xFF #x02 #x00 #xAC #xFF))))
    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (let ((result (sdm:read-uint64 in)))
        (is-true (= result 18423100188550100726) "Didn't read 18423100188550100726, read ~a instead" result)))

    (sdm:with-memory-stream (in :buffer bytes :read-only t)
      (let ((result (sdm:read-uint64/be in)))
        (is-true (= result 17795681453200157951) "Didn't read 17795681453200157951, read ~a instead" result)))))

;;;
;;; READ-LINE*
;;;

(test read-line*
  (with-input-from-string (in "This is a
test string
with multiple
lines")
    (macrolet
        ((expect-read (size expected)
           (sdm:with-gensyms (line)
             `(let ((,line (sdm:read-line* in ,size)))
                (is-true (string= ,line ,expected) "Expected ~s, not ~s" ,expected ,line)))))
      (expect-read 69 "This is a")
      (is-true (= (file-position in) 10))
      (expect-read 69 "test string")
      (is-true (= (file-position in) 22))
      (expect-read 5 "with ")
      (is-true (= (file-position in) 27))
      (expect-read 8 "multiple")
      (is-true (= (file-position in) 35))
      (expect-read 69 "")
      (is-true (= (file-position in) 36)))))
