(in-package :cl-sdm-tests)

(in-suite cl-sdm-test-suite)

(defmacro with-ini-parser ((ini stream) &body forms)
  `(let ((,ini (cl-sdm/ini:make-ini-parser ,stream)))
     ,@forms))

(test ini-parser/skip-whitespace
  (with-input-from-string (in "")
    (with-ini-parser (parser in)
      (cl-sdm/ini::skip-whitespace parser)
      (is-true (zerop (file-position (cl-sdm/ini::ini-parser-stream parser)))
               "Parser stream position is not at zero after reading an empty string.")))

  (with-input-from-string (in "     ")
    (with-ini-parser (parser in)
      (cl-sdm/ini::skip-whitespace parser)
      (is-true (= (file-position (cl-sdm/ini::ini-parser-stream parser)) 5)
               "Parser stream position is not at 5 after reading five spaces, it's at ~a"
               (file-position (cl-sdm/ini::ini-parser-stream parser)))))

  (with-input-from-string (in "     c")
    (with-ini-parser (parser in)
      (cl-sdm/ini::skip-whitespace parser)
      (is-true (= (file-position (cl-sdm/ini::ini-parser-stream parser)) 5)
               "Parser stream position is not at 5 after reading five spaces, it's at ~a"
               (file-position (cl-sdm/ini::ini-parser-stream parser)))
      (is-true (char= (read-char (cl-sdm/ini::ini-parser-stream parser)) #\c)
               "Parser stream did not read a #\c")))

  (with-input-from-string (in "x")
    (with-ini-parser (parser in)
      (cl-sdm/ini::skip-whitespace parser)
      (is-true (= (file-position (cl-sdm/ini::ini-parser-stream parser)) 0)
               "Parser stream position is not at zero after reading no spaces before an #\x")
      (is-true (char= (read-char (cl-sdm/ini::ini-parser-stream parser)) #\x)
               "Parser stream did not read an #\x")))

  (with-input-from-string (in "a     q")
    (with-ini-parser (parser in)
      (cl-sdm/ini::skip-whitespace parser)
      (is-true (= (file-position (cl-sdm/ini::ini-parser-stream parser)) 0)
               "Parser stream position is not at zero after reading no spaces before an #\a")
      (is-true (char= (read-char (cl-sdm/ini::ini-parser-stream parser)) #\a)
               "Parser stream did not read an #\a")
      (cl-sdm/ini::skip-whitespace parser)
      (is-true (= (file-position (cl-sdm/ini::ini-parser-stream parser)) 6)
               "Parser stream position is not at 6 after reading an #\a followed by five spaces, it's at ~a"
               (file-position (cl-sdm/ini::ini-parser-stream parser)))
      (is-true (char= (read-char (cl-sdm/ini::ini-parser-stream parser)) #\q)
               "Parser stream did not read a #\q"))))

(test ini-parser/skip-to-eol
  (with-input-from-string (in "")
    (with-ini-parser (parser in)
      (cl-sdm/ini::skip-to-eol parser)
      (is-true (zerop (file-position (cl-sdm/ini::ini-parser-stream parser)))
               "Parser stream position is not at zero after reading an empty string.")))

  (with-input-from-string (in "hello")
    (with-ini-parser (parser in)
      (cl-sdm/ini::skip-to-eol parser)
      (is-true (= (file-position (cl-sdm/ini::ini-parser-stream parser)) 5)
               "Parser stream position is not at 5 after skipping to the end of line, it's at ~a"
               (file-position (cl-sdm/ini::ini-parser-stream parser)))))

  (with-input-from-string (in (format nil "this is a test~%with a newline"))
    (with-ini-parser (parser in)
      (cl-sdm/ini::skip-to-eol parser)
      (is-true (= (file-position (cl-sdm/ini::ini-parser-stream parser)) 15)
               "Parser stream position is not at 15 after skipping to the end of line, it's at ~a"
               (file-position (cl-sdm/ini::ini-parser-stream parser)))
      (is-true (equalp "with a newline" (read-line (cl-sdm/ini::ini-parser-stream parser)))
               "Parser stream did not return the correct string."))))

(test ini-parser/read-until
  (with-input-from-string (in "")
    (with-ini-parser (parser in)
      (let ((str (cl-sdm/ini::read-until parser #\] nil nil)))
        (is-true (zerop (length str)) "Parser read something when it shouldn't have"))))

  (with-input-from-string (in "     ")
    (with-ini-parser (parser in)
      (let ((str (cl-sdm/ini::read-until parser #\] nil nil)))
        (is-true (equalp str "") "Did not drop five spaces"))))

  (with-input-from-string (in "   ]      ")
    (with-ini-parser (parser in)
      (let ((str (cl-sdm/ini::read-until parser #\] nil nil)))
        (is-true (equalp str "") "Did not drop three spaces"))))

  (with-input-from-string (in "hello]      ")
    (with-ini-parser (parser in)
      (let ((str (cl-sdm/ini::read-until parser #\] nil nil)))
        (is-true (equalp str "hello") "Did not read 'hello', got ~s" str))))

  (with-input-from-string (in (format nil "hello~%world]      "))
    (with-ini-parser (parser in)
      (let ((str (cl-sdm/ini::read-until parser #\] t nil)))
        (is-true (equalp str "hello") "Did not stop at the newline, got ~s" str))))

  (with-input-from-string (in "hello     ]      ")
    (with-ini-parser (parser in)
      (let ((str (cl-sdm/ini::read-until parser #\] nil nil)))
        (is-true (equalp str "hello") "Did not read 'hello' and drop spaces, got ~s" str))))

  (with-input-from-string (in "hello   ]      ")
    (with-ini-parser (parser in)
      (let ((str (cl-sdm/ini::read-until parser #\] nil t)))
        (is-true (equalp str "hello   ") "Did not read 'hello   ' with the spaces, got ~s" str))))

  (with-input-from-string (in (format nil "hello     ~%world]      "))
    (with-ini-parser (parser in)
      (let ((str (cl-sdm/ini::read-until parser #\] t t)))
        (is-true (equalp str "hello     ") "Did not stop at the newline and keep the spaces, got ~s" str)))))

(test ini-parser/parse-ini/empty-ini
  (with-input-from-string (in "")
    (is-true (zerop (hash-table-count (cl-sdm/ini:parse-ini in)))
             "Could not parse an empty INI")))

(test ini-parser/parse-ini/basic-section-name
  (with-input-from-string (in "[valid]")
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (= (hash-table-count ini) 1) "No sections parsed")
      (is-true (hash-table-p (gethash "valid" ini))
               "The [valid] section returned a ~a, not a HASH-TABLE"
               (type-of (gethash "valid" ini)))))

  (with-input-from-string (in "[valid with spaces]")
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (= (hash-table-count ini) 1) "No [valid with spaces] parsed")
      (is-true (hash-table-p (gethash "valid with spaces" ini))
               "The [valid with spaces] section returned a ~a, not a HASH-TABLE"
               (type-of (gethash "valid with spaces" ini)))))

  (with-input-from-string (in "[  valid with spaces around  ]")
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (= (hash-table-count ini) 1) "No [valid with spaces around] section parsed")
      (is-true (hash-table-p (gethash "valid with spaces around" ini))
               "The [valid with spaces around] section returned a ~a, not a HASH-TABLE"
               (type-of (gethash "valid with spaces" ini))))))

(test ini-parser/parse-ini/section-name-with-trailing-spaces
  (with-input-from-string (in "[valid]     ")
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (= (hash-table-count ini) 1) "No sections parsed")
      (is-true (hash-table-p (gethash "valid" ini))
               "The [valid] section returned a ~a, not a HASH-TABLE"
               (type-of (gethash "valid" ini))))))

(test ini-parser/parse-ini/section-name-with-newline
  (with-input-from-string (in "[valid]
")
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (= (hash-table-count ini) 1) "No sections parsed")
      (is-true (hash-table-p (gethash "valid" ini))
               "The [valid] section returned a ~a, not a HASH-TABLE"
               (type-of (gethash "valid" ini))))))

(test ini-parser/parse-ini/section-name-with-spaces-and-newline
  (with-input-from-string (in (format nil "[valid]     ~% "))
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (= (hash-table-count ini) 1) "No sections parsed")
      (is-true (hash-table-p (gethash "valid" ini))
               "The [valid] section returned a ~a, not a HASH-TABLE"
               (type-of (gethash "valid" ini))))))

(test ini-parser/parse-ini/section-name-with-junk
  (with-input-from-string (in "[valid] a")
    (signals (cl-sdm/ini:ini-error "Junk after the section name did not signal an error")
      (cl-sdm/ini:parse-ini in)))

  (with-input-from-string (in "[valid] a
")
    (signals (cl-sdm/ini:ini-error "Junk after the section name did not signal an error")
      (cl-sdm/ini:parse-ini in)))


  (with-input-from-string (in (format nil "[valid] a    ~%"))
    (signals (cl-sdm/ini:ini-error "Junk after the section name did not signal an error")
      (cl-sdm/ini:parse-ini in))))

(test ini-parser/parse-ini/empty-section-name
  (with-input-from-string (in "[]")
    (signals (cl-sdm/ini:ini-error "Empty section name did not signal an error")
      (cl-sdm/ini:parse-ini in))))

(test ini-parser/parse-ini/section-names-with-newlines
  (with-input-from-string (in "[
invalid]")
    (signals (cl-sdm/ini:ini-error "Section name with a newline at the start did not signal an error")
      (cl-sdm/ini:parse-ini in)))

  (with-input-from-string (in "[invalid
]")
    (signals (cl-sdm/ini:ini-error "Section name with a newline at the end did not signal an error")
      (cl-sdm/ini:parse-ini in)))

  (with-input-from-string (in "[inv
alid]")
    (signals (cl-sdm/ini:ini-error "Section name with a newline in the middle did not signal an error")
      (cl-sdm/ini:parse-ini in))))

(test ini-parser/parse-ini/valid-key-names
  (with-input-from-string (in "[section]
a = 1
b = 2

c = 3
d = 4")
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (= (hash-table-count ini) 1) "No sections parsed")
      (let ((section (gethash "section" ini)))
        (is-true (hash-table-p section) "The [section] section returned a ~a, not a HASH-TABLE"
                 (type-of section))

        (is-true (eq (gethash "a" section) 1) "The 'a' key did not return 1")
        (is-true (eq (gethash "b" section) 2) "The 'b' key did not return 2")
        (is-true (eq (gethash "c" section) 3) "The 'c' key did not return 3")
        (is-true (eq (gethash "d" section) 4) "The 'd' key did not return 4")))))

(test ini-parser/parse-ini/string-and-integer-values
  (with-input-from-string (in "[section]
a = 1
b = Foo")
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (= (hash-table-count ini) 1) "No sections parsed")
      (let ((section (gethash "section" ini)))
        (is-true (hash-table-p section) "The [section] section returned a ~a, not a HASH-TABLE"
                 (type-of section))
        (is-true (eq (gethash "a" section) 1) "The 'a' key did not return 1")
        (is-true (equal (gethash "b" section) "Foo") "The 'b' key did not return 'Foo'")))))

(test ini-parser/parse-ini/spaces-around-key-vals
  (with-input-from-string (in "[section]
a= 1
b   =    2
c =3
d=4")
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (= (hash-table-count ini) 1) "No sections parsed")
      (let ((section (gethash "section" ini)))
        (is-true (hash-table-p section) "The [section] section returned a ~a, not a HASH-TABLE"
                 (type-of section))
        (is-true (eq (gethash "a" section) 1) "The 'a' key did not return 1")
        (is-true (eq (gethash "b" section) 2) "The 'b' key did not return 2")
        (is-true (eq (gethash "c" section) 3) "The 'c' key did not return 3")
        (is-true (eq (gethash "d" section) 4) "The 'd' key did not return 4"))))

  (with-input-from-string (in "[section]
a
= 1")
    (signals (cl-sdm/ini:ini-error "A newline after the key name but before the = did not signal an error")
      (cl-sdm/ini:parse-ini in))))

(test ini-parser/parse-ini/comments
  (with-input-from-string (in "[section] ;; Test")
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (= (hash-table-count ini) 1) "No sections parsed")
      (is-true (hash-table-p (gethash "section" ini))
               "The [section] section returned a ~a, not a HASH-TABLE"
               (type-of (gethash "section" ini)))))

  (with-input-from-string (in "
;; Fine

[section] ;; Fine
a= 1 ;; Fine
;; b   =    2

;; Fine
c =3
d=4

;; Fine")
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (= (hash-table-count ini) 1) "No sections parsed")
      (let ((section (gethash "section" ini)))
        (is-true (hash-table-p section) "The [section] section returned a ~a, not a HASH-TABLE"
                 (type-of section))
        (is-true (eq (gethash "a" section) 1) "The 'a' key did not return 1")
        ;; No 'b', it's commented out
        (is-true (eq (gethash "c" section) 3) "The 'c' key did not return 3")
        (is-true (eq (gethash "d" section) 4) "The 'd' key did not return 4"))))

  (with-input-from-string (in "[section]
a = 1 # comment
b = 2 ;; not comment")
    (let ((ini (cl-sdm/ini:parse-ini (cl-sdm/ini:make-ini-parser in :comment-style #\#))))
      (is-true (= (hash-table-count ini) 1) "No sections parsed")
      (is-true (hash-table-p (gethash "section" ini))
               "The [section] section returned a ~a, not a HASH-TABLE"
               (type-of (gethash "section" ini)))
      (is-true (eq (gethash "a" (gethash "section" ini)) 1) "The 'a' key is not equal to 1")
      (is-true (equal (gethash "b" (gethash "section" ini)) "2 ;; not comment")
               "The 'b' key is not equal to '2 ;; not comment'")))

  (with-input-from-string (in "[section ;; foo ]
a = 1")
    (signals (cl-sdm/ini:ini-error "A comment in a section name did not signal an error")
      (cl-sdm/ini:parse-ini in)))

  (with-input-from-string (in "[ ;; technically an empty section ]
a = 1")
    (signals (cl-sdm/ini:ini-error "A comment in a (technically empty) section name did not signal an error")
      (cl-sdm/ini:parse-ini in)))

  (with-input-from-string (in "[section]
a ;; bad = 1")
    (signals (cl-sdm/ini:ini-error "A comment between a key and the = did not signal an error.")
      (cl-sdm/ini:parse-ini in)))

  (with-input-from-string (in "[section]
a ;; very bad
= 1")
    (signals (cl-sdm/ini:ini-error "A comment between a key and the = with a newline did not signal an error.")
      (cl-sdm/ini:parse-ini in)))

  (with-input-from-string (in "[section]
;a = 1")
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (zerop (hash-table-count (gethash "section" ini)))
               "A comment before a key did not parse as a comment.")))

  (with-input-from-string (in "[section]
a = ;; 1")
    (let ((parser (cl-sdm/ini:make-ini-parser in :allow-empty-values nil)))
      (signals (cl-sdm/ini:ini-error "A comment after the = did not result in an empty value.")
        (cl-sdm/ini:parse-ini parser)))))

(test ini-parser/parse-ini/empty-values
  (with-input-from-string (in "[section]
a =
b = 2")
    (let ((parser (cl-sdm/ini:make-ini-parser in :allow-empty-values nil)))
      (signals (cl-sdm/ini:ini-error ":ALLOW-EMPTY-VALUES NIL did not work.")
        (cl-sdm/ini:parse-ini parser))))

  (with-input-from-string (in "[section]
a =
b = 2
c = Foo")
    (let* ((parser (cl-sdm/ini:make-ini-parser in :allow-empty-values t))
           (ini (cl-sdm/ini:parse-ini parser))
           (section (gethash "section" ini)))
      (is-true (null (gethash "a" section)) "An empty value for 'a' did not produce a NIL, but ~w."
               (gethash "a" section))
      (is-true (eq (gethash "b" section) 2))
      (is-true (equal (gethash "c" section) "Foo")))))

(test ini-parser/parse-ini/multiple-unique-sections
  (with-input-from-string (in "[section1]
a = 1
c = 69

[section 2]
b = 2")
    (let* ((ini (cl-sdm/ini:parse-ini in))
           (s1 (gethash "section1" ini))
           (s2 (gethash "section 2" ini)))
      (is-true (= (hash-table-count ini) 2) "Did not parse two sections.")
      (is-true (hash-table-p s1) "[section1] did not result in a HASH-TABLE")
      (is-true (hash-table-p s2) "[section 2] did not result in a HASH-TABLE")
      (is-true (= (hash-table-count s1) 2) "[section1] Did not have two keys")
      (is-true (= (hash-table-count s2) 1) "[section 2] Did not have one key")
      (is-true (= (gethash "a" s1) 1) "[section1]'s 'a' did not equal 1")
      (is-true (= (gethash "c" s1) 69) "[section1]'s 'c' did not equal 69")
      (is-true (= (gethash "b" s2) 2) "[section 2]'s 'b' did not equal 2"))))

(test ini-parser/parse-ini/duplicate-sections
  (with-input-from-string (in "[section]
a = 1

[section]
b = 2")
    (signals (cl-sdm/ini:ini-error ":ON-DUPLICATE-SECTIONS :ERROR did not work.")
      (cl-sdm/ini:parse-ini (cl-sdm/ini:make-ini-parser in :on-duplicate-sections :error))))

  (with-input-from-string (in "[section]
a = 1
other = foo

[section]
b = 2")
    (let ((ini (cl-sdm/ini:parse-ini (cl-sdm/ini:make-ini-parser in :on-duplicate-sections :overwrite))))
      (is-true (= (hash-table-count ini) 1) ":ON-DUPLICATE-SECTIONS :OVERWRITE did not produce one section")
      (is-true (= (hash-table-count (gethash "section" ini)) 1) "[section] does not have one value")
0      (is-true (eq (gethash "b" (gethash "section" ini)) 2) "[section]'s 'b' does not equal 2")))

  (with-input-from-string (in "[section]
a = 1
other = foo

[section]
b = 2")
    (let* ((ini (cl-sdm/ini:parse-ini (cl-sdm/ini:make-ini-parser in :on-duplicate-sections :store)))
           (section (gethash "section" ini)))
      (is-true (hash-table-p ini) "Didn't get a HASH-TABLE for the results")
      (is-true (= (hash-table-count ini) 1) ":ON-DUPLICATE-SECTIONS :STORE did not produce one section")

      (is-true (vectorp section) "Did not get a VECTOR for two sections")
      (is-true (= (length section) 2) "Two sections did not produce a vector of two elements, but one with ~a"
               (length section))
      (is-true (every #'hash-table-p section) "Two sections did not produce two hash tables")

      (loop for sec across section do
        (case (hash-table-count sec)
          (1
           (is-true (eq (gethash "b" sec) 2) "The second [section]'s 'b' does not equal 2"))

          (2
           (is-true (eq (gethash "a" sec) 1) "The first [section]'s '1' does not equal 1")
           (is-true (equal (gethash "other" sec) "foo") "The first [section]'s 'other' does not equal 'foo'"))

          (otherwise
           (fail "A section with 1 or 2 keys was not found (got one with ~:d)" (hash-table-count sec))))))))

(test ini-parser/parse-ini/duplicate-keys
  (with-input-from-string (in "[section]
a = 1
a = 2")
    (signals (cl-sdm/ini:ini-error ":ON-DUPLICATE-KEYS :ERROR did not work.")
      (cl-sdm/ini:parse-ini (cl-sdm/ini:make-ini-parser in :on-duplicate-keys :error))))

  (with-input-from-string (in "[section]
a = 1
a = foo")
    (let ((ini (cl-sdm/ini:parse-ini (cl-sdm/ini:make-ini-parser in :on-duplicate-keys :overwrite))))
      (is-true (= (hash-table-count ini) 1) ":ON-DUPLICATE-KEYS :OVERWRITE did not produce one key")
      (is-true (= (hash-table-count (gethash "section" ini)) 1) "[section] does not have one value")
      (is-true (equal (gethash "a" (gethash "section" ini)) "foo") "[section]'s 'a' does not equal 'foo'")))

  (with-input-from-string (in "[section]
a = 1
a = 2
b = 3")
    (let* ((ini (cl-sdm/ini:parse-ini (cl-sdm/ini:make-ini-parser in :on-duplicate-keys :store)))
           (section (gethash "section" ini)))
      (is-true (hash-table-p ini) "Didn't get a HASH-TABLE for the results")
      (is-true (= (hash-table-count ini) 1) ":ON-DUPLICATE-KEYS :STORE did not produce one section")

      (is-true (hash-table-p section) "Did not get a HASH-TABLE for one section")
      (is-true (= (hash-table-count section) 2) "Did not get a HASH-TABLE with two elements for three keys")

      (let ((vals (gethash "a" section)))
        (is-true (vectorp vals) "Did not get a VECTOR for the two values for 'a'")
        (is-true (= (length vals) 2) "Two 'a' keys did not produce a vector of two elements, but one with ~a"
                 (length vals))
        (is-true (find 1 vals :test #'=) "Did not find the value 1")
        (is-true (find 2 vals :test #'=) "Did not find the value 2"))

      (is-true (eq (gethash "b" section) 3) "The key 'b' does not equal 3"))))

(test ini-parser/parse-ini/section-name-case
  (with-input-from-string (in "[section 1]
a = 1

[Section 1]
b = 2")
    (let ((ini (cl-sdm/ini:parse-ini in)))
      (is-true (= (hash-table-count ini) 2) "Did not produce two sections")
      (is-true (hash-table-p (gethash "section 1" ini)) "[section 1] did not produce a hash table")
      (is-true (hash-table-p (gethash "Section 1" ini)) "[Section 1] did not produce a hash table")
      (is-true (not (eq (gethash "section 1" ini)
                        (gethash "Section 1" ini)))
               "The two sections are the same when they shouldn't be")
      (is-true (eq (gethash "a" (gethash "section 1" ini)) 1) "The key 'a' is not equal to 1")
      (is-true (eq (gethash "b" (gethash "Section 1" ini)) 2) "The key 'b' is not equal to 2")))

  (with-input-from-string (in "[section 1]
a = 1

[Section 1]
b = 2")
    (signals (cl-sdm/ini:ini-error "Did not see the two sections as duplicates")
      (cl-sdm/ini:parse-ini (cl-sdm/ini:make-ini-parser in :downcase-section-names t))))

  (with-input-from-string (in "[section 1]
a = 1

[Section 1]
b = 2")
    (let* ((parser (cl-sdm/ini:make-ini-parser in :on-duplicate-sections :store
                                                  :downcase-section-names t))
           (ini (cl-sdm/ini:parse-ini parser))
           (sec1 (gethash "section 1" ini))
           (sec2 (gethash "Section 1" ini)))
      (is-true (= (hash-table-count ini) 1) "Did not produce one section from two")
      (is-true (vectorp sec1) "Did not produce a VECTOR for the sections")
      (is-true (null sec2) "Did not downcase the name properly"))))


(test ini-parser/parse-ini/key-name-case
  (with-input-from-string (in "[section]
a = 1
A = 2")
    (let* ((ini (cl-sdm/ini:parse-ini in))
           (section (gethash "section" ini)))
      (is-true (= (hash-table-count section) 2) "Did not get two keys")
      (is-true (eq (gethash "a" section) 1) "The key 'a' is not equal to 1")
      (is-true (eq (gethash "A" section) 2) "The key 'A' is not equal to 2")))

  (with-input-from-string (in "[section]
a = 1
A = 2")
    (signals (cl-sdm/ini:ini-error "Did not see the two keys as duplicates")
      (cl-sdm/ini:parse-ini (cl-sdm/ini:make-ini-parser in :downcase-key-names t))))

  (with-input-from-string (in "[section]
a = 1
A = 2")
    (let* ((parser (cl-sdm/ini:make-ini-parser in :on-duplicate-keys :store
                                                  :downcase-key-names t))
           (ini (cl-sdm/ini:parse-ini parser))
           (section (gethash "section" ini))
           (vals (gethash "a" section)))
      (is-true (= (hash-table-count section) 1) "Did not get a section with one key")
      (is-true (vectorp vals) "Did not produce a VECTOR for the keys")
      (is-true (= (length vals) 2) "Did not produce two values for one key")
      (is-true (find 1 vals :test #'=) "Did not find the value 1")
      (is-true (find 2 vals :test #'=) "Did not find the value 2"))))

(test ini-parser/parse-ini/expand-newline
  (with-input-from-string (in "[section]
a = foo\\nbar")
    (let* ((ini (cl-sdm/ini:parse-ini (cl-sdm/ini:make-ini-parser in :expand-newline nil)))
           (section (gethash "section" ini)))
      (is-true (string= (gethash "a" section) "foo\\nbar") "The \\n did not stay as-is")))

  (with-input-from-string (in "[section]
a = foo\\nbar")
    (let* ((ini (cl-sdm/ini:parse-ini (cl-sdm/ini:make-ini-parser in :expand-newline t)))
           (section (gethash "section" ini)))
      (is-true (string= (gethash "a" section) "foo
bar") "The \\n did not get expanded"))))
