(in-package :cl-sdm-tests)

(in-suite cl-sdm-test-suite)

(defmacro assert-file-matches (pattern path)
  `(is-true (sdm-file:file-matches-p ,pattern ,path)
            "Expected the pattern ~s to match the path ~s" ,pattern ,path))

(defmacro refute-file-matches (pattern path)
  `(is-false (sdm-file:file-matches-p ,pattern ,path)
             "Expected the pattern ~s to NOT match the path ~s" ,pattern ,path))

(defmacro file-match-signals (pattern path)
  `(signals (sdm-file:file-pattern-error "~s matching ~s did not raise a FILE-PATTERN-ERROR")
     (is-true (sdm-file:file-matches-p ,pattern ,path))))

(test file-matches-p/basic-patterns
  :description "FILE-MATCHES-P matches basic patterns"
  (assert-file-matches  "abc" "abc")
  (assert-file-matches "*" "abc")
  (assert-file-matches "*c" "abc")
  (assert-file-matches "a*" "a")
  (assert-file-matches "a*" "abc")
  (assert-file-matches "a*/b" "abc/b")
  (assert-file-matches "*x" "xxx"))

(test file-matches-p/multiple-expansions
  :description "FILE-MATCHES-P matches patterns with multiple expansions."
  (assert-file-matches "a*b*c*d*e*/f" "axbxcxdxe/f")
  (assert-file-matches "a*b*c*d*e*/f" "axbxcxdxexxx/f")
  (assert-file-matches "a*b?c*x" "abxbbxdbxebxczzx")
  (refute-file-matches "a*b?c*x" "abxbbxdbxebxczzy"))

(test file-matches-p/unicode-characters
  :description "FILE-MATCHES-P matches patterns with Unicode characters."
  (assert-file-matches "a?b" "a☺b")
  (refute-file-matches "a???b" "a☺b"))

(test file-matches-p/dont-match-path-separator
  :description "FILE-MATCHES-P does not match * to a path separator."
  (refute-file-matches "a*" "ab/c")
  (refute-file-matches "a*/b" "a/c/b")
  (refute-file-matches "a*b*c*d*e*/f" "axbxcxdxe/xxx/f")
  (refute-file-matches "a*b*c*d*e*/f" "axbxcxdxexxx/fff"))

(test file-matches-p/double-start
  :description "FILE-MATCHES-P matches double-star with path separators properly."
  (assert-file-matches "a**" "ab/c")
  (assert-file-matches "a**/b" "a/c/b")
  (assert-file-matches "a*b*c*d*e**/f" "axbxcxdxe/xxx/f")
  (assert-file-matches "a*b*c*d*e**/f" "axbxcxdxexxx/f")
  (refute-file-matches "a*b*c*d*e**/f" "axbxcxdxexxx/fff"))

(test file-matches-p/classes
  :description "FILE-MATCHES-P handles match classes."
  (assert-file-matches "ab[c]" "abc")
  (assert-file-matches "ab[b-d]" "abc")
  (refute-file-matches "ab[d-b]" "abc")
  (refute-file-matches "ab[e-g]" "abc")
  (assert-file-matches "ab[e-gc]" "abc")
  (refute-file-matches "ab[^c]" "abc")
  (refute-file-matches "ab[^b-d]" "abc")
  (assert-file-matches "ab[^e-g]" "abc")
  (assert-file-matches "a[^a]b" "a☺b")
  (refute-file-matches "a[^a][^a][^a]b" "a☺b")
  (assert-file-matches "[a-ζ]*" "α")
  (refute-file-matches "*[a-ζ]" "A"))

(test file-matches-p/escapes
  :description "FILE-MATCHES-P handles escapes."
  #-windows (assert-file-matches "a\\*b" "a*b")
  (refute-file-matches "a\\*b" "ab")
  #-windows (assert-file-matches "a\\**b" "a*bb")
  (refute-file-matches "a\\**b" "abb")
  #-windows (assert-file-matches "a*\\*b" "ab*b")
  (refute-file-matches "a*\\*b" "abb")

  (assert-file-matches "a\\[b\\]" "a[b]")
  (refute-file-matches "a\\[b\\]" "ab")
  (assert-file-matches "a\\[bb\\]" "a[bb]")
  (refute-file-matches "a\\[bb\\]" "abb")
  (assert-file-matches "a[b]\\[b\\]" "ab[b]")
  (refute-file-matches "a[b]\\[b\\]" "abb"))

(test file-matches-p/special-chars
  :description "FILE-MATCHES-P handles special characters."
  (refute-file-matches "a?b" "a/b")
  (refute-file-matches "a*b" "a/b"))

(test file-matches-p/classes
  :description "FILE-MATCHES-P classes escapes."
  (assert-file-matches "[\\]a]" "]")
  (assert-file-matches "[\\-]" "-")
  (assert-file-matches "[x\\-]" "x")
  (assert-file-matches "[x\\-]" "-")
  (refute-file-matches "[x\\-]" "z")
  (assert-file-matches "[\\-x]" "x")
  (assert-file-matches "[\\-x]" "-")
  (refute-file-matches "[\\-x]" "a")

  (file-match-signals "[]a]" "]")
  (file-match-signals "[-]" "-")
  (file-match-signals "[x-]" "x")
  (file-match-signals "[-x]" "x")
  (file-match-signals "\\" "a")
  (file-match-signals "[a-b-c]" "a")
  (file-match-signals "[" "a")
  (file-match-signals "[^" "a")
  (file-match-signals "[^bc" "a")
  (file-match-signals "a[" "a"))

(test file-matches-p/alternates
  :description "FILE-MATCHES-P matches alternates."
  (assert-file-matches "{abc,def}" "abc")
  (assert-file-matches "ab{c,}" "abc")
  (assert-file-matches "ab{c,}" "ab")
  (refute-file-matches "ab{d,e}" "abc")
  (assert-file-matches "ab{*,/cde}" "abcde")
  (assert-file-matches "ab{*,/cde}" "ab/cde")
  (assert-file-matches "ab{?,/}de" "abcde")
  (assert-file-matches "ab{?,/}de" "ab/de")
  (assert-file-matches "ab{{c,d}ef,}" "ab")
  (assert-file-matches "ab{{c,d}ef,}" "abcef")
  (assert-file-matches "ab{{c,d}ef,}" "abdef"))
