(in-package :cl-sdm-tests)

(in-suite cl-sdm-test-suite)

(defmacro with-bit-reader ((var &rest initial-contents) &body forms)
  (sdm:with-gensyms (seq in)
    `(let ((,seq (vector ,@initial-contents)))
       (flex:with-input-from-sequence (,in ,seq)
         (let ((,var (sdm:make-bit-reader ,in)))
           ,@forms)))))

(test bit-reader-read-bits
  (with-bit-reader (stream #b11010010 #b00101101 #b11111111)
    (is-true (= (sdm:bit-reader-cur-byte stream) #b11010010) "Initial byte incorrect")
    (is-true (= (sdm:bit-reader-bit-pos stream) 0) "Initial bit position incorrect")
    (is-true (= (sdm:bit-reader-read stream 8) #b11010010) "Reading 8 bits failed")
    (is-true (= (sdm:bit-reader-bit-pos stream) 0) "Bit position incorrect after reading 8 bits")

    (is-true (= (sdm:bit-reader-read stream 3) #b001))
    (is-true (= (sdm:bit-reader-bit-pos stream) 3))
    (is-true (= (sdm:bit-reader-read stream 5) #b01101))
    (is-true (= (sdm:bit-reader-bit-pos stream) 0))

    (is-true (= (sdm:bit-reader-read stream 1) 1))
    (is-true (= (sdm:bit-reader-bit-pos stream) 1))
    (is-true (= (sdm:bit-reader-read stream 1) 1))
    (is-true (= (sdm:bit-reader-bit-pos stream) 2))
    (is-true (= (sdm:bit-reader-read stream 6) #b111111))))

(test bit-reader-read-bits-across-bytes
  (with-bit-reader (stream #b11010010 #b00101101 #b11111111)
    (is-true (= (sdm:bit-reader-read stream 6) #b110100))
    (is-true (= (sdm:bit-reader-read stream 10) #b1000101101))
    (is-true (= (sdm:bit-reader-read stream 4) #b1111))))

(test bit-reader-sets-working-byte
  (with-bit-reader (stream #b11010010 #b00101101 #b11111111)
    (is-true (= (sdm:bit-reader-read stream 3) #b110))
    (is-true (= (sdm:bit-reader-bit-pos stream) 3))
    (setf (sdm:bit-reader-cur-byte stream) #b00011010)
    (is-true (= (sdm:bit-reader-read stream 5) #b11010))))

(test bit-reader-counts-zeros
  (with-bit-reader (stream #b11010010 #b00101101)
    (is-true (= (sdm:bit-reader-cur-byte stream) #b11010010) "Current byte incorrect")
    (is-true (= (sdm:bit-reader-bit-pos stream) 0) "Initial bit position incorrect")
    (is-true (= (sdm:bit-reader-read stream 8) #b11010010) "Reading 8 bits failed")
    (is-true (= (sdm:bit-reader-bit-pos stream) 0) "Bit position incorrect after reading 8 bits")

    (is-true (= (sdm:bit-reader-count-zeros stream) 2) "Could not count 2 zeros")
    (is-true (= (sdm:bit-reader-bit-pos stream) 2) "Bit position incorrect")
    (is-true (= (sdm:bit-reader-read stream 6) #b101101) "Failed to read 6 bits"))

  (with-bit-reader (stream #b00000000 #b00101101)
    (is-true (= (sdm:bit-reader-count-zeros stream) 10) "Could not cound 10 zeros")
    (is-true (= (sdm:bit-reader-bit-pos stream) 2) "Bit position wrong after counting 10 zeros")))

(test bit-reader-counts-zeros-discarding-first-1-bit
  (with-bit-reader (stream #b11010010 #b00101101)
    (is-true (= (sdm:bit-reader-cur-byte stream) #b11010010) "Current byte incorrect")
    (is-true (= (sdm:bit-reader-bit-pos stream) 0) "Initial bit position incorrect")
    (is-true (= (sdm:bit-reader-read stream 8) #b11010010) "Reading 8 bits failed")
    (is-true (= (sdm:bit-reader-bit-pos stream) 0) "Bit position incorrect after reading 8 bits")

    (is-true (= (sdm:bit-reader-count-zeros stream t) 2) "Could not count 2 zeros")
    (is-true (= (sdm:bit-reader-bit-pos stream) 3) "Bit position incorrect")
    (is-true (= (sdm:bit-reader-read stream 5) #b01101) "Failed to read 5 bits"))

  (with-bit-reader (stream #b00000000 #b00101101)
    (is-true (= (sdm:bit-reader-count-zeros stream t) 10) "Could not cound 10 zeros")
    (is-true (= (sdm:bit-reader-bit-pos stream) 3) "Bit position wrong after counting 10 zeros")))

(test bit-reader-peeks
  (with-bit-reader (stream #b11010010 #b00101101)
    (is-true (= (sdm:bit-reader-read stream 1) 1) "First bit incorrect")
    (is-true (= (sdm:bit-reader-bit-pos stream) 1) "Bit position incorrect after first read")

    (is-true (= (sdm:bit-reader-peek stream 1) 1) "Peeking 1 bit failed")
    (is-true (= (sdm:bit-reader-bit-pos stream) 1) "Bit position incorrect after peeking 1 bit")

    (is-true (= (sdm:bit-reader-peek stream 3) #b101) "Peeking 3 bits failed")
    (is-true (= (sdm:bit-reader-bit-pos stream) 1) "Bit position incorrect after peeking 3 bits")

    (is-true (= (sdm:bit-reader-peek stream 7) #b1010010) "Peeking 7 bits failed")
    (is-true (= (sdm:bit-reader-bit-pos stream) 1) "Bit position incorrect after peeking 7 bits")

    (is-true (= (sdm:bit-reader-peek stream 10) #b1010010001) "Peeking 10 bits failed")
    (is-true (= (sdm:bit-reader-bit-pos stream) 1) "Bit position incorrect after peeking 10 bits")
    (is-true (= (sdm:bit-reader-cur-byte stream) #b11010010) "Current byte changed")

    (is-true (= (sdm:bit-reader-read stream 10) #b1010010001) "Reading 10 bits failed")
    (is-true (= (sdm:bit-reader-bit-pos stream) 3) "Bit position incorrect after reading 10 bits")))

(test bit-reader-read-into-seq
  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (let ((buf (sdm:new-array 3 'sdm:t/uint8)))
      (is-true (= (sdm:bit-reader-read-sequence stream buf) 3))
      (is-true (equalp buf #(#b11010010 #b00101101 #b11110000)))))

  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (let ((buf (sdm:new-array 6 'sdm:t/uint8)))
      (is-true (= (sdm:bit-reader-read-sequence stream buf :start 2 :end 5) 5))
      (is-true (equalp buf #(0 0 #b11010010 #b00101101 #b11110000 0)))))

  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (let ((buf (make-list 3 :initial-element 0)))
      (is-true (= (sdm:bit-reader-read-sequence stream buf) 3))
      (is-true (equalp buf '(#b11010010 #b00101101 #b11110000)))))

  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (let ((buf (make-list 6 :initial-element 0)))
      (is-true (= (sdm:bit-reader-read-sequence stream buf :start 2 :end 5) 5))
      (is-true (equalp buf '(0 0 #b11010010 #b00101101 #b11110000 0)))))

  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (let ((buf (make-list 6 :initial-element 0)))
      (sdm:bit-reader-read stream 1)
      (signals sdm:not-on-byte-error (sdm:bit-reader-read-sequence stream buf)))))

(test bit-reader-reads-new-seqs
  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (let ((buf (sdm:bit-reader-read-bytes stream 3)))
      (is-true (equalp buf #(#b11010010 #b00101101 #b11110000)))
      (is-true (typep buf 'sdm:t/uint8-array))))

  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (let ((buf (sdm:bit-reader-read-bytes stream 3 t)))
      (is-true (equalp buf '(#b11010010 #b00101101 #b11110000)))
      (is-true (listp buf))))

  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (let ((buf (sdm:bit-reader-read-bytes stream 100 t)))
      (is-true (equalp buf '(#b11010010 #b00101101 #b11110000 #b00001111 #b01111001)))
      (is-true (listp buf))))

  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (let ((buf (sdm:bit-reader-read-bytes! stream 3)))
      (is-true (equalp buf #(#b11010010 #b00101101 #b11110000)))
      (is-true (typep buf 'sdm:t/uint8-array))))

  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (let ((buf (sdm:bit-reader-read-bytes! stream 5)))
      (is-true (equalp buf #(#b11010010 #b00101101 #b11110000 #b00001111 #b01111001)))
      (is-true (typep buf 'sdm:t/uint8-array))))

  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (signals end-of-file (sdm:bit-reader-read-bytes! stream 6)))

  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (is-true (eq (sdm:bit-reader-read-bytes! stream 6 :eof-error-p nil :eof-value :lol) :lol)))

  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (let ((buf (sdm:bit-reader-read-bytes stream 3)))
      (is-true (equalp buf #(#b11010010 #b00101101 #b11110000)))
      (is-true (typep buf 'sdm:t/uint8-array))
      (is-true (= (sdm:bit-reader-read stream 5) #b00001))
      (is-true (= (sdm:bit-reader-bit-pos stream) 5)))))


(test bit-reader-advances-to-next-byte
  (with-bit-reader (stream #b11010010 #b00101101 #b11110000 #b00001111 #b01111001)
    (sdm:bit-reader-read stream 3)
    (is-true (= (sdm:bit-reader-bit-pos stream) 3))
    (is-true (= (sdm:bit-reader-cur-byte stream) #b11010010))
    (finishes (sdm:bit-reader-advance-to-next-byte stream))
    (is-true (= (sdm:bit-reader-bit-pos stream) 0))
    (is-true (= (sdm:bit-reader-cur-byte stream) #b00101101))))

(test bit-reader-reads-strings
  (with-bit-reader (stream 67 111 109 109 111 110 32 76 105 115 112 32 114 111 99 107 115 33)
    (is-true (string= (sdm:bit-reader-read-string stream 18) "Common Lisp rocks!")
             "Could not read string"))

  (with-bit-reader (stream 67 111 109 109 111 110 32 76 105 115 112 32 114 111 99 107 115 33 0 0 0 0 0)
    (is-true (string= (sdm:bit-reader-read-string stream 18) "Common Lisp rocks!")
             "Could not read string with nulls trimmed"))

  (with-bit-reader (stream 67 111 109 109 111 110 32 76 105 115 112 32 114 111 99 107 115 33)
    (is-true (string= (sdm:bit-reader-read-string stream 200) "Common Lisp rocks!")
             "Could not read correctly sized string"))

  (with-bit-reader (stream 67 111 109 109 111 110 32 76 105 115 112 32 114 111 99 107 115 33 0 0 0 0 0)
    (is-true (string= (sdm:bit-reader-read-string stream 200) "Common Lisp rocks!")
             "Could not read correctly sized string with nulls trimmed"))

  (with-bit-reader (stream 67 111 109 109 111 110 32 76 105 115 112 32 114 111 99 107 115 33)
    (signals end-of-file (string= (sdm:bit-reader-read-string! stream 200) "")))

  (with-bit-reader (stream 67 111 109 109 111 110 32 76 105 115 112 32 114 111 99 107 115 33 0 0 0 0 0)
    (signals end-of-file (string= (sdm:bit-reader-read-string! stream 200) "")))

  (with-bit-reader (stream 67 111 109 109 111 110 32 76 105 115 112 32 114 111 99 107 115 33 0 0 0 0 0)
    (is-true (string= (sdm:bit-reader-read-string stream 200 t)
                      (with-output-to-string (out)
                        (format out "Common Lisp rocks!")
                        (write-char #\Nul out)
                        (write-char #\Nul out)
                        (write-char #\Nul out)
                        (write-char #\Nul out)
                        (write-char #\Nul out)))
             "Could not read string keeping nulls")))
