(in-package :cl-sdm-tests)

(in-suite cl-sdm-test-suite)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defclass simple-utf-8-input-stream (sdm:binary-utf-8-input-stream)
  ((stream
    :initarg :stream)))

(defun make-simple-utf-8-input-stream (string)
  (make-instance 'simple-utf-8-input-stream
                 :stream (flex:make-in-memory-input-stream (babel:string-to-octets string))))

(defmethod trivial-gray-streams:stream-read-byte ((stream simple-utf-8-input-stream))
  (read-byte (slot-value stream 'stream) nil :eof))

(defmethod trivial-gray-streams:stream-file-position ((stream simple-utf-8-input-stream))
  (- (file-position (slot-value stream 'stream))
     (sdm:binary-utf-8-input-stream-back-buffer-byte-length stream)))

(defmacro with-simple-utf-8-input-stream ((stream string) &body forms)
  `(let ((,stream (make-simple-utf-8-input-stream ,string)))
     ,@forms))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defclass simple-utf-8-output-stream (sdm:binary-utf-8-output-stream)
  ((stream
    :initform (flex:make-in-memory-output-stream))))

(defmethod trivial-gray-streams:stream-write-sequence ((stream simple-utf-8-output-stream)
                                                       sequence start end &key &allow-other-keys)
  (write-sequence sequence (slot-value stream 'stream) :start (or start 0) :end end))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(test binary-utf-8-stream
  (with-simple-utf-8-input-stream (stream "abcdefg合気道")
    (macrolet
        ((expect-char (ch)
           (sdm:with-gensyms (result)
             `(let ((,result (read-char stream nil nil)))
                (is-true (eql ,result ,ch) "Expected to read ~s, not ~s" ,ch ,result)))))
      (expect-char #\a)
      (expect-char #\b)
      (expect-char #\c)
      (expect-char #\d)
      (expect-char #\e)
      (expect-char #\f)
      (expect-char #\g)
      (expect-char #\合)
      (expect-char #\気)
      (expect-char #\道)
      (is-true (null (read-char stream nil nil))))))

(test binary-utf-8-stream/low-level
  (with-simple-utf-8-input-stream (stream "a合")
    (multiple-value-bind (char size)
        (sdm:binary-utf-8-input-stream-do-read-char stream)
      (is-true (char= char #\a))
      (is-true (= size 1)))

    (multiple-value-bind (char size)
        (sdm:binary-utf-8-input-stream-do-read-char stream)
      (is-true (char= char #\合))
      (is-true (= size 3)))))

(test binary-utf-8-stream/read-line
  (with-simple-utf-8-input-stream (stream (format nil "hello~%world~%multi~%line"))
    (macrolet
        ((expect-line (str)
           (sdm:with-gensyms (result)
             `(let ((,result (read-line stream nil nil)))
                (is-true (equalp ,result ,str) "Expected to read ~s, not ~s" ,str ,result)))))
      (expect-line "hello")
      (expect-line "world")
      (expect-line "multi")
      (expect-line "line")

      #-ecl ;; ECL does not handle READ-LINE correctly.
      (let ((result (read-line stream nil nil)))
        (is-true (null result) "Expected NIL, not ~s" result)))))

(test binary-utf-8-stream/unread-char
  (with-simple-utf-8-input-stream (stream "abc合気道")
    (macrolet
        ((expect-pos (pos)
           (sdm:with-gensyms (val)
             `(let ((,val (file-position stream)))
                (is-true (= ,val ,pos) "Expected to be at position ~a, not ~a" ,pos ,val)))))
      (is-true (char= (read-char stream) #\a))
      (is-true (char= (read-char stream) #\b))
      (expect-pos 2)
      (unread-char #\b stream)
      (expect-pos 1)

      (is-true (char= (read-char stream) #\b))
      (is-true (char= (read-char stream) #\c))
      (is-true (char= (read-char stream) #\合))
      (is-true (char= (read-char stream) #\気))
      (expect-pos 9)
      (unread-char #\気 stream)
      (expect-pos 6)
      (is-true (char= (read-char stream) #\気))
      (is-true (char= (read-char stream) #\道)))))

(test binary-utf-8-stream/read-line*
  (with-simple-utf-8-input-stream (stream "This is a
test string
with 合気道 and
日本語")
    (macrolet
        ((expect-read (size expected)
           (sdm:with-gensyms (line)
             `(let ((,line (sdm:read-line* stream ,size)))
                (is-true (string= ,line ,expected) "Expected ~s, not ~s" ,expected ,line))))

         (expect-pos (pos)
           (sdm:with-gensyms (val)
           `(let ((,val (file-position stream)))
              (is-true (= ,val ,pos) "Expected to be at position ~a, not ~a" ,pos ,val)))))

      (expect-read 69 "This is a")
      (expect-pos 10)
      (expect-read 69 "test string")
      (expect-pos 22)

      (expect-read 15 "with 合気道 ")
      (expect-pos 37)

      (expect-read 3 "and")
      (expect-pos 40)
      (expect-read 69 "")
      (expect-pos 41)

      (expect-read 8 "日本")
      (expect-pos 47)
      (expect-read 69 "語")
      (expect-pos 50)
      (expect-read 69 "")
      (expect-pos 50))))

(test binary-utf-8-stream/write-char
  (macrolet
      ((expect-char (ch)
         (sdm:with-gensyms (val expected)
           `(let ((,val (flex:get-output-stream-sequence (slot-value stream 'stream) :as-list t))
                  (,expected (sdm:uint->bytes (sdm:char->utf-8 ,ch))))
              (is-true (equalp ,val ,expected) "Expected the bytes ~a, not ~a" ,expected ,val)))))
    (let ((stream (make-instance 'simple-utf-8-output-stream)))
      (write-char #\a stream)
      (expect-char #\a))

    (let ((stream (make-instance 'simple-utf-8-output-stream)))
      (write-char #\道 stream)
      (expect-char #\道))))
