(in-package :cl-sdm-tests)

(in-suite cl-sdm-test-suite)

;;;
;;; Utilities
;;;

(defun make-arg-parser (&optional allow-dash-dash)
  (sdm-args:make-parser :program-name "test" :program-version "0.1.0" :allow-dash-dash allow-dash-dash))

(defun make-argv-list (&rest values)
  (append (list "test-program") values))

;;;
;;; Macros to define basic tests
;;;

(defmacro %define-basic-arg-creation-test (name arg-name arg-type value)
  (sdm:with-gensyms (parser arg)
    `(test ,name
       (let* ((,parser (make-arg-parser))
              (,arg (sdm-args:defargument ,parser ,arg-name :type ,arg-type)))
         (is-true (string= (sdm-args:arg-name ,arg) ,arg-name)
                  "Argument long name should be ~s, not ~w" ,arg-name (sdm-args:arg-name ,arg))
         (is-true (null (sdm-args:arg-short-name ,arg))
                  "Argument short name should be NIL, not ~w" (sdm-args:arg-short-name ,arg))

         (is-true (sdm:empty-string-p (sdm-args:arg-group ,arg)) "Argument group should be empty.")
         (is-true (sdm:empty-string-p (sdm-args:arg-help ,arg)) "Argument help should be empty.")

         (is-false (slot-value ,arg 'sdm-args::called?) "Argument should not be called")

         (is-true (eq (sdm-args:arg-type ,arg) ,arg-type)
                  "Argument type should be ~s, not ~s" ,arg-type (sdm-args:arg-type ,arg))
         (is-false (sdm-args:arg-value ,arg) "Argument value should be NIL")

         (setf (sdm-args:arg-value ,arg) ,value)
         (is-true (equalp (sdm-args:arg-value ,arg) ,value)
                  "Argument value did not get set correctly.")))))

(defmacro %define-arg-constraint-test (name arg-name arg-type value constraint constraint-needed
                                       &optional (bad-constraint nil bad-constraint-supplied-p))
  (sdm:with-gensyms (parser arg)
    `(test ,name
       (let* ((,parser (make-arg-parser))
              (,arg (sdm-args:defargument ,parser ,arg-name :type ,arg-type
                                                            :init-val ,value
                                                            :constraint ,constraint)))
         (is-true (equalp (sdm-args:arg-constraint ,arg) ,constraint-needed))

         ,(when bad-constraint-supplied-p
            `(signals sdm-args:argdef-error
               (sdm-args:defargument ,parser "alt-arg"
                                     :type ,arg-type
                                     :init-val ,value
                                     :constraint ,bad-constraint)))))))

(defmacro %define-arg-parse-test (name arg-name type value expected-value)
  (sdm:with-gensyms (parser call-name)
    `(test ,name
       (let* ((,parser (make-arg-parser))
              (,call-name (sdm:strings+ "--" ,arg-name)))
         (is-true (typep (sdm-args:defargument ,parser ,arg-name :type ,type) 'sdm-args:argument)
                  "DEFARGUMENT did not return an ARGUMENT")
         (finishes
           (sdm-args:parse-arguments ,parser (make-argv-list ,call-name ,value) :dont-quit t))
         (is-true (sdm-args:arg-called-p ,parser ,arg-name) "Argument was not flagged as being called.")
         (is-true (equalp (sdm-args:get-arg-value ,parser ,arg-name) ,expected-value)
                  "Argument value was ~s, not ~s."
                  (sdm-args:get-arg-value ,parser ,arg-name)
                  ,expected-value)))))

(defmacro %define-multi-arg-parse-test (name arg-name type expected-value &rest values)
  (sdm:with-gensyms (parser call-name)
    `(test ,name
       (let* ((,parser (make-arg-parser))
              (,call-name (sdm:strings+ "--" ,arg-name)))
         (is-true (typep (sdm-args:defargument ,parser ,arg-name :type ,type) 'sdm-args:argument)
                  "DEFARGUMENT did not return an ARGUMENT")
         (finishes
           (sdm-args:parse-arguments ,parser
                                     ,(loop for val in values
                                            appending (list call-name val) into argv
                                            finally (return (append '(make-argv-list) argv)))
                                     :dont-quit t))
         (is-true (sdm-args:arg-called-p ,parser ,arg-name) "Argument was not flagged as being called.")
         (is-true (equalp (sdm-args:get-arg-value ,parser ,arg-name) ,expected-value)
                  "Argument value was ~s, not ~s."
                  (sdm-args:get-arg-value ,parser ,arg-name)
                  ,expected-value)))))

(defmacro %define-arg-parse-test-with-constraint (name arg-name type good-value bad-value constraint)
  (sdm:with-gensyms (parser call-name)
    `(test ,name
       (let* ((,parser (make-arg-parser))
              (,call-name (sdm:strings+ "--" ,arg-name)))
         (sdm-args:defargument ,parser ,arg-name :type ,type :constraint ,constraint)
         (finishes
           (sdm-args:parse-arguments ,parser (make-argv-list ,call-name ,good-value) :dont-quit t)))

       (let* ((,parser (make-arg-parser))
              (,call-name (sdm:strings+ "--" ,arg-name)))
         (sdm-args:defargument ,parser ,arg-name :type ,type :constraint ,constraint)
         (signals (sdm-args:argument-error "Expected ~s to signals an ARGUMENT-ERROR" ,bad-value)
           (sdm-args:parse-arguments ,parser (make-argv-list ,call-name ,bad-value) :dont-quit t))))))

(defmacro %define-multi-arg-parse-test-with-constraint (name arg-name type good-values bad-values constraint)
  (sdm:with-gensyms (parser call-name)
    `(test ,name
       (let* ((,parser (make-arg-parser))
              (,call-name (sdm:strings+ "--" ,arg-name)))
         (sdm-args:defargument ,parser ,arg-name :type ,type :constraint ,constraint)
         (finishes
           (sdm-args:parse-arguments ,parser
                                     ,(loop for val in good-values
                                            appending (list call-name val) into argv
                                            finally (return (append '(make-argv-list) argv)))
                                     :dont-quit t)))

       (let* ((,parser (make-arg-parser))
              (,call-name (sdm:strings+ "--" ,arg-name)))
         (sdm-args:defargument ,parser ,arg-name :type ,type :constraint ,constraint)
         (signals sdm-args:argument-error
           (sdm-args:parse-arguments ,parser
                                     ,(loop for val in bad-values
                                            appending (list call-name val) into argv
                                            finally (return (append '(make-argv-list) argv)))
                                     :dont-quit t))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Basic argument creation tests
;;;

(%define-basic-arg-creation-test arg-parser/create-flag-argument "flag" :flag t)

(test arg-parser/create-flag-argument-with-short-name
  (let* ((parser (make-arg-parser))
         (arg-name "flag")
         (arg (sdm-args:defargument parser arg-name :short-name #\f :type :flag)))
    (is-true (string= (sdm-args:arg-name arg) arg-name)
             "Argument long name should be ~s, not ~s" arg-name (sdm-args:arg-name arg))
    (is-true (char= (sdm-args:arg-short-name arg) #\f)
             "Argument short name should be #\f, not ~s" (sdm-args:arg-short-name arg))
    (is-true (sdm:empty-string-p (sdm-args:arg-group arg)) "Argument group should be empty.")
    (is-true (sdm:empty-string-p (sdm-args:arg-help arg)) "Argument help should be empty.")
    (is-false (slot-value arg 'sdm-args::called?) "Argument should not be called")
    (is-false (sdm-args:arg-value arg) "Argument value should be NIL")))

(test arg-parser/create-flag-argument-with-group
  (let* ((parser (make-arg-parser))
         (arg-name "flag")
         (group-name "Test Group")
         (arg (sdm-args:defargument parser arg-name :type :flag :group group-name)))
    (is-true (string= (sdm-args:arg-name arg) arg-name)
             "Argument long name should be ~s, not ~s" arg-name (sdm-args:arg-name arg))
    (is-true (null (sdm-args:arg-short-name arg))
             "Argument short name should be NIL, not ~w" (sdm-args:arg-short-name arg))
    (is-true (string= (sdm-args:arg-group arg) group-name)
             "Argument group should be ~s, not ~s." group-name (sdm-args:arg-group arg))
    (is-true (sdm:empty-string-p (sdm-args:arg-help arg)) "Argument help should be empty.")
    (is-false (slot-value arg 'sdm-args::called?) "Argument should not be called")
    (is-false (sdm-args:arg-value arg) "Argument value should be NIL")))

(test arg-parser/create-flag-argument-with-help
  (let* ((parser (make-arg-parser))
         (arg-name "flag")
         (help-str "Help string")
         (arg (sdm-args:defargument parser arg-name :type :flag :help help-str)))
    (is-true (string= (sdm-args:arg-name arg) arg-name)
             "Argument long name should be ~s, not ~s" arg-name (sdm-args:arg-name arg))
    (is-true (null (sdm-args:arg-short-name arg))
             "Argument short name should be NIL, not ~w" (sdm-args:arg-short-name arg))
    (is-true (sdm:empty-string-p (sdm-args:arg-group arg)) "Argument group should be empty.")
    (is-true (string= (sdm-args:arg-help arg) help-str)
             "Argument help should be ~s, not ~s." help-str (sdm-args:arg-help arg))
    (is-false (slot-value arg 'sdm-args::called?) "Argument should not be called")
    (is-false (sdm-args:arg-value arg) "Argument value should be NIL")))

(%define-basic-arg-creation-test arg-parser/create-multi-flag-argument "mflag" :multi-flag 1)
(%define-basic-arg-creation-test arg-parser/create-string-argument "str" :string "lol")
(%define-basic-arg-creation-test arg-parser/create-multi-string-argument "mstr" :multi-string '("foo"))
(%define-basic-arg-creation-test arg-parser/create-integer-argument "inum" :integer 69)
(%define-basic-arg-creation-test arg-parser/create-float-argument "fnum" :float 69.42d0)
(%define-basic-arg-creation-test arg-parser/create-file-argument "filename" :file "test")
(%define-basic-arg-creation-test arg-parser/create-existing-file-argument "efilename" :existing-file "test")
(%define-basic-arg-creation-test arg-parser/create-multi-file-argument "mfilename" :multi-file '("test"))
(%define-basic-arg-creation-test arg-parser/create-multi-existing-file-argument "mefilename"
                                 :multi-existing-file '("test"))

(%define-arg-constraint-test arg-parser/make-flag-argument-with-constraint
    "flag" :flag nil nil nil)
(%define-arg-constraint-test arg-parser/make-multi-flag-argument-with-constraint
    "mflag" :multi-flag 1 nil nil)
(%define-arg-constraint-test arg-parser/make-string-argument-with-constraint
    "str" :string "lol" '("foo" "lol") '("foo" "lol") '("foo" "bar"))
(%define-arg-constraint-test arg-parser/make-multi-string-argument-with-constraint
    "mstr" :multi-string '("foo") '("foo" "bar") '("foo" "bar"))
(%define-arg-constraint-test arg-parser/make-integer-argument-with-constraint
    "inum" :integer 69 (cons 0 100) (cons 0 100) (cons 0 10))
(%define-arg-constraint-test arg-parser/make-float-argument-with-constraint
    "fnum" :float 69.42d0 (cons 0.0d0 100.0d0) (cons 0.0d0 100.0d0) (cons 0.0d0 1.0d0))
(%define-arg-constraint-test arg-parser/make-file-argument-with-constraint
    "filename" :file "test" nil nil)
(%define-arg-constraint-test arg-parser/make-existing-file-argument-with-constraint
    "efilename" :existing-file "test" nil nil)
(%define-arg-constraint-test arg-parser/make-multi-file-argument-with-constraint
    "mfilename" :multi-file '("test") nil nil)
(%define-arg-constraint-test arg-parser/make-multi-existing-file-argument-with-constraint
    "mefilename" :multi-existing-file '("test") nil nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Parsing tests
;;;

(test arg-parser/parses-flags
  (let* ((parser (make-arg-parser))
         (arg-name "flag")
         (call-name (sdm:strings+ "--" arg-name)))
    (is-true (typep (sdm-args:defargument parser arg-name :type :flag) 'sdm-args:argument)
             "DEFARGUMENT did not return an ARGUMENT")
    (finishes
      (sdm-args:parse-arguments parser (make-argv-list call-name) :dont-quit t))
    (is-true (sdm-args:arg-called-p parser arg-name) "Argument was not flagged as being called.")
    (is-true (sdm-args:get-arg-value parser arg-name) "Argument value was not true.")))

(test arg-parser/parses-multi-flags
  (let* ((parser (make-arg-parser))
         (arg-name "flag")
         (call-name (sdm:strings+ "--" arg-name)))
    (is-true (typep (sdm-args:defargument parser arg-name :type :multi-flag) 'sdm-args:argument)
             "DEFARGUMENT did not return an ARGUMENT")
    (finishes
      (sdm-args:parse-arguments parser (make-argv-list call-name call-name call-name) :dont-quit t))
    (is-true (sdm-args:arg-called-p parser arg-name) "Argument was not flagged as being called.")
    (is-true (equalp (sdm-args:get-arg-value parser arg-name) 3)
             "Argument was not flagged as being called three times.")))

(%define-arg-parse-test arg-parser/parses-strings "str" :string "lol" "lol")
(%define-multi-arg-parse-test arg-parser/parses-multi-strings "mstr" :multi-string '("foo" "bar") "foo" "bar")
;; These two are disabled on Clisp and ECL because they can't properly compare
;; two pathnames with EQUALP.
#-(or clisp ecl) (%define-arg-parse-test arg-parser/parses-files "filename" :file "lol" #P"lol")
#-(or clisp ecl) (%define-multi-arg-parse-test arg-parser/parses-multi-files
                                      "mfilename" :multi-file '(#P"foo" #P"bar") "foo" "bar")
(%define-arg-parse-test arg-parser/parses-integers "num" :integer "69" 69)

(test arg-parser/parses-floats
  (dolist (value (list (cons "69" 69.0d0)
                       (cons "42.0" 42.0d0)))
    (let* ((parser (make-arg-parser))
           (arg-name "num")
           (call-name (sdm:strings+ "--" arg-name)))
      (is-true (typep (sdm-args:defargument parser arg-name :type :float) 'sdm-args:argument)
               "DEFARGUMENT did not return an ARGUMENT")

      (finishes
        (sdm-args:parse-arguments parser (make-argv-list call-name (car value)) :dont-quit t))

      (is-true (sdm-args:arg-called-p parser arg-name) "Argument was not flagged as being called.")
      (is-true (= (sdm-args:get-arg-value parser arg-name) (cdr value))
               "Expected argument value to be ~s, not ~s"
               (cdr value) (sdm-args:get-arg-value parser arg-name)))))

(%define-arg-parse-test-with-constraint arg-parser/parses-integers-with-constraints
                                        "num"
                                        :integer
                                        "69"
                                        "200"
                                        '(0 . 100))

(%define-arg-parse-test-with-constraint arg-parser/parses-floats-with-constraints
                                        "num"
                                        :float
                                        "69.0"
                                        "200.0"
                                        '(0d0 . 100d0))

(%define-arg-parse-test-with-constraint arg-parser/parses-strings-with-constraints
                                        "num"
                                        :string
                                        "bar"
                                        "lol"
                                        '("foo" "bar" "baz"))

(%define-multi-arg-parse-test-with-constraint arg-parser/parses-multi-strings-with-constraints
                                              "num"
                                              :multi-string
                                              ("bar" "foo")
                                              ("lol" "yeah")
                                              '("foo" "bar" "baz"))

(test arg-parser/test-with-called-arg
  (let* ((parser (make-arg-parser))
         (arg-name "test")
         (call-name (sdm:strings+ "--" arg-name))
         (val "lol")
         (ok? nil))
    (sdm-args:defargument parser arg-name)
    (sdm-args:parse-arguments parser (make-argv-list call-name val) :dont-quit t)
    (sdm-args:with-called-arg (parser arg-name check-val)
      (is-true (string= check-val val))
      (setf ok? t))
    (is-true ok?)))

(test arg-parser/test-if-arg
  (let* ((parser (make-arg-parser))
         (arg-name-1 "test-1")
         (call-name-1 (sdm:strings+ "--" arg-name-1))
         (arg-name-2 "test-2")
         (val "lol")
         (ok? nil))
    (sdm-args:defargument parser arg-name-1)
    (sdm-args:defargument parser arg-name-2)
    (sdm-args:parse-arguments parser (make-argv-list call-name-1 val) :dont-quit t)
    (sdm-args:if-arg (parser arg-name-1 check-val)
      (is-true (string= check-val val))
      (error "~a should have been called" arg-name-1))

    (sdm-args:if-arg (parser arg-name-2 check-val)
      (error "~a should not have been called" arg-name-2)
      (setf ok? t))
    (is-true ok?)))

(test arg-parser/dash-dash
  (let* ((parser (make-arg-parser t))
         (flag-arg "test-flag")
         (call-arg (sdm:strings+ "--" flag-arg)))
    (sdm-args:defargument parser flag-arg :type :flag)

    (sdm-args:parse-arguments parser
                              (make-argv-list "pos1"
                                              call-arg
                                              "--"
                                              "pos2"
                                              call-arg
                                              "pos3"
                                              "--help")
                              :dont-quit t)
    (is-true (sdm-args:arg-called-p parser flag-arg))
    (is-true (= (length (sdm-args:parser-positional-args parser)) 5))
    (is-true (string= (elt (sdm-args:parser-positional-args parser) 0) "pos1"))
    (is-true (string= (elt (sdm-args:parser-positional-args parser) 1) "pos2"))
    (is-true (string= (elt (sdm-args:parser-positional-args parser) 2) call-arg))
    (is-true (string= (elt (sdm-args:parser-positional-args parser) 3) "pos3"))
    (is-true (string= (elt (sdm-args:parser-positional-args parser) 4) "--help")))

  (let* ((parser (make-arg-parser nil)))
    (signals sdm-args:argument-error
      (sdm-args:parse-arguments parser (make-argv-list "--") :dont-quit t))))

(test arg-parser/no-positional-args
  (macrolet ((with-parser (&body forms)
               `(let* ((parser (make-arg-parser))
                       (flag-arg "test-flag")
                       (call-arg (sdm:strings+ "--" flag-arg)))
                  (sdm-args:defargument parser flag-arg :type :flag)
                  ,@forms)))
    (with-parser
        (finishes
          (sdm-args:parse-arguments parser (make-argv-list "pos1" call-arg "pos2")
                                    :dont-quit t
                                    :no-positional-arguments nil)))

    (with-parser
        (signals sdm-args:argument-error
          (sdm-args:parse-arguments parser (make-argv-list "pos1" call-arg "pos2")
                                    :dont-quit t
                                    :no-positional-arguments t)))

    (with-parser
        (signals sdm-args:argument-error
          (sdm-args:parse-arguments parser (make-argv-list "pos1" call-arg "pos2" "--" "pos3")
                                    :dont-quit t
                                    :no-positional-arguments t)))))

(test arg-parser/reset-arguments
  (let* ((parser (make-arg-parser))
         (flag-arg "test-flag")
         (call-flag-arg (sdm:strings+ "--" flag-arg))
         (str-arg "test-str")
         (call-str-arg (sdm:strings+ "--" str-arg))
         (str-value "the string"))
    (sdm-args:defargument parser flag-arg :type :flag)
    (sdm-args:defargument parser str-arg :type :string)

    (sdm-args:parse-arguments parser (make-argv-list call-flag-arg "pos1" "pos2" call-str-arg str-value))
    (is-true (= (length (sdm-args:parser-positional-args parser)) 2))
    (is-true (sdm-args:arg-called-p parser flag-arg))
    (is-true (sdm-args:arg-called-p parser str-arg))
    (is-true (string= (sdm-args:arg-called-as parser str-arg) call-str-arg))
    (is-true (string= (sdm-args:get-arg-value parser str-arg) str-value))
    (is-true (sdm-args:parser-arguments-used-p parser))

    (sdm-args:parser-reset parser)

    (is-true (zerop (length (sdm-args:parser-positional-args parser))) "Positional arguments were not reset.")
    (is-false (sdm-args:arg-called-p parser flag-arg) "Flag argument was not reset.")
    (is-true  (null (sdm-args:get-arg-value parser str-arg)) "String argument was not reset.")
    (is-true  (null (sdm-args:arg-called-as parser str-arg)) "CALLED-AS? was not reset.")
    (is-false (sdm-args:parser-arguments-used-p parser) "ARGUMENTS-USED-P was not rest.")))

(test arg-parser/arg-called-as
  (let* ((parser (make-arg-parser))
         (flag-arg "test-flag")
         (flag-arg-short #\t)
         (call-arg (sdm:strings+ "--" flag-arg)))
    (sdm-args:defargument parser flag-arg :type :flag :short-name flag-arg-short)

    (sdm-args:parse-arguments parser (make-argv-list call-arg))
    (is-true (sdm-args:arg-called-p parser flag-arg))
    (is-true (equalp (sdm-args:arg-called-as parser flag-arg) call-arg)))

  (let* ((parser (make-arg-parser))
         (flag-arg "test-flag")
         (flag-arg-short #\t)
         (call-arg-short (sdm:strings+ "-" flag-arg-short)))
    (sdm-args:defargument parser flag-arg :type :flag :short-name flag-arg-short)

    (sdm-args:parse-arguments parser (make-argv-list call-arg-short))
    (is-true (sdm-args:arg-called-p parser flag-arg))
    (is-true (equalp (sdm-args:arg-called-as parser flag-arg) call-arg-short))))

(test arg-parser/test-single-dash-and-thats-it
  (let ((parser (make-arg-parser)))
    (sdm-args:parse-arguments parser (make-argv-list "-")
                              :dont-quit t :dont-check-help-or-ver t)
    (is-true (equalp (sdm-args:parser-positional-args parser) '("-"))))

  (let ((parser (make-arg-parser)))
    (signals (sdm-args:argument-error
              "Expected a single dash to signal an error when no positionals are accepted")
      (sdm-args:parse-arguments parser (make-argv-list "-")
                                :no-positional-arguments t
                                :dont-quit t :dont-check-help-or-ver t)))

  ;; Test "-" before arguments
  (let* ((parser (make-arg-parser))
         (flag-arg "test-flag")
         (flag-arg-short #\t)
         (call-arg-short (sdm:strings+ "-" flag-arg-short)))
    (sdm-args:defargument parser flag-arg :type :flag :short-name flag-arg-short)

    (sdm-args:parse-arguments parser (make-argv-list "-" call-arg-short)
                              :dont-quit t :dont-check-help-or-ver t)
    (is-true (sdm-args:arg-called-p parser flag-arg))
    (is-true (equalp (sdm-args:arg-called-as parser flag-arg) call-arg-short))
    (is-true (equalp (sdm-args:parser-positional-args parser) '("-"))))

  ;; Test "-" at end
  (let* ((parser (make-arg-parser))
         (flag-arg "test-flag")
         (flag-arg-short #\t)
         (call-arg-short (sdm:strings+ "-" flag-arg-short)))
    (sdm-args:defargument parser flag-arg :type :flag :short-name flag-arg-short)

    (sdm-args:parse-arguments parser (make-argv-list call-arg-short "-")
                              :dont-quit t :dont-check-help-or-ver t)
    (is-true (sdm-args:arg-called-p parser flag-arg))
    (is-true (equalp (sdm-args:arg-called-as parser flag-arg) call-arg-short))
    (is-true (equalp (sdm-args:parser-positional-args parser) '("-")))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Error checks
;;;

(test arg-parser/require-long-name
  (let ((parser (make-arg-parser)))
    (signals sdm-args:argdef-error
      (sdm-args:defargument parser ""))))

(test arg-parser/validate-long-name
  (let ((parser (make-arg-parser))
        (arg-name "-foo"))
    (signals sdm-args:argdef-error
      (sdm-args:defargument parser arg-name))))

(test arg-parser/validate-short-name
  (let ((parser (make-arg-parser))
        (arg-name "test"))
    (signals sdm-args:argdef-error
      (sdm-args:defargument parser arg-name :short-name #\Nul))
    (signals sdm-args:argdef-error
      (sdm-args:defargument parser arg-name :short-name #\Bel))
    (signals sdm-args:argdef-error
      (sdm-args:defargument parser arg-name :short-name #\-))
    (signals sdm-args:argdef-error
      (sdm-args:defargument parser arg-name :short-name #\0))

    (is-true (eq (sdm-args:arg-short-name (sdm-args:defargument parser arg-name :short-name #\語))
                 #\語)
             "Arguments should accept non-ASCII short names.")))

(test arg-parser/long-name-cannot-be-eq-to-short-name
  (let ((parser (make-arg-parser))
        (arg-name "f")
        (short-name #\f))
    (signals sdm-args:argdef-error
      (sdm-args:defargument parser arg-name :short-name short-name))))

(test arg-parser/does-not-allow-duplicate-long-names
  (let ((parser (make-arg-parser))
        (arg-name "test"))
    (finishes
      (sdm-args:defargument parser arg-name))
    (signals sdm-args:argdef-error
      (sdm-args:defargument parser arg-name))))

(test arg-parser/does-not-allow-duplicate-short-names
  (let ((parser (make-arg-parser))
        (arg-name-1 "test1")
        (arg-name-2 "test2")
        (short-name #\t))
    (finishes
      (sdm-args:defargument parser arg-name-1 :short-name short-name))
    (signals sdm-args:argdef-error
      (sdm-args:defargument parser arg-name-2 :short-name short-name))))

(test arg-parser/validate-constraints
  (let ((parser (make-arg-parser))
        (arg-name "test"))
    (signals (sdm-args:argdef-error "Should not allow a number to be a constraint for a :STRING")
      (sdm-args:defargument parser arg-name :type :string :constraint 69))
    (signals (sdm-args:argdef-error "Should not allow a symbol to be a constraint for a :STRING")
      (sdm-args:defargument parser arg-name :type :string :constraint 'foo))
    (signals (sdm-args:argdef-error "Should not allow a string to be a constraint for a :STRING")
      (sdm-args:defargument parser arg-name :type :string :constraint "foo"))

    (signals (sdm-args:argdef-error "Should not allow a number to be a constraint for a :MULTI-STRING")
      (sdm-args:defargument parser arg-name :type :multi-string :constraint 69))
    (signals (sdm-args:argdef-error "Should not allow a symbol to be a constraint for a :MULTI-STRING")
      (sdm-args:defargument parser arg-name :type :multi-string :constraint 'foo))
    (signals (sdm-args:argdef-error "Should not allow a string to be a constraint for a :MULTI-STRING")
      (sdm-args:defargument parser arg-name :type :multi-string :constraint "foo"))

    (signals (sdm-args:argdef-error "Should not allow a number to be a constraint for a :INTEGER")
      (sdm-args:defargument parser arg-name :type :integer :constraint 69))
    (signals (sdm-args:argdef-error "Should not allow a symbol to be a constraint for a :INTEGER")
      (sdm-args:defargument parser arg-name :type :integer :constraint 'foo))
    (signals (sdm-args:argdef-error "Should not allow a string to be a constraint for a :INTEGER")
      (sdm-args:defargument parser arg-name :type :integer :constraint "foo"))
    (signals (sdm-args:argdef-error "Should not allow a list to be a constraint for a :INTEGER")
      (sdm-args:defargument parser arg-name :type :integer :constraint '(1 2)))
    (signals (sdm-args:argdef-error "Should not allow a CONS of floats to be a constraint for a :INTEGER")
      (sdm-args:defargument parser arg-name :type :integer :constraint '(1d0 . 2.0)))
    (signals (sdm-args:argdef-error "Should not allow a CONS of symbols to be a constraint for a :INTEGER")
      (sdm-args:defargument parser arg-name :type :integer :constraint '(1 . foo)))
    (signals (sdm-args:argdef-error "Should not allow a CONS of strings to be a constraint for a :INTEGER")
      (sdm-args:defargument parser arg-name :type :integer :constraint '(1 . "foo")))

    (signals (sdm-args:argdef-error "Should not allow a number to be a constraint for a :FLOAT")
      (sdm-args:defargument parser arg-name :type :float :constraint 69))
    (signals (sdm-args:argdef-error "Should not allow a symbol to be a constraint for a :FLOAT")
      (sdm-args:defargument parser arg-name :type :float :constraint 'foo))
    (signals (sdm-args:argdef-error "Should not allow a string to be a constraint for a :FLOAT")
      (sdm-args:defargument parser arg-name :type :float :constraint "foo"))
    (signals (sdm-args:argdef-error "Should not allow a list to be a constraint for a :FLOAT")
      (sdm-args:defargument parser arg-name :type :float :constraint '(1.0 2)))
    (signals (sdm-args:argdef-error "Should not allow a CONS of symbols to be a constraint for a :FLOAT")
      (sdm-args:defargument parser arg-name :type :float :constraint '(1l0 . foo)))
    (signals (sdm-args:argdef-error "Should not allow a CONS of strings to be a constraint for a :FLOAT")
      (sdm-args:defargument parser arg-name :type :float :constraint '(1d0 . "foo")))))

(test arg-parser/cant-use-allow-dash-dash-with-no-positionals
  (let ((parser (make-arg-parser t)))
    (signals error
      (sdm-args:parse-arguments parser (make-arg-parser)
                                :no-positional-arguments t))))
