;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;; Copyright (c) 2010-2013 iMatix Corporation and Contributors
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; z85 Binary to Text
;;;;
;;;; Based on the reference implementation: https://rfc.zeromq.org/spec/32/
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-simple-error z85-error nil ())

(defining-consts
  (+byte->z85+
   #(#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9
     #\a #\b #\c #\d #\e #\f #\g #\h #\i #\j
     #\k #\l #\m #\n #\o #\p #\q #\r #\s #\t
     #\u #\v #\w #\x #\y #\z #\A #\B #\C #\D
     #\E #\F #\G #\H #\I #\J #\K #\L #\M #\N
     #\O #\P #\Q #\R #\S #\T #\U #\V #\W #\X
     #\Y #\Z #\. #\- #\: #\+ #\= #\^ #\! #\/
     #\* #\? #\& #\< #\> #\( #\) #\[ #\] #\{
     #\} #\@ #\% #\$ #\#))

  (+z85->byte+
   #(#x00 #x44 #x00 #x54 #x53 #x52 #x48 #x00
     #x4B #x4C #x46 #x41 #x00 #x3F #x3E #x45
     #x00 #x01 #x02 #x03 #x04 #x05 #x06 #x07
     #x08 #x09 #x40 #x00 #x49 #x42 #x4A #x47
     #x51 #x24 #x25 #x26 #x27 #x28 #x29 #x2A
     #x2B #x2C #x2D #x2E #x2F #x30 #x31 #x32
     #x33 #x34 #x35 #x36 #x37 #x38 #x39 #x3A
     #x3B #x3C #x3D #x4D #x00 #x4E #x43 #x00
     #x00 #x0A #x0B #x0C #x0D #x0E #x0F #x10
     #x11 #x12 #x13 #x14 #x15 #x16 #x17 #x18
     #x19 #x1A #x1B #x1C #x1D #x1E #x1F #x20
     #x21 #x22 #x23 #x4F #x00 #x50 #x00 #x00)))

(defgeneric z85-encode (thing)
  (:documentation "Encodes THING using z85 binary-to-text encoding.  Returns a new string
containing the encoded data."))

(defmethod z85-encode ((thing vector))
  "Encodes a vector of bytes using z85 binary-to-text encoding.  Returns a new
string containing the encoded data.

THING must be a T/UINT8-VECTOR or a subtype.  The length of THING must be a
multiple of 4."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (assert (typep thing 't/uint8-vector))
  (unless (zerop (mod (length thing) 4))
    (z85-error () "The length of the data (~:d) is not a multiple of 4." (length thing)))

  (with-output-to-string (out)
    (loop with acc of-type t/uint32 = 0
          for idx of-type t/ufixnum from 0 below (length thing) by 4
          for val-1 of-type t/uint8 = (muffling (elt thing idx))
          for val-2 of-type t/uint8 = (muffling (elt thing (1+ idx)))
          for val-3 of-type t/uint8 = (muffling (elt thing (+ idx 2)))
          for val-4 of-type t/uint8 = (muffling (elt thing (+ idx 3)))
          do (setf acc (+ (* val-1 256) val-2))
             (setf acc (+ (* acc 256) val-3))
             (setf acc (+ (* acc 256) val-4))
             (write-char (elt +byte->z85+ (mod (truncate acc #.(expt 85 4)) 85)) out)
             (write-char (elt +byte->z85+ (mod (truncate acc #.(expt 85 3)) 85)) out)
             (write-char (elt +byte->z85+ (mod (truncate acc #.(expt 85 2)) 85)) out)
             (write-char (elt +byte->z85+ (mod (truncate acc 85) 85)) out)
             (write-char (elt +byte->z85+ (mod acc 85)) out))))

(defmethod z85-encode ((thing list))
  "Encodes a LIST of bytes using z85 binary-to-text encoding.  Returns a new string
containing the encoded data.

THING must be a LIST that contains only T/UINT8 values.  The length of THING
must be a multiple of 4."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (unless (zerop (mod (length thing) 4))
    (z85-error () "The length of the data (~:d) is not a multiple of 4." (length thing)))

  (with-output-to-string (out)
    (loop with acc of-type t/uint32 = 0
          for idx of-type t/ufixnum from 0 below (length thing) by 4
          for val-1 of-type t/uint8 = (elt thing idx)
          for val-2 of-type t/uint8 = (elt thing (1+ idx))
          for val-3 of-type t/uint8 = (elt thing (+ idx 2))
          for val-4 of-type t/uint8 = (elt thing (+ idx 3))
          do (setf acc (+ (* val-1 256) val-2))
             (setf acc (+ (* acc 256) val-3))
             (setf acc (+ (* acc 256) val-4))
             (write-char (elt +byte->z85+ (mod (truncate acc #.(expt 85 4)) 85)) out)
             (write-char (elt +byte->z85+ (mod (truncate acc #.(expt 85 3)) 85)) out)
             (write-char (elt +byte->z85+ (mod (truncate acc #.(expt 85 2)) 85)) out)
             (write-char (elt +byte->z85+ (mod (truncate acc 85) 85)) out)
             (write-char (elt +byte->z85+ (mod acc 85)) out))))

(defun z85-decode (string)
  "Decodes a z85 encoded string to a T/UINT8-ARRAY.  The string must be a
SIMPLE-STRING, and its length must be a multiple of 5."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (assert (typep string 'simple-string))

  (let* ((slen (length string))
         (ret (new-array (truncate (* slen 4) 5) t/uint8))
         (acc 0))
    (declare (type t/ufixnum slen)
             (type t/uint32 acc))
    (unless (zerop (mod slen 5))
      (z85-error () "Length of z85 string is not a multiple of 5."))

    (labels
        ((decode-char (char)
           (declare (type character char)
                    (optimize speed (debug 0) (safety 1) (compilation-speed 0) (space 0)))
           (aref +z85->byte+ (- (char-code char) 32))))
      (declare (inline decode-char))

      (loop for sidx of-type t/ufixnum from 0 below slen by 5
            for didx of-type t/ufixnum from 0 by 4
            do (setf acc (decode-char (elt string sidx)))
               (setf acc (+ (* acc 85) (decode-char (elt string (1+ sidx)))))
               (setf acc (+ (* acc 85) (decode-char (elt string (+ sidx 2)))))
               (setf acc (+ (* acc 85) (decode-char (elt string (+ sidx 3)))))
               (setf acc (+ (* acc 85) (decode-char (elt string (+ sidx 4)))))
               (setf (aref ret didx) (mod (truncate acc #.(expt 256 3)) 256))
               (setf (aref ret (1+ didx)) (mod (truncate acc #.(expt 256 2)) 256))
               (setf (aref ret (+ didx 2)) (mod (truncate acc 256) 256))
               (setf (aref ret (+ didx 3)) (mod acc 256))))
    ret))
