;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/uds)

;;;;
;;;; Grey Streams Interface
;;;;
;;;; All socket streams support both input and output, and always use the
;;;; element type (UNSIGNED-BYTE 8) internally.
;;;;

;;;
;;; CL package methods
;;;

(defmethod cl:open-stream-p ((stream client-socket))
  (declare (optimize speed (debug 1)))
  (socket-open-p stream))

#-clisp
(defmethod cl:input-stream-p ((stream client-socket))
  t)

#-clisp
(defmethod cl:output-stream-p ((stream client-socket))
  t)

(defmethod cl:stream-element-type ((stream client-socket))
  '(unsigned-byte 8))

(defmethod cl:close ((stream client-socket) &key abort)
  (declare (optimize speed (debug 1)))
  (declare (ignore abort))
  (socket-close stream))

;;;
;;; TRIVIAL-GRAY-STREAMS package methods
;;;

(defmethod trivial-gray-streams:stream-listen ((stream client-socket))
  (declare (optimize speed (debug 1))
           (ignore stream))
  ;; We don't support this, so just return NIL.
  nil)

(defmethod trivial-gray-streams:stream-file-position ((stream client-socket))
  (declare (optimize speed (debug 1))
           (ignore stream))
  ;; We don't support this, so just return NIL.
  nil)

(defmethod trivial-gray-streams:stream-write-string ((stream client-socket) (string string) &optional start end)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (check-type start (or null (integer 0 *)))
  (check-type end (or null (integer 0 *)))
  (trivial-gray-streams:stream-write-sequence stream (string->bytes string :as-vector t) start end))

(defmethod (setf trivial-gray-streams:stream-file-position) (new-offset (stream client-socket))
  (declare (optimize speed (debug 1))
           (ignore new-offset stream))
  ;; We don't support this, so just return NIL.
  nil)

(defmethod trivial-gray-streams:stream-read-byte ((stream client-socket))
  (declare (optimize speed (debug 1)))
  (ensuring-open-socket (stream)
    (read-byte (slot-value stream 'strm))))

(defmethod trivial-gray-streams:stream-read-sequence ((stream client-socket) sequence start end &key)
  (declare (optimize speed (debug 1)))
  (ensuring-open-socket (stream)
    (read-sequence sequence (slot-value stream 'strm) :start start :end end)))

(defmethod trivial-gray-streams:stream-write-byte ((stream client-socket) integer)
  (declare (optimize speed (debug 1)))
  (ensuring-open-socket (stream)
    (write-byte integer (slot-value stream 'strm))))

(defmethod trivial-gray-streams:stream-write-sequence ((stream client-socket) sequence start end &key)
  (declare (optimize speed (debug 1)))
  (ensuring-open-socket (stream)
    (write-sequence sequence (slot-value stream 'strm) :start start :end end)))

(defmethod trivial-gray-streams:stream-finish-output ((stream client-socket))
  (declare (optimize speed (debug 1)))
  (ensuring-open-socket (stream)
    (finish-output (slot-value stream 'strm))))

(defmethod trivial-gray-streams:stream-clear-output ((stream client-socket))
  (declare (optimize speed (debug 1)))
  (finish-output stream))

(defmethod trivial-gray-streams:stream-force-output ((stream client-socket))
  (declare (optimize speed (debug 1)))
  (force-output (slot-value stream 'strm)))

(defmethod trivial-gray-streams:stream-read-char ((stream client-socket))
  (declare (optimize speed (debug 1)))
  (let ((byte (trivial-gray-streams:stream-read-byte stream)))
    (if (eq byte :eof)
        :eof
        (code-char byte))))

(defmethod trivial-gray-streams:stream-read-line ((stream client-socket))
  (declare (optimize speed (debug 1)))
  (values
   (with-output-to-string (out)
     (loop for char = (trivial-gray-streams:stream-read-char stream) do
       (case char
         (:eof
          (return-from trivial-gray-streams:stream-read-line
            (values (get-output-stream-string out) t)))
         (otherwise
          (if (not (char= char #\Newline))
              (write-char char out)
              (loop-finish))))))
   nil))

(defmethod trivial-gray-streams:stream-write-char ((stream client-socket) (character character))
  (declare (optimize speed (debug 1)))
  (let ((code (char-code character)))
    (if (< code 256)
        (write-byte code stream)
        (write-sequence (uint->byte-vector code) stream))))
