;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/uds)

(defun make-server-socket (filename &key (type :stream) (backlog 128) (keepalive t) (read-timeout 0))
  "Creates a new SERVER-SOCKET instance that will be bound to FILENAME and will
begin listening for connections.  To accept a connection, use SOCKET-ACCEPT."
  (check-type filename (or pathname string))
  (check-type backlog fixnum)
  (check-type type t/socket-type)
  (check-type read-timeout t/read-timeout)

  (let* ((impl (%make-server-socket type keepalive read-timeout filename backlog))
         (ret (%wrap-socket 'server-socket impl type keepalive read-timeout (sdm-file:native-pathname filename))))
    ;; Bind the socket, then start listening, then return
    (%bind-socket impl (etypecase filename
                         (pathname (uiop:native-namestring filename))
                         (string (uiop:native-namestring (uiop:parse-native-namestring filename)))))
    (%socket-listen impl backlog)
    (setf (slot-value ret 'open?) t)
    ret))

(defun make-client-socket (filename &key (type :stream) (keepalive t) (read-timeout 0))
  "Creates a new CLIENT-SOCKET instance that will connect to FILENAME to enable
bidirectional communication."
  (check-type filename (or pathname string))
  (check-type type t/socket-type)
  (check-type read-timeout t/read-timeout)

  (let* ((impl (%make-client-socket type keepalive read-timeout filename))
         (ret (%wrap-socket 'client-socket impl type keepalive read-timeout (sdm-file:native-pathname filename))))
    ;; Connect the socket, setup any implementation-specific stuff, then return
    (%socket-connect impl (etypecase filename
                            (pathname (uiop:native-namestring filename))
                            (string (uiop:native-namestring
                                     (uiop:parse-native-namestring filename)))))
    (%setup-client-socket ret type keepalive read-timeout filename)
    (setf (slot-value ret 'open?) t)
    ret))

(defmethod socket-close ((socket server-socket))
  (when (next-method-p)
    (call-next-method))
  (with-slots (filename)
      socket
    (when (probe-file filename)
      (delete-file filename))))

(defmacro with-server ((socket path &key (type :stream) (backlog 128)) &body forms)
  "Creates a new SERVER-SOCKET bound to SERVER that will listen on the file
given by PATH, then executes FORMS.  This will automatically call SOCKET-CLOSE
before returning."
  `(let ((,socket (make-server-socket ,path :type ,type :backlog ,backlog)))
     (unwind-protect
          (progn ,@forms)
       (socket-close ,socket))))

(defmacro with-client ((socket path &key (type :stream) dont-finish-output) &body forms)
  "Creates a new CLIENT-SOCKET that is connected to the file given by PATH, then
executes FORMS.  This will automatically call SOCKET-CLOSE before returning.

If DONT-FINISH-OUTPUT is NIL, then FINISH-OUTPUT is also automatically called on
the socket before this returns.  Otherwise, the socket is closed, but
FINISH-OUTPUT is not called."
  `(let ((,socket (make-client-socket ,path :type ,type)))
     (unwind-protect
          (progn ,@forms)
       ,(unless dont-finish-output
          `(finish-output ,socket))
       (socket-close ,socket))))

(defmacro with-accept ((client server &optional dont-finish-output) &body forms)
  "Accepts a new connection by calling SOCKET-ACCEPT on SERVER, binding a new
CLIENT-SOCKET to CLIENT, executes FORMS.  This will automatically call
SOCKET-CLOSE before returning.

If DONT-FINISH-OUTPUT is NIL, then FINISH-OUTPUT is also automatically called on
the socket before this returns.  Otherwise, the socket is closed, but
FINISH-OUTPUT is not called."
  `(let ((,client (socket-accept ,server)))
     (unwind-protect
          (progn ,@forms)
       ,(unless dont-finish-output
          `(finish-output ,client))
       (socket-close ,client))))

(defmacro ensuring-open-socket ((socket) &body forms)
  `(if (socket-open-p ,socket)
       (progn ,@forms)
       (error "Socket is closed")))
