;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/uds)

;;;;
;;;; SBCL Backend
;;;;

(defun %bind-socket (impl filename)
  (check-type filename string)
  (sb-bsd-sockets:socket-bind impl filename))

(defun %wrap-socket (class impl type keepalive read-timeout &optional filename)
  (check-type type t/socket-type)
  (check-type read-timeout (or null t/read-timeout))
  (check-type filename (or null pathname))
  (unless (subtypep class 'socket)
    (error "Class is not a subtype of CL-SDM-UDS:SOCKET - ~a" class))

  (let ((ret (make-instance class)))
    (setf (slot-value ret 'impl) impl)
    (setf (slot-value ret 'type) type)
    (setf (slot-value ret 'keepalive?) (if keepalive t nil))
    (setf (slot-value ret 'read-timeout) (or read-timeout 0))
    (when filename
      (setf (slot-value ret 'filename) filename))

    ;; Setup any options
    (setf (sb-bsd-sockets:sockopt-keep-alive impl) (if keepalive t nil))

    ret))

(defun %socket-listen (impl backlog)
  (sb-bsd-sockets:socket-listen impl backlog))

(defun %make-server-socket (type keepalive read-timeout filename backlog)
  (declare (ignore keepalive read-timeout filename backlog))
  (make-instance 'sb-bsd-sockets:local-socket :type type))

(defun %make-client-socket (type keepalive read-timeout filename)
  (declare (ignore keepalive read-timeout filename))
  (make-instance 'sb-bsd-sockets:local-socket :type type))

(defun %setup-client-socket (socket type keepalive read-timeout filename)
  (declare (ignore type keepalive read-timeout filename))
  (with-slots (impl read-timeout)
      socket
    (setf (slot-value socket 'strm)
          (sb-bsd-sockets:socket-make-stream impl :timeout (if (zerop read-timeout) nil read-timeout)
                                                  :element-type '(unsigned-byte 8)
                                                  :input t
                                                  :output t))))

(defun %socket-connect (socket filename)
  (check-type filename string)
  (sb-bsd-sockets:socket-connect socket filename))

(defmethod socket-close ((socket socket))
  (with-slots (impl open?)
      socket
    (prog1 (when (sb-bsd-sockets:socket-open-p impl)
             (sb-bsd-sockets:socket-close impl))
      (setf open? nil))))

(defmethod socket-accept ((socket server-socket))
  (with-slots (impl type read-timeout keepalive?)
      socket
    (let ((impl (sb-bsd-sockets:socket-accept impl))
          (ret (%wrap-socket 'client-socket impl type keepalive? read-timeout)))
      ;; On SBCL, we use an underlying socket stream
      (setf (slot-value ret 'strm)
            (sb-bsd-sockets:socket-make-stream impl :timeout (if (zerop read-timeout) nil read-timeout)
                                                    :element-type '(unsigned-byte 8)
                                                    :input t
                                                    :output t))
      (setf (slot-value ret 'open?) t)
      ret)))
