;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/uds)

;;;;
;;;; CLISP Backend
;;;;

(require "syscalls")
(require "rawsock")

(defun %test-read (filename)
  (let ((sock (rawsock:socket :unix :stream 0)))
    (rawsock:bind sock (rawsock:make-sockaddr :unix (string->bytes filename)))
    (setf (rawsock:socket-option sock :keepalive :level :sol-socket) t)
    (rawsock:sock-listen sock 20)

    (let ((client (rawsock:accept sock nil)))
      (let ((stream (ext:make-stream client :direction :io :element-type '(unsigned-byte 8)))
            (buf (new-array 256 t/uint8)))
        (rawsock:sock-close client)
        (let ((pos (read-sequence buf stream)))
          (bytes->string (subseq buf 0 pos)))))))

(defun %bind-socket (impl filename)
  (check-type filename string)
  (rawsock:bind impl (rawsock:make-sockaddr :unix (string->bytes filename))))

(defun %wrap-socket (class impl type keepalive read-timeout &optional filename)
  (check-type type t/socket-type)
  (check-type read-timeout (or null t/read-timeout))
  (check-type filename (or null pathname))

  (unless (subtypep class 'socket)
    (error "Class is not a subtype of CL-SDM-UDS:SOCKET - ~a" class))

  (let ((ret (make-instance class)))
    (setf (slot-value ret 'impl) impl)
    (setf (slot-value ret 'type) type)
    (setf (slot-value ret 'keepalive?) (if keepalive t nil))
    (setf (slot-value ret 'read-timeout) (or read-timeout 0))
    (when filename
      (setf (slot-value ret 'filename) filename))

    ;; Setup any options
    (setf (rawsock:socket-option impl :keepalive :level :sol-socket) (if keepalive t nil))

    ret))

(defun %socket-listen (impl backlog)
  (rawsock:sock-listen impl backlog))

(defun %make-server-socket (type keepalive read-timeout filename backlog)
  (declare (ignore keepalive read-timeout filename backlog))
  (check-type type t/socket-type)
  (rawsock:socket :unix (if (eq type :datagram) :dgram type) 0))

(defun %make-client-socket (type keepalive read-timeout filename)
  (declare (ignore keepalive read-timeout filename))
  (check-type type t/socket-type)
  (rawsock:socket :unix (if (eq type :datagram) :dgram type) 0))

(defun %setup-client-socket (socket type keepalive read-timeout filename)
  (declare (ignore type keepalive read-timeout filename))
  (with-slots (impl read-timeout)
      socket
    (prog1 (setf (slot-value socket 'strm)
                 (ext:make-stream impl :direction :io
                                       :element-type '(unsigned-byte 8)))
      (rawsock:sock-close impl))))

(defun %socket-connect (socket filename)
  (check-type filename string)
  (rawsock:connect socket (rawsock:make-sockaddr :unix (string->bytes filename))))

(defmethod socket-close ((socket socket))
  (with-slots (impl open?)
      socket
    (when open?
      (rawsock:sock-close impl)
      (setf open? nil))))

(defmethod socket-accept ((socket server-socket))
  (with-slots (impl type read-timeout keepalive?)
      socket
    (let ((impl (rawsock:accept impl nil))
          (ret (%wrap-socket 'client-socket impl type keepalive? read-timeout)))
      ;; On CLISP, we use EXT:MAKE-STREAM, then close the original socket FD
      ;; because MAKE-STREAM will dup() the FD.  Without the RAWSOCK:SOCK-CLOSE,
      ;; then the runtime could get confused with all the stream buffering.
      (setf (slot-value ret 'strm)
            (ext:make-stream impl :direction :io
                                  :element-type '(unsigned-byte 8)))
      (rawsock:sock-close impl)
      (setf (slot-value ret 'open?) t)
      ret)))
