;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Convenience Types
;;;;
;;;; These are basically proxy types of commonly used things to save on some
;;;; typing while working with heavily-typed code.
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;
;;; Basic integer types
;;;

(deftype t/ufixnum ()
  "A type of (AND UNSIGNED-BYTE FIXNUM)."
  '(and unsigned-byte fixnum))

(deftype t/uint8 ()
  "An unsigned 8-bit integer."
  '(unsigned-byte 8))

(deftype t/uint16 ()
  "An unsigned 16-bit integer."
  '(unsigned-byte 16))

(deftype t/uint24 ()
  "An unsigned 24-bit integer."
  '(unsigned-byte 24))

(deftype t/uint32 ()
  "An unsigned 32-bit integer."
  '(unsigned-byte 32))

(deftype t/uint64 ()
  "An unsigned 64-bit integer."
  '(unsigned-byte 64))

(deftype t/uint128 ()
  "An unsigned 128-bit integer."
  '(unsigned-byte 128))

(deftype t/int8 ()
  "A signed 8-bit integer."
  '(signed-byte 8))

(deftype t/int16 ()
  "A signed 16-bit integer."
  '(signed-byte 16))

(deftype t/int24 ()
  "A signed 24-bit integer."
  '(signed-byte 24))

(deftype t/int32 ()
  "A signed 32-bit integer."
  '(signed-byte 32))

(deftype t/int64 ()
  "A signed 64-bit integer."
  '(signed-byte 64))

(deftype t/int128 ()
  "A signed 128-bit integer."
  '(signed-byte 128))

;;;
;;; Floating point vectors and arrays
;;;

(deftype t/shfloat-vector (&optional (size '*))
  "A vector of SHORT-FLOAT of the given size."
  `(vector short-float ,size))

(deftype t/shfloat-array (&optional size)
  "A SIMPLE-ARRAY of SHORT-FLOAT of the given size."
  (if size
      `(simple-array short-float (,size))
      `(simple-array short-float)))

(deftype t/sfloat-vector (&optional (size '*))
  "A vector of SINGL-FLOAT of the given size."
  `(vector single-float ,size))

(deftype t/sfloat-array (&optional size)
  "A SIMPLE-ARRAY of SINGLE-FLOAT of the given size."
  (if size
      `(simple-array single-float (,size))
      `(simple-array single-float)))

(deftype t/dfloat-vector (&optional (size '*))
  "A vector of DOUBLE-FLOAT of the given size."
  `(vector double-float ,size))

(deftype t/dfloat-array (&optional size)
  "A SIMPLE-ARRAY of DOUBLE-FLOAT of the given size."
  (if size
      `(simple-array double-float (,size))
      `(simple-array double-float)))

(deftype t/lfloat-vector (&optional (size '*))
  "A vector of LONG-FLOAT of the given size."
  `(vector long-float ,size))

(deftype t/lfloat-array (&optional size)
  "A SIMPLE-ARRAY of LONG-FLOAT of the given size."
  (if size
      `(simple-array long-float (,size))
      `(simple-array long-float)))

;;;
;;; Integer vectors and arrays
;;;

(deftype t/uint8-vector (&optional (size '*))
  "A vector of unsigned 8-bit integers."
  `(vector t/uint8 ,size))

(deftype t/uint16-vector (&optional (size '*))
  "A vector of unsigned 16-bit integers."
  `(vector t/uint16 ,size))

(deftype t/uint24-vector (&optional (size '*))
  "A vector of unsigned 24-bit integers."
  `(vector t/uint24 ,size))

(deftype t/uint32-vector (&optional (size '*))
  "A vector of unsigned 32-bit integers."
  `(vector t/uint32 ,size))

(deftype t/uint64-vector (&optional (size '*))
  "A vector of unsigned 64-bit integers."
  `(vector t/uint64 ,size))

(deftype t/int8-vector (&optional (size '*))
  "A vector of signed 8-bit integers."
  `(vector t/int8 ,size))

(deftype t/int16-vector (&optional (size '*))
  "A vector of signed 16-bit integers."
  `(vector t/int16 ,size))

(deftype t/int24-vector (&optional (size '*))
  "A vector of signed 24-bit integers."
  `(vector t/int24 ,size))

(deftype t/int32-vector (&optional (size '*))
  "A vector of signed 32-bit integers."
  `(vector t/int32 ,size))

(deftype t/int64-vector (&optional (size '*))
  "A vector of signed 64-bit integers."
  `(vector t/int64 ,size))

(deftype t/uint8-array (&optional (size '*))
  "A SIMPLE-ARRAY of unsigned 8-bit integers."
  (if size
      `(simple-array t/uint8 (,size))
      `(simple-array t/uint8)))

(deftype t/uint16-array (&optional (size '*))
  "A SIMPLE-ARRAY of unsigned 16-bit integers."
  (if size
      `(simple-array t/uint16 (,size))
      `(simple-array t/uint16)))

(deftype t/uint24-array (&optional (size '*))
  "A SIMPLE-ARRAY of unsigned 24-bit integers."
  (if size
      `(simple-array t/uint24 (,size))
      `(simple-array t/uint24)))

(deftype t/uint32-array (&optional (size '*))
  "A SIMPLE-ARRAY of unsigned 32-bit integers."
  (if size
      `(simple-array t/uint32 (,size))
      `(simple-array t/uint32)))

(deftype t/uint64-array (&optional (size '*))
  "A SIMPLE-ARRAY of unsigned 64-bit integers."
  (if size
      `(simple-array t/uint64 (,size))
      `(simple-array t/uint64)))

(deftype t/int8-array (&optional (size '*))
  "A SIMPLE-ARRAY of signed 8-bit integers."
  (if size
      `(simple-array t/int8 (,size))
      `(simple-array t/int8)))

(deftype t/int16-array (&optional (size '*))
  "A SIMPLE-ARRAY of signed 16-bit integers."
  (if size
      `(simple-array t/int16 (,size))
      `(simple-array t/int16)))

(deftype t/int24-array (&optional (size '*))
  "A SIMPLE-ARRAY of signed 24-bit integers."
  (if size
      `(simple-array t/int24 (,size))
      `(simple-array t/int24)))

(deftype t/int32-array (&optional (size '*))
  "A SIMPLE-ARRAY of signed 32-bit integers."
  (if size
      `(simple-array t/int32 (,size))
      `(simple-array t/int32)))

(deftype t/int64-array (&optional (size '*))
  "A SIMPLE-ARRAY of signed 64-bit integers."
  (if size
      `(simple-array t/int64 (,size))
      `(simple-array t/int64)))
