;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/time)

(defgeneric ts+ (span amount))
(defgeneric ts- (span amount))

(defstruct (timespan (:conc-name %ts-)
                     (:constructor %make-timespan))
  (seconds 0 :type fixnum)
  (minutes 0 :type fixnum)
  (hours   0 :type fixnum)
  (days    0 :type fixnum))

;;;
;;; Constructors
;;;

(defun make-timespan (&key (days 0) (hours 0) (minutes 0) (seconds 0))
  (declare (type fixnum days hours minutes seconds))
  (assert (typep seconds '(integer 0 59)) (seconds) "Seconds must be between 0 and 59, inclusive.")
  (assert (typep minutes '(integer 0 59)) (minutes) "Minutes must be between 0 and 59, inclusive.")
  (assert (typep hours '(integer 0 23)) (hours) "Hours must be between 0 and 23, inclusive.")
  (%make-timespan :seconds seconds :minutes minutes :hours hours :days days))

(define-typed-fn make-timespan-from-total-seconds ((fixnum value))
    (timespan t)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))

  (let* ((secs      (mod value 60))
         (tot-mins  (floor value 60))
         (mins      (mod tot-mins 60))
         (tot-hours (floor tot-mins 60))
         (hours     (mod tot-hours 24))
         (days      (floor tot-hours 24)))
    (%make-timespan :days days :hours hours :minutes mins :seconds secs)))

;;;
;;; Accessors
;;;

(define-typed-fn ts-seconds ((timespan ts))
    (fixnum t)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (%ts-seconds ts))

(define-typed-fn ts-minutes ((timespan ts))
    (fixnum t)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (%ts-minutes ts))

(define-typed-fn ts-hours ((timespan ts))
    (fixnum t)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (%ts-hours ts))

(define-typed-fn ts-days ((timespan ts))
    (fixnum t)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (%ts-days ts))

(define-typed-fn (setf ts-seconds) ((fixnum new-value) (timespan ts))
    (fixnum t)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (unless (and (>= new-value 0)
               (<= new-value 59))
    (error "Seconds must be between 0 and 59, inclusive."))
  (setf (%ts-seconds ts) new-value))

(define-typed-fn (setf ts-minutes) ((fixnum new-value) (timespan ts))
    (fixnum t)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (unless (and (>= new-value 0)
               (<= new-value 59))
    (error "Minutes must be between 0 and 59, inclusive."))
  (setf (%ts-minutes ts) new-value))

(define-typed-fn (setf ts-hours) ((fixnum new-value) (timespan ts))
    (fixnum t)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (unless (and (>= new-value 0)
               (<= new-value 23))
    (error "Hours must be between 0 and 23, inclusive."))
  (setf (%ts-hours ts) new-value))

(define-typed-fn (setf ts-days) ((fixnum new-value) (timespan ts))
    (fixnum t)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (setf (%ts-days ts) new-value))

;;;
;;; Utility Functions
;;;

(define-typed-fn ts-total-seconds ((timespan span))
    (fixnum t)
  "Returns the total number of seconds represented by SPAN."
  (declare (type timespan span)
           (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (+ (%ts-seconds span)
     (the fixnum (* (%ts-minutes span) 60))
     (the fixnum (* (%ts-hours   span) (* 60 60)))
     (the fixnum (* (%ts-days    span) (* 60 60 24)))))

(define-typed-fn ts-total-minutes ((timespan span))
    (fixnum t)
  "Returns the total number of whole minutes represented by SPAN.  The number of
seconds in SPAN is ignored."
  (declare (type timespan span)
           (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (+ (%ts-minutes span)
     (the fixnum (* (%ts-hours   span) (* 60)))
     (the fixnum (* (%ts-days    span) (* 60 24)))))

(define-typed-fn ts-total-hours ((timespan span))
    (fixnum t)
  "Returns the total number of whole hours represented by SPAN.  The number of
seconds and minutes in SPAN is ignored."
  (declare (type timespan span)
           (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (+ (%ts-hours span)
     (the fixnum (* (%ts-days  span) (* 60 24)))))

;;;
;;; Comparison and Addition/Subtraction Functions
;;;

(defmethod ts+ ((span1 timespan) (span2 timespan))
  "Adds the span of time in SPAN2 to SPAN1, returning a new TIMESPAN instance."
  (let ((span1-secs (ts-total-seconds span1))
        (span2-secs (ts-total-seconds span2)))
    (make-timespan-from-total-seconds (+ span1-secs span2-secs))))

(defmethod ts- ((span1 timespan) (span2 timespan))
  "Subtracts the span of time in SPAN2 from SPAN1, returning a new TIMESPAN
instance."
  (let ((span1-secs (ts-total-seconds span1))
        (span2-secs (ts-total-seconds span2)))
    (make-timespan-from-total-seconds (- span1-secs span2-secs))))

(defmethod ts+ ((span timespan) (amount integer))
  "Adds AMOUNT seconds to SPAN, returning a new TIMESPAN instance."
  (setf span (make-timespan-from-total-seconds (+ (ts-total-seconds span) amount))))

(defmethod ts- ((span timespan) (amount integer))
  "Subtracts AMOUNT seconds to SPAN, returning a new TIMESPAN instance."
  (setf span (make-timespan-from-total-seconds (- (ts-total-seconds span) amount))))

(define-typed-fn ts+/minutes ((timespan span) (fixnum value))
    (timespan t)
  "Adds VALUE minutes to SPAN, returning a new TIMESPAN instance."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (make-timespan-from-total-seconds (+ (ts-total-seconds span) (the fixnum (* value 60)))))

(define-typed-fn ts+/hours ((timespan span) (fixnum value))
    (timespan t)
  "Adds VALUE hours to SPAN, returning a new TIMESPAN instance."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (make-timespan-from-total-seconds (+ (ts-total-seconds span) (the fixnum (* value 60 60)))))

(define-typed-fn ts+/days ((timespan span) (fixnum value))
    (timespan t)
  "Adds VALUE days to SPAN, returning a new TIMESPAN instance."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (make-timespan-from-total-seconds (+ (ts-total-seconds span) (the fixnum (* value 24 60 60)))))

(define-typed-fn ts-/minutes ((timespan span) (fixnum value))
    (timespan t)
  "Subtracts VALUE minutes from SPAN, returning a new TIMESPAN instance."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (make-timespan-from-total-seconds (- (ts-total-seconds span) (the fixnum (* value 60)))))

(define-typed-fn ts-/hours ((timespan span) (fixnum value))
    (timespan t)
  "Subtracts VALUE hours from SPAN, returning a new TIMESPAN instance."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (make-timespan-from-total-seconds (- (ts-total-seconds span) (the fixnum (* value 60 60)))))

(define-typed-fn ts-/days ((timespan span) (fixnum value))
    (timespan t)
  "Subtracts VALUE days from SPAN, returning a new TIMESPAN instance."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (make-timespan-from-total-seconds (- (ts-total-seconds span) (the fixnum (* value 24 60 60)))))

(define-typed-fn ts= ((timespan ts1 ts2))
    (boolean t)
  "Returns T if TS1 and TS2 both represent the same span of time, or NIL
otherwise."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (= (ts-total-seconds ts1)
     (ts-total-seconds ts2)))

(define-typed-fn ts< ((timespan ts1 ts2))
    (boolean t)
  "Returns T if TS1 represents a strictly smaller amount of time than TS2, or
NIL otherwise."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (< (ts-total-seconds ts1)
     (ts-total-seconds ts2)))

(define-typed-fn ts<= ((timespan ts1 ts2))
    (boolean t)
  "Returns T if TS1 represents a smaller or equal amount of time than TS2, or
NIL otherwise."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (< (ts-total-seconds ts1)
     (ts-total-seconds ts2)))

(define-typed-fn ts> ((timespan ts1 ts2))
    (boolean t)
  "Returns T if TS1 represents a strictly larger amount of time than TS2, or NIL
otherwise."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (> (ts-total-seconds ts1)
     (ts-total-seconds ts2)))

(define-typed-fn ts>= ((timespan ts1 ts2))
    (boolean t)
  "Returns T if TS1 represents a larger or equal amount of time than TS2, or NIL
otherwise."
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (> (ts-total-seconds ts1)
     (ts-total-seconds ts2)))
