;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/time)

(defun dtformat/dow-name-full (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~a" (dt-day-of-week dt :name)))

(defun dtformat/dow-name (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~a" (dt-day-of-week dt :short-name)))

(defun dtformat/month-name (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~a" (month-num->name (dt-month dt))))

(defun dtformat/month-name-abbrev (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~a" (month-num->name (dt-month dt) t)))

;; (%Y-%m-%d %H:%M:%S)
(defun dtformat/full-date-time (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~4,'0d-~2,'0d-~2,'0d ~2,'0d:~2,'0d:~2,'0d"
          (dt-year dt)
          (dt-month dt)
          (dt-day dt)
          (dt-hour dt)
          (dt-minute dt)
          (dt-second dt)))

(defun dtformat/day-of-month (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~2,'0d" (dt-day dt)))

(defun dtformat/day-of-month-short (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~d" (dt-day dt)))

(defun dtformat/hour (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~2,'0d" (dt-hour dt)))

(defun dtformat/hour-12hr (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~2,'0d" (mod (dt-hour dt) 12)))

(defun dtformat/hour-short (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~d" (dt-hour dt)))

(defun dtformat/hour-12hr-short (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~d" (mod (dt-hour dt) 12)))

(defun dtformat/month-short (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~d" (dt-month dt)))

(defun dtformat/minute (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~2,'0d" (dt-minute dt)))

(defun dtformat/month (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~2,'0d" (dt-month dt)))

(defun dtformat/minute-short (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~d" (dt-minute dt)))

(defun dtformat/am-pm (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~a" (if (>= (dt-hour dt) 12) "pm" "am")))

(defun dtformat/am-pm-big (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~a" (if (>= (dt-hour dt) 12) "PM" "AM")))

(defun dtformat/dow-number (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~d" (dt-day-of-week dt)))

(defun dtformat/year (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~4,'0d" (dt-year dt)))

(defun dtformat/second (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~2,'0d" (dt-second dt)))

(defun dtformat/second-short (dt out)
  (declare (type datetime dt)
           (optimize (debug 0) (space 0) (compilation-speed 0)))
  (format out "~d" (dt-second dt)))

(defun parse-time-format (fmt)
  (declare (type simple-string fmt)
           (optimize (speed 3) (space 0) (debug 0) (compilation-speed 0)))

  (let ((ret ())
        (str (make-string-output-stream))
        (state :top))
    (declare (type (member :top :fmt) state)
             (type list ret)
             (type string-stream str))

    (macrolet ((store-str (the-str the-list)
                 (let ((val-sym (gensym)))
                   `(let ((,val-sym (get-output-stream-string ,the-str)))
                      (when (and ,val-sym (not (string= ,val-sym "")))
                        (push ,val-sym ,the-list))

                      (setf ,the-str (make-string-output-stream))))))

      (loop for c character across fmt do
        (ecase state
          (:top
           (if (char= c #\%)
               (setf state :fmt)
               (write-char c str)))

          (:fmt
           (cond
             ;; Literal %
             ((char= c #\%)
              (write-char c str))

             ;; Arbitrary whitespace
             ((char= c #\W)
              (write-char #\Space str))

             ;; Full name of the day of the week
             ((char= c #\a)
              (store-str str ret)
              (push #'dtformat/dow-name-full ret))

             ;; Abbreviated name of the day of the week
             ((char= c #\A)
              (store-str str ret)
              (push #'dtformat/dow-name ret))

             ;; Month name, full
             ((char= c #\b)
              (store-str str ret)
              (push #'dtformat/month-name ret))

             ;; Month name, abbreviated
             ((char= c #\B)
              (store-str str ret)
              (push #'dtformat/month-name-abbrev ret))

             ;; Date and time (%Y-%m-%d %H:%M:%S)
             ((char= c #\c)
              (store-str str ret)
              (push #'dtformat/full-date-time ret))

             ;; Day of month (01-31, depending on month)
             ((char= c #\d)
              (store-str str ret)
              (push #'dtformat/day-of-month ret))

             ;; Day of month (1-31, depending on month)
             ((char= c #\D)
              (store-str str ret)
              (push #'dtformat/day-of-month-short ret))

             ;; Hour (00-23)
             ((char= c #\H)
              (store-str str ret)
              (push #'dtformat/hour ret))

             ;; Hour, 12-hour clock (01-12)
             ((char= c #\h)
              (store-str str ret)
              (push #'dtformat/hour-12hr ret))

             ;; Hour (0-23)
             ((char= c #\I)
              (store-str str ret)
              (push #'dtformat/hour-short ret))

             ;; Hour, 12-hour clock (1-12)
             ((char= c #\i)
              (store-str str ret)
              (push #'dtformat/hour-12hr-short ret))

             ;; Month number 1-12
             ((char= c #\m)
              (store-str str ret)
              (push #'dtformat/month-short ret))

             ;; Minute (00-59)
             ((char= c #\M)
              (store-str str ret)
              (push #'dtformat/minute ret))

             ;; Month number 01-12
             ((char= c #\n)
              (store-str str ret)
              (push #'dtformat/month ret))

             ;; Minute (0-59)
             ((char= c #\N)
              (store-str str ret)
              (push #'dtformat/minute-short ret))

             ;; pm or pm
             ((char= c #\p)
              (store-str str ret)
              (push #'dtformat/am-pm ret))

             ;; AM or PM
             ((char= c #\P)
              (store-str str ret)
              (push #'dtformat/am-pm-big ret))

             ;; Second (00-59)
             ((char= c #\S)
              (store-str str ret)
              (push #'dtformat/second ret))

             ;; Second (0-59)
             ((char= c #\s)
              (store-str str ret)
              (push #'dtformat/second-short ret))

             ;; Day of week as a number (1-7)
             ((char= c #\w)
              (store-str str ret)
              (push #'dtformat/dow-number ret))

             ;; Full year
             ((char= c #\Y)
              (store-str str ret)
              (push #'dtformat/year ret))

             (t
              (warn "Unknown datetime format character ~c, skipping" c)))
           (setf state :top))))
      (store-str str ret)) ;; Store any trailing characters
    (nreverse ret)))

(defun dt->string (dt &optional fmt)
  (declare (type datetime dt)
           (type (or null string) fmt))
  (let ((instr (parse-time-format (if fmt fmt "%d %B %Y, %H:%M:%S"))))
    (declare (type list instr))

    (with-output-to-string (out)
      (dolist (inst instr)
        (if (stringp inst)
            (write-string inst out)
            (funcall inst dt out))))))

(defun ts->string (ts)
  (declare (type timespan ts)
           (optimize (speed 3) (debug 1) (safety 0)))
  (with-output-to-string (out)
    (when (> (the fixnum (ts-days ts)) 0)
      (format out "~d." (ts-days ts)))

    (let ((hours (abs (the fixnum (ts-hours ts)))))
      (format out "~:[~;0~]~d:" (< hours 10) hours))

    (let ((mins (abs (the fixnum (ts-minutes ts)))))
      (format out "~:[~;0~]~d:" (< mins 10) mins))

    (let ((secs (abs (the fixnum (ts-seconds ts)))))
      (format out "~:[~;0~]~d" (< secs 10) secs))))
