;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/time)

(define-condition datetime-parse-error (simple-error)
  ())

(define-condition bad-day-of-week-error (datetime-parse-error)
  ())

(defmacro datetime-parse-error (msg &rest fmt-args)
  `(error 'datetime-parse-error
          :format-control ,msg
          :format-arguments (list ,@fmt-args)))

(deftype t/date-format ()
  '(member :iso-8601 :rfc-822))

(defgeneric parse-time (format source)
  (:documentation "Parses a date/time in SOURCE using the given format.  FORMAT
can be any format specified by the type T/DATE-FORMAT."))

(define-typed-fn parse-number ((string-stream input) ((integer 1 #.most-positive-fixnum) len)
                               &optional as-string junk-ok min-len)
    ((or simple-string fixnum) t)
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (let ((ret (with-output-to-string (out)
               (loop for i fixnum from 1 to len
                     for num fixnum from 0
                     for c character = (peek-char nil input nil #\Nul)
                     do (cond
                          ((char= c #\Nul)
                           (loop-finish))

                          ((digit-char-p c 10)
                           (read-char input)
                           (write-char c out))

                          (t (if junk-ok
                                 (progn
                                   (read-char input)
                                   (write-char c out))
                                 (if (>= num (or min-len (1- most-positive-fixnum)))
                                     (loop-finish)
                                     (datetime-parse-error
                                      "Junk in number while attempting to read ~d digits" len)))))))))

    (if as-string ret (parse-integer ret))))

(defmacro define-string-parse-fn (name (err-msg &key capitalize? (offset 1)) &rest possibilities)
  (let ((ret-cases (if capitalize?
                       (mapcar #'string-capitalize possibilities)
                       possibilities))
        (max-size (loop for str in possibilities maximizing (length str))))
    (unless (and (every #'stringp possibilities))
      (error "POSSIBILITIES must only contain strings"))

    (unless (loop for str in possibilities
                  collecting (length str) into ret
                  finally (return (= (length (remove-duplicates ret :test #'=)) 1)))
      (error "The strings in POSSIBILITIES must all be the same length"))

    (macrolet ((get-chars (strings pos)
                 `(list 'quote (remove-duplicates
                                (map 'list #'(lambda (x)
                                               (char-downcase (elt x ,pos)))
                                     ,strings)
                                :test #'char=))))

      `(defun ,name (input &key return-numeric)
         (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
                  #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

         (loop with cur-str = (make-string-output-stream)
               with len fixnum = 0
               for c character = (peek-char nil input nil #\Nul)
               do (if (char= c #\Nul)
                      (loop-finish)

                      (case len
                        ,@(loop for i from 0 below max-size
                                collect `(,i (if (find (char-downcase c)
                                                       ,(get-chars possibilities i)
                                                       :test #'char=)
                                                 (write-char c cur-str)
                                                 (datetime-parse-error ,err-msg))))

                        (otherwise (loop-finish))))
                  (read-char input)
                  (incf len)

               finally (return
                         (let ((ret (get-output-stream-string cur-str)))
                           (string-case ,(if capitalize?
                                             `(string-capitalize ret)
                                             `ret)
                             ,@(loop for i from 0
                                     for str in ret-cases
                                     collect (list str (list 'if 'return-numeric (+ offset i) 'ret)))
                             (otherwise (datetime-parse-error ,err-msg))))))))))

(define-string-parse-fn parse-day-of-week-abbrev ("Invalid abbreviated day-of-week" :capitalize? t :offset 1)
                        "Mon" "Tue" "Wed" "Thu" "Fri" "Sat" "Sun")

(define-string-parse-fn parse-month-abbrev ("Invalid abbreviated month" :capitalize? t)
                        "Jan" "Feb" "Mar" "Apr" "May" "Jun" "Jul" "Aug" "Sep" "Oct" "Nov" "Dec")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; ISO 8601 Format
;;;

(define-typed-fn iso-8601/set-ordinal ((datetime dt) ((integer 1 366) day-num))
    (datetime t)
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (setf (dt-month dt) 1)
  (setf (dt-day dt) 1)
  (nth-value 0 (dt+ dt (make-timespan-from-total-seconds (* 60 60 24 day-num)))))

(define-typed-fn iso-8601/parse-start ((datetime dt) (string-stream input))
    (keyword t)
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  ;; Check the first character
  (let ((c (peek-char t input nil #\Nul)))
    (cond
      ((char= c #\Nul)
       (datetime-parse-error "Empty time string"))

      ((digit-char-p c 10)
       (setf (dt-year dt) (parse-number input 4))
       :after-year)

      (t (datetime-parse-error "Cannot parse time string: junk in year")))))

(define-typed-fn iso-8601/parse-after-year (dt (string-stream input))
    (keyword t)
  (declare (type datetime dt)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (let ((c (peek-char t input nil #\Nul)))
    (cond
      ;; Finished reading the entire date string
      ((char= c #\Nul)
       :done)

      ;; Week number or week number and a weekday day
      ((char= c #\W)
       (datetime-parse-error "Dates with week numbers are not yet supported"))

      ;; Will either be four digits (two month, two day), or three digits (ordinal date).
      ;; Note that YYYYMM is not allowed.
      ((digit-char-p c 10)
       (let ((str (parse-number input 4 t)))
         (cond
           ;; If all we got were three characters, it means we have an ordinal date and that's it
           ((= (length str) 3)
            (setf dt (iso-8601/set-ordinal dt (parse-integer (subseq str 0 3))))
            :done)

           ;; If the last character is a T, then we have an ordinal date
           ((char= (elt str 3) #\T)
            (unread-char #\T input)
            (setf dt (iso-8601/set-ordinal dt (parse-integer (subseq str 0 3))))
            :time)

           ;; If the last character is a digit, then we should have a month and day
           ((digit-char-p (elt str 3) 10)
            (setf (dt-month dt) (parse-integer (subseq str 0 2)))
            (setf (dt-day dt) (parse-integer (subseq str 2)))
            :time)

           (t (datetime-parse-error "Invalid time string: expected MMDD or DDD after year")))))

      ;; Could be a month, week number, or ordinal date
      ((char= c #\-)
       (read-char input)

       (setf c (peek-char nil input nil #\Nul))
       (cond
         ((char= c #\Nul)
          (datetime-parse-error "Invalid time string: expected month, week number part, or ordinal date"))

         ;; Week number or week number and a weekday day
         ((char= c #\W)
          (datetime-parse-error "Dates with week numbers are not yet supported"))

         ((digit-char-p c 10)
          (let ((str (parse-number input 3 t t)))
            (cond
              ;; If we only have two more characters, then it's a month
              ((= (length str) 2)
               (setf (dt-month dt) (parse-integer str))
               :done)

              ;; If the last character is a hyphen, then we have MM-DD format.
              ((char= (elt str 2) #\-)
               (setf (dt-month dt) (parse-integer (subseq str 0 2)))
               :day)

              ;; If the last character is a digit, then we have an ordinal DDD format
              ((digit-char-p (elt str 2) 10)
               (setf dt (iso-8601/set-ordinal dt (parse-integer str)))
               :time)

              (t (error "what")))))

         (t (datetime-parse-error "Invalid time string: expected MM-DD or -DDD after year"))))

      (t (datetime-parse-error "Invalid time string: expected month, week number, or ordinal date after year")))))

(define-typed-fn iso-8601/parse-time-start ((string-stream input))
    (keyword t)
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (case (read-char input nil #\Nul)
    (#\Nul :done)
    (#\T :hour)
    (otherwise (datetime-parse-error "Expected start of time part, but no T was found"))))

(define-typed-fn iso-8601/parse-day (dt (string-stream input))
    (keyword t)
  (declare (type datetime dt)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (unless (digit-char-p (peek-char nil input))
    (datetime-parse-error "Invalid time string: day is invalid"))

  (setf (dt-day dt) (parse-number input 2))
  :time)

(define-typed-fn iso-8601/parse-hour (dt (string-stream input))
    (keyword t)
  (declare (type datetime dt)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (unless (digit-char-p (peek-char nil input))
    (datetime-parse-error "Invalid time string: hour is invalid"))

  (setf (dt-hour dt) (parse-number input 2))

  (let ((c (peek-char nil input nil #\Nul)))
    (cond
      ((char= c #\Nul)
       :done)
      ((char= c #\:)
       (read-char input)
       :minute)

      ((digit-char-p c 10)
       :minute)

      (t (datetime-parse-error "Invalid time string: junk after hour")))))

(define-typed-fn iso-8601/parse-minute (dt (string-stream input))
    (keyword t)
  (declare (type datetime dt)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (unless (digit-char-p (peek-char nil input))
    (datetime-parse-error "Invalid time string: minute is invalid"))

  (setf (dt-minute dt) (parse-number input 2))

  (let ((c (peek-char nil input nil #\Nul)))
    (cond
      ((char= c #\Nul)
       :done)
      ((char= c #\:)
       (read-char input)
       :second)

      ((digit-char-p c 10)
       :second)

      (t (datetime-parse-error "Invalid time string: junk after minute")))))

(define-typed-fn iso-8601/parse-second (dt (string-stream input))
    (keyword t)
  (declare (type datetime dt)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (unless (digit-char-p (peek-char nil input))
    (datetime-parse-error "Invalid time string: second is invalid"))

  (setf (dt-second dt) (parse-number input 2))

  (let ((c (peek-char nil input nil #\Nul)))
    (cond
      ((char= c #\Nul)
       :done)

      ((char= c #\Z)
       :done)

      ((or (char= c #\+)
           (char= c #\-))
       (read-char input)
       :timezone)

      ((char= c #\.)
       ;;(warn "Fractional seconds unsupported")
       :done)
      (t (datetime-parse-error "Invalid time string: junk after seconds")))))

(defmethod parse-time ((format (eql :iso-8601)) (source string))
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (let ((state :start)
        (ret (%make-datetime)))
    (with-input-from-string (input source)
      (loop do
        (ecase state
          (:start (setf state (iso-8601/parse-start ret input)))
          (:after-year (setf state (iso-8601/parse-after-year ret input)))
          (:done (return-from parse-time ret))
          (:day (setf state (iso-8601/parse-day ret input)))
          (:time (setf state (iso-8601/parse-time-start input)))
          (:hour (setf state (iso-8601/parse-hour ret input)))
          (:minute (setf state (iso-8601/parse-minute ret input)))
          (:second (setf state (iso-8601/parse-second ret input)))
          (:timezone
           ;(warn "Timezones unsupported")
           (setf state :done)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; RFC 822 Format
;;;

(define-typed-fn %rfc-822/parse-start ((string-stream input))
    (keyword t)
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  ;; Check the first character
  (let ((c (peek-char t input nil #\Nul)))
    (cond
      ((char= c #\Nul)
       (datetime-parse-error "Empty time string"))

      ((find (locally
                 (declare #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
               (char-downcase c))
             '(#\m #\t #\w #\f #\s) :test #'char=)
       :abbrev-day-of-week)

      ((digit-char-p c 10)
       :day)

      (t (datetime-parse-error "Cannot parse time string: junk in year")))))

(define-typed-fn %rfc-822/parse-day (dt (string-stream input))
    (keyword t)
  (declare (type datetime dt)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  ;; Check the first character
  (let ((c (peek-char t input nil #\Nul)))
    (cond
      ((char= c #\Nul)
       (datetime-parse-error "Missing day value"))

      ((digit-char-p c 10)
       (setf (dt-day dt) (parse-number input 2 nil nil 1))
       :abbrev-month)

      (t (datetime-parse-error "Cannot parse time string: junk in day")))))

(define-typed-fn %rfc-822/parse-abbrev-month (dt (string-stream input))
    (keyword t)
  (declare (type datetime dt)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  ;; Check the first character
  (let ((c (peek-char t input nil #\Nul)))
    (cond
      ((char= c #\Nul)
       (datetime-parse-error "Missing day value"))

      ((alpha-char-p c)
       (setf (dt-month dt) (parse-month-abbrev input :return-numeric t))
       :year)

      (t (datetime-parse-error "Cannot parse date string: junk in abbreviated month: '~a'" c)))))

(define-typed-fn %rfc-822/parse-abbrev-day-of-week ((string-stream input))
    ((values fixnum keyword) t)
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  ;; Check the first character
  (let ((c (peek-char t input nil #\Nul)))
    (cond
      ((char= c #\Nul)
       (datetime-parse-error "Missing expected day-of-week value"))

      ((alpha-char-p c)
       (let ((ret (parse-day-of-week-abbrev input :return-numeric t)))
         (unless (and (char= (read-char input nil #\Nul) #\,)
                      (char= (read-char input nil #\Nul) #\Space))
           (datetime-parse-error "Abbreviated day-of-week is not followed by a comma and space"))
         (values ret :day)))

      (t (datetime-parse-error "Cannot parse date string: junk in abbreviated day-of-week")))))

(define-typed-fn %rfc-822/parse-year (dt (string-stream input))
    (keyword t)
  (declare (type datetime dt)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  ;; check the first character
  (let ((c (peek-char t input nil #\nul)))
    (cond
      ((char= c #\nul)
       (datetime-parse-error "missing day value"))

      ((digit-char-p c 10)
       (setf (dt-year dt) (parse-number input 4 nil nil 2))

       (unless (char= (peek-char nil input nil #\Nul) #\Space)
         (datetime-parse-error "No space after year"))

       (read-char input)
       :hour)

      (t (datetime-parse-error "cannot parse date string: junk in abbreviated month")))))

(define-typed-fn %rfc-822/parse-hour (dt (string-stream input))
    (keyword t)
  (declare (type datetime dt)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (unless (digit-char-p (peek-char nil input))
    (datetime-parse-error "Invalid time string: hour is invalid"))

  (setf (dt-hour dt) (parse-number input 2))

  (let ((c (peek-char nil input nil #\Nul)))
    (cond
      ((char= c #\Nul)
       :done)
      ((char= c #\:)
       (read-char input)
       :minute)

      ((digit-char-p c 10)
       :minute)

      (t (datetime-parse-error "Invalid time string: junk after hour")))))

(define-typed-fn %rfc-822/parse-minute (dt (string-stream input))
    (keyword t)
  (declare (type datetime dt)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (unless (digit-char-p (peek-char nil input))
    (datetime-parse-error "Invalid time string: minute is invalid"))

  (setf (dt-minute dt) (parse-number input 2))

  (let ((c (peek-char nil input nil #\Nul)))
    (cond
      ((char= c #\Nul)
       :done)

      ((char= c #\:)
       (read-char input)
       :second)

      ((digit-char-p c 10)
       :second)

      (t (datetime-parse-error "Invalid time string: junk after minute")))))

(define-typed-fn %rfc-822/parse-second (dt (string-stream input))
    (keyword t)
  (declare (type datetime dt)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (unless (digit-char-p (peek-char nil input))
    (datetime-parse-error "Invalid time string: second is invalid"))

  (setf (dt-second dt) (parse-number input 2))

  (let ((c (peek-char nil input nil #\Nul)))
    (case c
      (#\Nul
       :done)

      (#\Space
       (read-char input)
       (if (find (peek-char nil input nil #\Nul) '(#\+ #\-) :test #'char=)
            :timezone
            :done))

      (otherwise
       :done))))

(define-typed-fn %rfc-822/parse-timezone ((datetime dt) (string-stream input))
    ((values fixnum datetime keyword) t)
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (let* ((sign (ecase (read-char input)
                 (#\+ 1)
                 (#\- -1)))
         (hours (* sign (parse-number input 2)))
         (mins (* sign (parse-number input 2)))
         (ret (dt+hours dt hours)))
    (declare (type fixnum sign hours mins))
    (setf ret (dt+minutes ret mins))
    (values (dt-day-of-week dt nil) ret :done)))

(defmethod parse-time ((format (eql :rfc-822)) (source string))
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (let ((state :start)
        (ret (%make-datetime))
        (parsed-dow nil)
        (dow nil))
    (declare (type (or null fixnum) dow parsed-dow))

    (with-input-from-string (input source)
      (loop do
        (ecase state
          (:start (setf state (%rfc-822/parse-start input)))
          (:done (loop-finish))
          (:day (setf state (%rfc-822/parse-day ret input)))
          (:abbrev-day-of-week
           (multiple-value-bind (new-val new-state)
               (%rfc-822/parse-abbrev-day-of-week input)
             (setf dow new-val)
             (setf state new-state)))
          (:abbrev-month (setf state (%rfc-822/parse-abbrev-month ret input)))
          (:year (setf state (%rfc-822/parse-year ret input)))
          (:hour (setf state (%rfc-822/parse-hour ret input)))
          (:minute (setf state (%rfc-822/parse-minute ret input)))
          (:second (setf state (%rfc-822/parse-second ret input)))
          (:timezone
           (multiple-value-bind (old-dow new-dt new-state)
               (%rfc-822/parse-timezone ret input)
             (setf parsed-dow old-dow)
             (setf ret new-dt)
             (setf state new-state))))))

    (when dow
      (check-type dow (integer 1 7) "Invalid parsed day of the week")

      (unless (eq dow (or parsed-dow (dt-day-of-week ret nil)))
        (restart-case
            (error 'bad-day-of-week-error
                   :format-control "Date claims the day of the week is ~a, but it was calculated to be ~a"
                   :format-arguments (list (day-num->name dow)
                                           (day-num->name (or parsed-dow (dt-day-of-week ret nil)))))
          (use-actual-value ()
            :report "Use the correct Day-of-Week"))))

    ret))
