;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/time)

(defgeneric dt-second (dt))
(defgeneric dt-minute (dt))
(defgeneric dt-hour (dt))
(defgeneric dt-day (dt))
(defgeneric dt-month (dt))
(defgeneric dt-year (dt))

(defgeneric (setf dt-second) (new-value dt))
(defgeneric (setf dt-minute) (new-value dt))
(defgeneric (setf dt-hour) (new-value dt))
(defgeneric (setf dt-day) (new-value dt))
(defgeneric (setf dt-month) (new-value dt))
(defgeneric (setf dt-year) (new-value dt))

(defstruct (datetime (:constructor %make-datetime)
                     (:conc-name dt-))
  (ut 0 :type integer)
  (time-zone 0 :type rational))

(defmethod print-object ((obj datetime) out)
  (print-unreadable-object (obj out :type t)
    (multiple-value-bind (sec min hour day month year)
        (decode-universal-time (dt-ut obj) 0)
      (format out "~4,'0d-~2,'0d-~2,'0d ~2,'0d:~2,'0d:~2,'0d UTC"
              year month day hour min sec))))

;;;
;;; Constructors
;;;

(defun make-datetime (&key (year 1900) (month 1) (day 1) (hour 0) (minute 0) (second 0) (time-zone 0))
  "Creates a new DATETIME instance and returns it."
  (%make-datetime :ut (encode-universal-time second minute hour day month year time-zone)
                  :time-zone time-zone))

(defun universal-time->datetime (ut &optional (time-zone 0))
  "Creates a new DATETIME instance from the given universal time."
  (multiple-value-bind (sec min hour day month year)
      (decode-universal-time ut time-zone)
    (make-datetime :year year :month month :day day :hour hour :minute min :second sec :time-zone time-zone)))

(defun unix->datetime (ut &optional (time-zone 0))
  "Creates a new DATETIME instance from the given Unix time value."
  (multiple-value-bind (sec min hour day month year)
      (decode-universal-time (unix->universal-time ut) time-zone)
    (make-datetime :year year :month month :day day :hour hour :minute min :second sec :time-zone time-zone)))

(defun make-now-datetime (&optional (time-zone 0))
  "Creates a new DATETIME with the current date and time."
  (universal-time->datetime (get-universal-time) time-zone))

;;;
;;; Accessors
;;;

(defmethod dt-second ((dt datetime))
  (multiple-value-bind (second)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    second))

(defmethod dt-minute ((dt datetime))
  (multiple-value-bind (second minute)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    (declare (ignore second))
    minute))

(defmethod dt-hour ((dt datetime))
  (multiple-value-bind (second minute hour)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    (declare (ignore second minute))
    hour))

(defmethod dt-day ((dt datetime))
  (multiple-value-bind (second minute hour day)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    (declare (ignore second minute hour))
    day))

(defmethod dt-month ((dt datetime))
  (multiple-value-bind (second minute hour day month)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    (declare (ignore second minute hour day))
    month))

(defmethod dt-year ((dt datetime))
  (multiple-value-bind (second minute hour day month year)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    (declare (ignore second minute hour day month))
    year))

(defmethod (setf dt-second) ((new-value integer) (dt datetime))
  (assert (typep new-value '(integer 0 59)) (new-value) "Seconds must be between 0 and 59, inclusive.")

  (multiple-value-bind (second minute hour day month year)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    (declare (ignore second))
    (setf (dt-ut dt) (encode-universal-time new-value minute hour day month year (dt-time-zone dt))))
  dt)

(defmethod (setf dt-minute) ((new-value integer) (dt datetime))
  (assert (typep new-value '(integer 0 59)) (new-value) "Minutes must be between 0 and 59, inclusive.")

  (multiple-value-bind (second minute hour day month year)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    (declare (ignore minute))
    (setf (dt-ut dt) (encode-universal-time second new-value hour day month year (dt-time-zone dt))))
  dt)

(defmethod (setf dt-hour) ((new-value integer) (dt datetime))
  (assert (typep new-value '(integer 0 59)) (new-value) "Hours must be between 0 and 59, inclusive.")

  (multiple-value-bind (second minute hour day month year)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    (declare (ignore hour))
    (setf (dt-ut dt) (encode-universal-time second minute new-value day month year (dt-time-zone dt))))
  dt)

(defmethod (setf dt-day) ((new-value integer) (dt datetime))
  (multiple-value-bind (second minute hour day month year)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    (declare (ignore day))
    (case month
      ((4 6 9 11)
       (assert (typep new-value '(integer 1 30)) (new-value)
               "Day must be between 1 and 30, inclusive, for the currently set month (~a)." month))

      ((1 3 5 7 8 10 12)
       (assert (typep new-value '(integer 1 31)) (new-value)
               "Day must be between 1 and 31, inclusive, for the currently set month (~a)." month))

      (otherwise
       (if (leap-year-p dt)
           (assert (typep new-value '(integer 1 29)) (new-value)
                   "Day must be between 1 and 29, inclusive, for the currently set month (2) and year.")
           (assert (typep new-value '(integer 1 28)) (new-value)
                   "Day must be between 1 and 28, inclusive, for the currently set month (2) and year."))))

    (setf (dt-ut dt) (encode-universal-time second minute hour new-value month year (dt-time-zone dt))))
  dt)

(defmethod (setf dt-month) ((new-value integer) (dt datetime))
  (assert (typep new-value '(integer 1 12)) (new-value) "Months must be between 1 and 12, inclusive.")

  (multiple-value-bind (second minute hour day month year)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    (declare (ignore month))
    (setf (dt-ut dt) (encode-universal-time second minute hour day new-value year (dt-time-zone dt))))
  dt)

(defmethod (setf dt-year) ((new-value integer) (dt datetime))
  (multiple-value-bind (second minute hour day month)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    (setf (dt-ut dt) (encode-universal-time second minute hour day month new-value (dt-time-zone dt))))
  dt)

;;;
;;; Utility Methods
;;;

(defmethod leap-year-p ((dt datetime))
  "Returns T if the date in DT is in a leap year, or NIL otherwise."
  (leap-year-p (dt-year dt)))

(defun dt-day-of-week (dt &optional as)
  "Returns the day of the week for the date stored in DT.  AS can be one of the
following: NIL (returns the day number), :SHORT-NAME (returns the abbreviated
day name), :NAME (returns the full day name), or :T/DAY-OF-WEEK (returns a
T/DAY-OF-WEEK)."
  (declare (type datetime dt)
           (type (or null keyword) as))
  (multiple-value-bind (second minute hour day month year dow)
      (decode-universal-time (dt-ut dt) (dt-time-zone dt))
    (declare (ignore second minute hour day month year))
    (incf dow)
    (cond
      ((null as) dow)
      ((eq as :name) (day-num->name dow))
      ((eq as :short-name) (day-num->name dow t))
      ((eq as :t/day-of-week) (num->t/day-of-week dow))
      (t (error "Unexpected value for AS: ~a" as)))))

(declaim (inline dt->unix))
(defun dt->unix (dt)
  "Converts a DATETIME to a Unix time value."
  (declare (type datetime dt))
  (universal-time->unix (dt-ut dt)))

(declaim (inline dt->ut))
(defun dt->ut (dt)
  "Converts a DATETIME to a Lisp universal time."
  (declare (type datetime dt))
  (dt-ut dt))

(defun dt-day-of-year (dt)
  "Returns the ordinal day of the year for the given DATETIME."
  (declare (type datetime dt))
  (1+ (ts-days (dt-/ts dt (make-datetime :year (dt-year dt))))))

;;;
;;; Comparison functions
;;;

(defun dt= (dt1 dt2)
  "Compares two DATETIME instances.  If they are the same date, this returns T,
otherwise it returns NIL."
  (declare (type datetime dt1 dt2))
  (= (dt-ut dt1) (dt-ut dt2)))

(defun dt< (dt1 dt2)
  "Compares two DATETIME instances.  If DT1 is strictly earlier than DT2, this
returns T, otherwise it returns NIL."
  (declare (type datetime dt1 dt2))
  (< (dt-ut dt1) (dt-ut dt2)))

(defun dt<= (dt1 dt2)
  "Compares two DATETIME instances.  If DT1 is earlier or equal to than DT2,
this returns T, otherwise it returns NIL."
  (declare (type datetime dt1 dt2))
  (<= (dt-ut dt1) (dt-ut dt2)))

(defun dt> (dt1 dt2)
  "Compares two DATETIME instances.  If DT1 is strictly later than DT2, this
returns T, otherwise it returns NIL."
  (declare (type datetime dt1 dt2))
  (> (dt-ut dt1) (dt-ut dt2)))

(defun dt>= (dt1 dt2)
  "Compares two DATETIME instances.  If DT1 is later or equal to than DT2,
this returns T, otherwise it returns NIL."
  (declare (type datetime dt1 dt2))
  (>= (dt-ut dt1) (dt-ut dt2)))

(defun dt+ (dt1 dt2)
  "Adds DT2 to DT1, returning a new DATETIME instance."
  (declare (type datetime dt1 dt2))
  (%make-datetime :ut (+ (dt-ut dt1) (dt-ut dt2))))

(defun dt- (dt1 dt2)
  "Subtracts DT2 from DT1, returning a new DATETIME instance."
  (declare (type datetime dt1 dt2))
  (%make-datetime :ut (- (dt-ut dt1) (dt-ut dt2))))

(defmethod ts+ ((dt datetime) (span timespan))
  "Adds the span of time in SPAN to the datetime DT, returning a new DATETIME
instance."
  (%make-datetime :ut (+ (dt-ut dt) (ts-total-seconds span))
                  :time-zone (dt-time-zone dt)))

(defmethod ts- ((dt datetime) (span timespan))
  "Subtracts the span of time in SPAN from the datetime DT, returning a new
DATETIME instance."
  (%make-datetime :ut (- (dt-ut dt) (ts-total-seconds span))
                  :time-zone (dt-time-zone dt)))

(defun dt+/ts (dt1 dt2)
  "Adds DT2 to DT1, returning a new TIMESPAN instance."
  (declare (type datetime dt1 dt2))
  (make-timespan-from-total-seconds (+ (dt-ut dt1) (dt-ut dt2))))

(defun dt-/ts (dt1 dt2)
  "Subtracts DT2 from DT1, returning a new TIMEPSAN instance."
  (declare (type datetime dt1 dt2))
  (make-timespan-from-total-seconds (- (dt-ut dt1) (dt-ut dt2))))

(defun dt+seconds (dt value)
  "Adds VALUE seconds to DT, returning a new DATETIME instance."
  (declare (type datetime dt)
           (type integer value))
  (%make-datetime :ut (+ (dt-ut dt) value)))

(defun dt+minutes (dt value)
  "Adds VALUE minutes to DT, returning a new DATETIME instance."
  (declare (type datetime dt)
           (type integer value))
  (%make-datetime :ut (+ (dt-ut dt) (* 60 value))))

(defun dt+hours (dt value)
  "Adds VALUE hours to DT, returning a new DATETIME instance."
  (declare (type datetime dt)
           (type integer value))
  (%make-datetime :ut (+ (dt-ut dt) (* 60 60 value))))

(defun dt+days (dt value)
  "Adds VALUE days to DT, returning a new DATETIME instance."
  (declare (type datetime dt)
           (type integer value))
  (%make-datetime :ut (+ (dt-ut dt) (* 60 60 24 value))))

(defun dt+months (dt value)
  "Adds VALUE months to DT, returning a new DATETIME instance."
  (declare (type datetime dt)
           (type integer value))

  (multiple-value-bind (sec min hour day month year)
      (decode-universal-time (dt-ut dt) 0)
    (declare (ignore sec min hour day))

    (loop for i from 0 below value
          for add-month = (+ month i)
          with days = 0
          do (when (> add-month 12)
               (setf add-month 1)
               (incf year))
             (incf days (days-in-month add-month (leap-year-p year)))
          finally (return (dt+days dt days)))))

(defun dt+years (dt value)
  "Adds VALUE years to DT, returning a new DATETIME instance."
  (declare (type datetime dt)
           (type integer value))

  (multiple-value-bind (sec min hour day month year)
      (decode-universal-time (dt-ut dt) 0)
    (declare (ignore sec min hour day month))

    (if (plusp value)
        (loop for i from 1 to value
              for add-year = (+ year i)
              with days = 0
              do (incf days (if (leap-year-p add-year) 366 365))
              finally (return (dt+days dt days)))

        (loop for i from value below 0
              for add-year = (+ year i)
              with days = 0
              do (decf days (if (leap-year-p add-year) 366 365))
              finally (return (dt+days dt days))))))
