;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/terminal)

(sb-alien:define-alien-type winsize (sb-alien:struct
                                     winsize
                                     (ws-row (sb-alien:unsigned 16))
                                     (ws-col (sb-alien:unsigned 16))
                                     (ws-x-pixel (sb-alien:unsigned 16))
                                     (ws-y-pixel (sb-alien:unsigned 16))))

(declaim (ftype (function (&optional t) (values (integer 0 65536) (integer 0 65536)))
                get-terminal-size))
(defun get-terminal-size (&optional (stream #+sbcl sb-sys:*stdout*))
  (let ((stream (typecase stream
                  (synonym-stream (synonym-stream-symbol stream))
                  (otherwise stream))))
    (unless (typep stream 'sb-sys:fd-stream)
      (error "GET-TERMINAL-SIZE only works with SB-SYS:FD-STREAM instances, or SYNONYM-STREAM instances that point to one"))

    (sb-alien:with-alien ((ret-alien (sb-alien:* winsize)))
      (unwind-protect
           (progn
             (setf ret-alien (sb-alien:make-alien winsize 1))
             (sb-unix:unix-ioctl (sb-sys:fd-stream-fd stream) #x5413 (sb-alien:alien-sap ret-alien))
             (values (sb-alien:slot ret-alien 'ws-col)
                     (sb-alien:slot ret-alien 'ws-row)))
        (sb-alien:free-alien ret-alien)))))
