;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/terminal)

;;;
;;; Conditions
;;;

(eval-when (:compile-toplevel :load-toplevel)
  (define-simple-error progress-bar-error () ())
  (define-simple-error progress-bar-done-error progress-bar-error ()))

;;;
;;; API
;;;

(defgeneric progress-bar-refresh (bar)
  (:documentation "Re-draws the progress bar.  This can be called even when
PROGRESS-BAR-DONE-P is truthy."))

(defgeneric progress-bar-done (bar)
  (:documentation "Refreshes the bar, then prints a newline to the output and
sets PROGRESS-BAR-DONE-P to T.  Calling this more than once is effectively a
non-op."))

(defgeneric progress-bar-pump (bar)
  (:documentation "Increases the step by one.  If PROGRESS-BAR-AUTO-REFRESH-P is
truthy, then the bar will be automatically redrawn.  This cannot be called once
PROGRESS-BAR-DONE-P is truthy or a PROGRESS-BAR-DONE-ERROR is raised."))

(defgeneric progress-bar-reset (bar)
  (:documentation "Resets the progress bar's step to 0.  If PROGRESS-BAR-DONE-P
is truthy, then it is reset to NIL."))

(defgeneric (setf progress-bar-auto-refresh-p) (value bar))
(defgeneric (setf progress-bar-allow-over-100-p) (value bar))

(defgeneric (setf progress-bar-label) (label bar))
(defgeneric (setf progress-bar-post-label) (label bar))
(defgeneric (setf progress-bar-stream) (stream bar))
(defgeneric (setf progress-bar-label-width) (width bar))
(defgeneric (setf progress-bar-post-label-width) (width bar))
(defgeneric (setf progress-bar-max) (max bar))
(defgeneric (setf progress-bar-step) (step bar))

(defclass progress-bar ()
  ((stream
    :initarg :stream
    :initform *standard-output*
    :type stream
    :reader progress-bar-stream
    :documentation "The output stream where the progress bar will be printed.")

   (auto-refresh?
    :initarg :auto-refresh
    :initform t
    :type boolean
    :reader progress-bar-auto-refresh-p
    :documentation "When T, then the progress bar will automatically be redrawn
after changing any of its values. If NIL, then you must call
PROGRESS-BAR-REFRESH manually to redraw the progress bar.")

   (add-ellipses?
    :initarg :add-elipses
    :initform t
    :type boolean
    :reader progress-bar-add-ellipses-p
    :documentation "When T, ellipses ('...') will be added to a clipped label,
if possible.")

   (allow-over-100?
    :initarg :allow-over-100
    :initform t
    :type boolean
    :reader progress-bar-allow-over-100-p
    :documentation "When NIL, then percentages that are above 100 will be capped
to 100.")

   (done?
    :initform nil
    :type boolean
    :reader progress-bar-done-p
    :documentation "Whether the progress bar is done or not.")

   (step
    :initform 0
    :type (integer 0 *)
    :reader progress-bar-step
    :documentation "The current step.")

   (max
    :initarg :max
    :initform 100
    :type (integer 1 *)
    :reader progress-bar-max
    :documentation "The maximum number of steps.")

   (label
    :initarg :label
    :initform ""
    :type string
    :reader progress-bar-label)

   (label-width
    :initarg :label-width
    :initform 0
    :type (integer 0 *)
    :reader progress-bar-label-width
    :documentation "The maximum width for the progress bar label.  If this is 0,
then the max width is computed to be 1/5th of the width of the console.")

   (post-label
    :initarg :post-label
    :initform nil
    :type (or null string)
    :reader progress-bar-post-label
    :documentation "An optional label that can be printed after the percentage
at the end of the progress bar.  If this is `nil`, then no extra label is
printed.  This can be up to POST-LABEL-WIDTH characters long.")

   (post-label-width
    :initarg :post-label-width
    :initform 0
    :type (integer 0 *)
    :reader progress-bar-post-label-width
    :documentation "The maximum width for the progress bar's post-bar label.  If
this is 0, then the max width is computed to be 1/5th of the width of the
console.")))

(defmethod initialize-instance :after ((bar progress-bar) &key &allow-other-keys)
  ;; Check initargs
  (with-slots (max label-width) bar
    (unless (> max 0)
      (error 'progress-bar-error :text "PROGRESS-BAR-MAX must be at least 1"))

    (unless (>= label-width 0)
      (error 'progress-bar-error :text "PROGRESS-BAR-LABEL-WIDTH must be at least 0"))))

(defmacro with-progress-bar ((bar label &key (max 100)) &body forms)
  `(let ((,bar (make-instance 'progress-bar :max ,max :label ,label)))
     ,@forms
     (progress-bar-done ,bar)))

;;;
;;; API Definitions
;;;

(defmethod progress-bar-reset ((bar progress-bar))
  (prog1 (setf (slot-value bar 'step) 0)
    (setf (slot-value bar 'done?) nil)
    (when (progress-bar-auto-refresh-p bar)
      (progress-bar-refresh bar))))

(defmethod (setf progress-bar-auto-refresh-p) (value (bar progress-bar))
  (setf (slot-value bar 'auto-refresh?) (if value t nil)))

(defmethod (setf progress-bar-allow-over-100-p) (value (bar progress-bar))
  (when (progress-bar-done-p bar)
    (progress-bar-error (progress-bar-done-error) "Progress bar is already finished"))
  (setf (slot-value bar 'allow-over-100?) (if value t nil)))

(defmethod (setf progress-bar-label) ((label string) (bar progress-bar))
  (when (progress-bar-done-p bar)
    (progress-bar-error (progress-bar-done-error) "Progress bar is already finished"))
  (prog1 (setf (slot-value bar 'label) label)
    (when (progress-bar-auto-refresh-p bar)
      (progress-bar-refresh bar))))

(defmethod (setf progress-bar-post-label) ((label string) (bar progress-bar))
  (when (progress-bar-done-p bar)
    (progress-bar-error (progress-bar-done-error) "Progress bar is already finished"))
  (prog1 (setf (slot-value bar 'post-label) label)
    (when (progress-bar-auto-refresh-p bar)
      (progress-bar-refresh bar))))

(defmethod (setf progress-bar-stream) ((stream stream) (bar progress-bar))
  (when (progress-bar-done-p bar)
    (progress-bar-error (progress-bar-done-error) "Progress bar is already finished"))
  (prog1 (setf (slot-value bar 'stream) stream)
    (when (progress-bar-auto-refresh-p bar)
      (progress-bar-refresh bar))))

(defmethod (setf progress-bar-label-width) ((width integer) (bar progress-bar))
  (check-type width (integer 0 *))
  (when (progress-bar-done-p bar)
    (progress-bar-error (progress-bar-done-error) "Progress bar is already finished"))
  (prog1 (setf (slot-value bar 'label-width) width)
    (when (progress-bar-auto-refresh-p bar)
      (progress-bar-refresh bar))))

(defmethod (setf progress-bar-post-label-width) ((width integer) (bar progress-bar))
  (check-type width (integer 0 *))
  (when (progress-bar-done-p bar)
    (progress-bar-error (progress-bar-done-error) "Progress bar is already finished"))
  (prog1 (setf (slot-value bar 'post-label-width) width)
    (when (progress-bar-auto-refresh-p bar)
      (progress-bar-refresh bar))))

(defmethod (setf progress-bar-max) ((max integer) (bar progress-bar))
  (check-type max (integer 1 *))
  (when (progress-bar-done-p bar)
    (progress-bar-error (progress-bar-done-error) "Progress bar is already finished"))
  (prog1 (setf (slot-value bar 'max) max)
    (when (progress-bar-auto-refresh-p bar)
      (progress-bar-refresh bar))))

(defmethod (setf progress-bar-step) ((value integer) (bar progress-bar))
  (check-type value (integer 0 *))
  (when (progress-bar-done-p bar)
    (progress-bar-error (progress-bar-done-error) "Progress bar is already finished"))
  (with-slots (step max) bar
    (prog1 (setf step (min value max))
      (when (progress-bar-auto-refresh-p bar)
        (progress-bar-refresh bar)))))

(defmethod progress-bar-pump ((bar progress-bar))
  (when (progress-bar-done-p bar)
    (progress-bar-error (progress-bar-done-error) "Progress bar is already finished"))
  (prog1 (incf (slot-value bar 'step))
    (when (progress-bar-auto-refresh-p bar)
      (progress-bar-refresh bar))))

(defmethod progress-bar-done ((bar progress-bar))
  (unless (progress-bar-done-p bar)
    (progress-bar-refresh bar)
    (ansi/cursor-ctrl (progress-bar-stream bar) :direction :to-column :cells 1)
    (terpri (progress-bar-stream bar))
    (setf (slot-value bar 'done?) t)))

;;;
;;; Main refresh routine
;;;

(defmethod progress-bar-refresh ((bar progress-bar))
  (declare (optimize (speed 3) (safety 0) (debug 1)))

  (with-typed-slots ((stream stream)
                     ((integer 1 *) max)
                     (string label)
                     ((or null string) post-label)
                     (boolean add-ellipses? allow-over-100? done?)
                     ((integer 0 *) label-width step post-label-width))
      bar
    (ansi/cursor-ctrl stream :direction :to-column :cells 1)
    ;;(ansi/erase-in-line stream) ;; Likely not needed

    (let* ((width 0)
           (actual-lbl-length (length label))
           (post-lbl-width 0)
           (lbl-width 0)
           (bar-width 0)
           (con-width 0))
      (declare (type t/int32 width actual-lbl-length lbl-width bar-width con-width post-lbl-width))

      (handler-case
          (setf con-width (get-terminal-size))
        (error ()
          (setf con-width 80)))

      (if (< con-width 10)
          ;; Not enough room
          (return-from progress-bar-refresh)
          (setf width (1- con-width)))

      ;; Calculate label width
      (setf lbl-width (if (zerop label-width)
                          (let ((new-width (truncate width 5)))
                            (if (zerop new-width) 5 new-width))
                          label-width))
      (when (< actual-lbl-length lbl-width)
        (setf lbl-width actual-lbl-length))

      ;; Calculate post-label width
      (setf post-lbl-width (if (zerop post-label-width)
                               (let ((new-width (truncate width 5)))
                                 (if (zerop new-width) 5 new-width))
                               post-label-width))
      (when-var (lbl post-label)
        (when (< (length lbl) post-lbl-width)
          (setf post-lbl-width (length lbl))))

      ;; Calculate bar width.
      ;;
      ;; The -5 is to account for the percent display.  The -2 is to account for
      ;; the two vertical bars.  The -1 is to account for a space between the
      ;; label and bar.
      (setf bar-width (- width lbl-width 5 2 1))
      (when post-label
        (decf bar-width 1) ;; Extra space after the percentage
        (decf bar-width post-lbl-width))

      (when (minusp bar-width)
        (error "BAR-WIDTH went negative"))

      (if (< actual-lbl-length (1+ lbl-width))
          (write-string label stream)
          (if (and add-ellipses? (>= lbl-width 4))
              (format stream "~a..." (subseq label 0 (- lbl-width 3)))
              (write-string (subseq label 0 lbl-width) stream)))
      (write-char #\: stream)

      ;; Write the bar
      (let* ((percent 0.0)
             (progress 0))
        (declare (single-float percent)
                 (type t/uint32 progress))
        (muffling
          (setf percent (* (/ (coerce step 'single-float) max) 100.0)))
        (when (and (> percent 100) (not allow-over-100?))
          (setf percent 100.0))
        (muffling
          (setf progress (truncate (/ (* (min percent 100.0) bar-width) 100))))
        (format stream "|~a~a|"
                (make-string progress :initial-element #\*)
                (make-string (- bar-width progress) :initial-element #\-))

        ;; Write percent
        (format stream "~4d%" (truncate percent)))

      ;; Write the post-bar label, if needed
      (when-var (lbl post-label)
        (write-char #\Space stream)
        (if (< (length lbl) (1+ post-lbl-width))
            (write-string lbl stream)
            (if (and add-ellipses? (>= post-lbl-width 4))
                (format stream "~a..." (subseq lbl 0 (- post-lbl-width 3)))
                (write-string (subseq lbl post-lbl-width) stream)))))
    (finish-output stream)))
