;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/terminal)

(deftype t/erase-in-display ()
  '(member :cursor->end :cursor->start :whole-screen :screen+scrollback))

(deftype t/erase-in-line ()
  '(member :cursor->end :cursor->start :whole-line))

(deftype t/cursor-ctrl-direction ()
  '(member :up :down :forward :backward :next-line :prev-line :to-column))

(deftype t/ansi-simple-color ()
  "The core 16 ANSI terminal colors."
  '(member :black :red :green :yellow :blue :magenta :cyan :white :bright-black :bright-red
    :bright-green :bright-yellow :bright-blue :bright-magenta :bright-cyan :bright-white))

(defining-consts
  (+ansi-csi+   (strings+ #\Escape #\[))
  (+ansi-reset+ (strings+ #\Escape #\[ #\m))

  ;; (foreground . background)
  (+ansi-color-map+ '(:black          (30 . 40)
                      :red            (31 . 41)
                      :green          (32 . 42)
                      :yellow         (33 . 43)
                      :blue           (34 . 44)
                      :magenta        (35 . 45)
                      :cyan           (36 . 46)
                      :white          (37 . 47)
                      :bright-black   (90 . 100)
                      :bright-red     (91 . 101)
                      :bright-green   (92 . 102)
                      :bright-yellow  (93 . 103)
                      :bright-blue    (94 . 104)
                      :bright-magenta (95 . 105)
                      :bright-cyan    (96 . 106)
                      :bright-white   (97 . 107)
                      :default        (39 . 49))))

;;;
;;; Internal stuff
;;;

(declaim (ftype (function (string) (values string string)) chop-off-final-blank-newlines))
(defun chop-off-final-blank-newlines (string)
  (declare (type string string)
           (optimize (speed 3) (debug 1)))

  (let ((split (split-string string #\Newline)))
    (declare (type simple-vector split))
    (loop for line string across (reverse split)
          for count fixnum from 0
          while (string= line "")
          finally (return (values (subseq string 0 (- (length string) count))
                                  (make-string count :initial-element #\Newline))))))

(defmacro ansi/with-csi-block ((var output) &body forms)
  `(progn
     (princ (with-output-to-string (,var)
              (write-string +ansi-csi+ ,var)
              ,@forms
              (write-string +ansi-reset+ ,var))
            ,output)
     nil))

;;;
;;; Simple formatting API
;;;

(defmacro ansi/format-bold (output msg &rest fmt-args)
  "Instructs OUTPUT to output bold characters, then formats MSG using
FMT-ARGS and writes the result to OUTPUT.  If OUTPUT is T, then
*STANDARD-OUTPUT* will be used."
  (with-gensyms (final-str)
    `(ansi/with-csi-block (,final-str ,output)
       (write-string (strings+ 1 #\m) ,final-str)
       (format ,final-str ,msg ,@fmt-args))))

(defmacro ansi/format-colored ((output fg-color &key (bg-color :default) with-blank-lines) msg &rest fmt-args)
  "Instructs OUTPUT to output text using the specified colors, then
formats MSG using FMT-ARGS and writes the result to OUTPUT.  If OUTPUT
is T, then *STANDARD-OUTPUT* will be used.

If WITH-BLANK-LINES is non-NIL, any final blank lines are also
formatted using these colors (which may look odd and unexpected).
Otherwise, any final blank lines are formatted without colors."
  (with-gensyms (fg-cons bg-cons final-str real-str blank-lines)
    `(let ((,fg-cons (getf +ansi-color-map+ ,fg-color))
           (,bg-cons (getf +ansi-color-map+ ,bg-color)))

       (unless ,fg-cons
         (error "Could not find the foreground color ~a" ,fg-color))

       (unless ,bg-cons
         (error "Could not find the background color ~a" ,bg-color))

       ,(if (not with-blank-lines)
            `(multiple-value-bind (,real-str ,blank-lines)
                 (chop-off-final-blank-newlines (format nil ,msg ,@fmt-args))
               (ansi/with-csi-block (,final-str ,output)
                 (write-string (strings+ (car ,fg-cons) #\; (cdr ,bg-cons) #\m) ,final-str)
                 (write-string ,real-str ,final-str))
               (write-string ,blank-lines ,output))

            `(ansi/with-csi-block (,final-str ,output)
               (write-string (strings+ (car ,fg-cons) #\; (cdr ,bg-cons) #\m) ,final-str)
               (format ,final-str ,msg ,@fmt-args))))))

(defmacro ansi/with-color ((output fg-color &optional (bg-color :default)) &body forms)
  (with-gensyms (fg-cons bg-cons)
    `(let ((,fg-cons (getf +ansi-color-map+ ,fg-color))
           (,bg-cons (getf +ansi-color-map+ ,bg-color)))

       (unless ,fg-cons
         (error "Could not find the foreground color ~a" ,fg-color))

       (unless ,bg-cons
         (error "Could not find the background color ~a" ,bg-color))

       (princ +ansi-csi+ ,output)
       (princ (strings+ (car ,fg-cons) #\; (cdr ,bg-cons) #\m) ,output)
       ,@forms
       (princ +ansi-reset+ ,output))))

(defmacro ansi/with-256-color ((output fg-color &optional bg-color) &body forms)
  `(progn
     (locally
         (declare #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
       (unless (typep ,fg-color 't/uint8)
         (error "Bad 256-color for foreground: ~a" ,fg-color))
       ,(when bg-color
          `(unless (typep ,bg-color '(or null t/uint8))
             (error "Bad 256-color for background: ~a" ,bg-color))))

     (princ +ansi-csi+ ,output)
     (princ (format nil "38;5;~am" ,fg-color) ,output)
     ,(when bg-color
        `(princ (format nil "48;5;~am" ,bg-color) ,output))
     (prog1 (progn ,@forms)
       (princ +ansi-reset+ ,output))))

(defun colorize-string (string fg-color &optional (bg-color :default))
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (with-output-to-string (out)
    (ansi/with-color (out fg-color bg-color)
      (write-string string out))))

(defun colorize-string-256 (string fg-color &optional (bg-color :default))
  (declare (type string string)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (with-output-to-string (out)
    (ansi/with-256-color (out fg-color bg-color)
      (write-string string out))))

(defun bold-string (string)
  (declare (type string string)
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (with-output-to-string (out)
    (ansi/format-bold out string)))

;;;
;;; Complex formatting API
;;;

(defmacro ansi-block/define-block (var start-num end-num &body forms)
  `(progn
     (write-string +ansi-csi+ ,var)
     (write-string (strings+ ,start-num #\m) ,var)
     (ansi-block/expanding-forms ,var ,@forms)
     (format ,var +ansi-csi+)
     (write-string (strings+ ,end-num #\m) ,var)))

(defmacro ansi-block/bold (var &body forms)
  `(ansi-block/define-block ,var 1 22 ,@forms))

(defmacro ansi-block/faint (var &body forms)
  `(ansi-block/define-block ,var 2 22 ,@forms))

(defmacro ansi-block/italic (var &body forms)
  `(ansi-block/define-block ,var 3 23 ,@forms))

(defmacro ansi-block/underline (var &body forms)
  `(ansi-block/define-block ,var 4 24 ,@forms))

(defmacro ansi-block/slow-blink (var &body forms)
  `(ansi-block/define-block ,var 5 25 ,@forms))

(defmacro ansi-block/fast-blink (var &body forms)
  `(ansi-block/define-block ,var 6 25 ,@forms))

(defmacro ansi-block/reverse-video (var &body forms)
  `(ansi-block/define-block ,var 7 7 ,@forms))

(defmacro ansi-block/conceal (var &body forms)
  `(ansi-block/define-block ,var 8 28 ,@forms))

(defmacro ansi-block/crossed-out (var &body forms)
  `(ansi-block/define-block ,var 9 29 ,@forms))

(defmacro ansi-block/fraktur (var &body forms)
  `(ansi-block/define-block ,var 20 23 ,@forms))

(defmacro ansi-block/framed (var &body forms)
  `(ansi-block/define-block ,var 51 54 ,@forms))

(defmacro ansi-block/encircled (var &body forms)
  `(ansi-block/define-block ,var 52 54 ,@forms))

(defmacro ansi-block/overlined (var &body forms)
  `(ansi-block/define-block ,var 53 55 ,@forms))

(defmacro ansi-block/colored ((var fg-color &optional (bg-color :default)) &body forms)
  (with-gensyms (fg-cons bg-cons)
    `(let ((,fg-cons (getf +ansi-color-map+ ,fg-color))
           (,bg-cons (getf +ansi-color-map+ ,bg-color)))

       (unless ,fg-cons
         (error "Could not find the foreground color ~a" ,fg-color))

       (unless ,bg-cons
         (error "Could not find the background color ~a" ,bg-color))

       (write-string +ansi-csi+ ,var)
       (write-string (strings+ (car ,fg-cons) #\m) ,var)
       ,(unless (eq bg-color :default)
          `(progn
             (write-string +ansi-csi+ ,var)
             (write-string (strings+ (cdr ,bg-cons) #\m) ,var)))
       (ansi-block/expanding-forms ,var ,@forms)
       (write-string +ansi-csi+ ,var)
       (write-string (strings+ 39 #\m) ,var)
       ,(unless (eq bg-color :default)
          `(progn
             (write-string +ansi-csi+ ,var)
             (write-string (strings+ 49 #\m) ,var))))))

(defmacro ansi-block/expanding-forms (var &body forms)
  (let ((ret ()))
    (dolist (form forms)
      (typecase form
        (list
         (cond
           ((equal (car form) :bold)
            (push `(ansi-block/bold ,var ,@(cdr form)) ret))

           ((equal (car form) :faint)
            (push `(ansi-block/faint ,var ,@(cdr form)) ret))

           ((equal (car form) :italic)
            (push `(ansi-block/italic ,var ,@(cdr form)) ret))

           ((equal (car form) :underline)
            (push `(ansi-block/underline ,var ,@(cdr form)) ret))

           ((equal (car form) :slow-blink)
            (push `(ansi-block/slow-blink ,var ,@(cdr form)) ret))

           ((equal (car form) :fast-blink)
            (push `(ansi-block/fast-blink ,var ,@(cdr form)) ret))

           ((equal (car form) :reverse-video)
            (push `(ansi-block/reverse-video ,var ,@(cdr form)) ret))

           ((equal (car form) :conceal)
            (push `(ansi-block/conceal ,var ,@(cdr form)) ret))

           ((equal (car form) :crossed-out)
            (push `(ansi-block/crossed-out ,var ,@(cdr form)) ret))

           ((equal (car form) :fraktur)
            (push `(ansi-block/fraktur ,var ,@(cdr form)) ret))

           ((equal (car form) :framed)
            (push `(ansi-block/framed ,var ,@(cdr form)) ret))

           ((equal (car form) :encircled)
            (push `(ansi-block/encircled ,var ,@(cdr form)) ret))

           ((equal (car form) :overlined)
            (push `(ansi-block/overlined ,var ,@(cdr form)) ret))

           ((equal (car form) :colored)
            (unless (keywordp (cadr form))
              (error "No foreground color specified.  This is required."))

            (push `(ansi-block/colored (,var ,(cadr form)
                                             ,(if (keywordp (caddr form))
                                                  (caddr form)
                                                  :default))
                     ,@(if (keywordp (caddr form))
                           (cdddr form)
                           (cddr form)))
                  ret))

           ((equal (car form) :format)
            (push `(funcall #'format ,var ,(cadr form) ,@(cddr form)) ret))

           (t (push `(cl:write-string ,form ,var) ret))))

        (string
         (push `(format ,var ,form) ret))

        (t (error "Unsure how to handle form: ~s" form))))

    (if (= (length ret) 1)
        (car ret)
        `(progn ,@(nreverse ret)))))

(defmacro ansi/format (output &body forms)
  "This is the generic formatting macro.  `OUTPUT` is the stream to
write to, and can be `T` (for `*STANDARD-OUTPUT*`) or a stream.

`FORMS` is a set of s-expressions or strings that are used to create the
desired output.  S-expression blocks are used to set the desired
formatting and can be nested.  Each one of these blocks starts with a
keyword to indicate the desired type of formatting:

|
  :BOLD   - Bold formatted text
  :FAINT  - Faint text
  :ITALIC - Italic formatted text
  :UNDERLINE - Underlined text
  :SLOW-BLINK - Text that slowly blinks
  :FAST-BLINK - Text that blinks quickly
  :REVERSE-VIDEO - Swapped foreground and background colors
  :CONCEAL - Concealed text
  :CROSSED-OUT - Text that is crossed out
  :FRAKTUR - Blackletter text
  :FRAMED - Text with a frame around it
  :ENCIRCLED - Text with a circle around it (?)
  :OVERLINED - Text with a line over it
|

There is also a special form for colored text:
|
  (:COLORED <foreground color> [optional background color] ...forms...)
|

One last special form is `(:FORMAT <format string> [format args...])`,
which is equivalent to calling the normal `FORMAT` function."
  (with-gensyms (var)
    `(progn
       (format ,output (with-output-to-string (,var)
                         (ansi-block/expanding-forms ,var ,@forms)
                         (write-string +ansi-reset+ ,var)))
       t)))

;;;
;;; Remaining API
;;;

(declaim (ftype (function (T &key (:direction t/cursor-ctrl-direction) (:cells (integer 1 65536))) null)
                ansi/cursor-ctrl)
         (inline ansi/cursor-ctrl))
(defun ansi/cursor-ctrl (output &key (direction :up) (cells 1))
  "Basic cursor control for the OUTPUT stream.  DIRECTION can
be: :UP, :DOWN, :FORWARD:, BACKWARD, :NEXT-LINE, :PREV-LINE,
or :TO-COLUMN.  CELLS controls the number of cells to move (or in the
case of :TO-COLUMN, the column to move to)."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type cells (integer 1 65536))
  (check-type direction t/cursor-ctrl-direction)

  (princ (with-output-to-string (str)
           (write-string +ansi-csi+ str)
           (format str "~d" cells)
           (ecase direction
             (:up        (write-char #\A str))
             (:down      (write-char #\B str))
             (:forward   (write-char #\C str))
             (:backward  (write-char #\D str))
             (:next-line (write-char #\E str))
             (:prev-line (write-char #\F str))
             (:to-column (write-char #\G str))))
         output)
  nil)

(declaim (ftype (function (T &key (:row fixnum) (:column fixnum)) null) ansi/set-cursor-pos)
         (inline ansi/set-cursor-pos))
(defun ansi/set-cursor-pos (output &key (row 1) (column 1))
  "Moves the cursor on the OUTPUT stream to the given row and column."
  (declare (type (integer 1 65536) row column)
           (optimize (speed 3) (debug 1)))

  #-sbcl (unless (> row 0)
    "ROW must be positive")
  #-sbcl (unless (> column 0)
    "COLUMN must be positive")

  (princ (with-output-to-string (str)
           (write-string +ansi-csi+ str)
           (format str "~d;~dH" row column))
         output)
  nil)

(declaim (ftype (function (T &optional t/erase-in-display) null) ansi/erase-in-display)
         (inline ansi/erase-in-display))
(defun ansi/erase-in-display (output &optional (from :cursor->end))
  "Erases the display on the OUTPUT stream according to FROM.  FROM
  can be one of:

  :CURSOR->END       - Erase from the cursor to the end of the display
  :CURSOR->START     - Erase from the cursor to the start of the
                       display
  :WHOLE-SCREEN      - Erase the entire screen
  :SCREEN+SCROLLBACK - Erase the entire screen and the scrollback
                       buffer"
  (declare (type t/erase-in-display from)
           (optimize (speed 3) (debug 1)))

  (princ (with-output-to-string (str)
           (write-string +ansi-csi+ str)
           (format str "~dJ" (case from
                               (:cursor->end 0)
                               (:cursor->start 1)
                               (:whole-screen 2)
                               (:screen+scrollback 3))))
         output)
  nil)

(declaim (ftype (function (T &optional t/erase-in-line) null) ansi/erase-in-line)
         (inline ansi/erase-in-line))
(defun ansi/erase-in-line (output &optional (from :cursor->end))
  "Erases the current line on the OUTPUT stream according to FROM.
  FROM can be one of:

  :CURSOR->END   - Erase from the cursor to the end of the line
  :CURSOR->START - Erase from the cursor to the start of the line
  :WHOLE-LINE    - Erases the entire line"
  (declare (type t/erase-in-line from)
           (optimize (speed 3) (debug 1)))

  (princ (with-output-to-string (str)
           (write-string +ansi-csi+ str)
           (format str "~dK" (case from
                               (:cursor->end 0)
                               (:cursor->start 1)
                               (:whole-line 2))))
         output)
  nil)

(declaim (ftype (function (T &key (:lines (integer 1 65536)) (:direction (member :up :down))) null)
                ansi/scroll)
         (inline ansi/scroll))
(defun ansi/scroll (output &key (lines 1) (direction :up))
  "Scrolls the OUTPUT stream LINES number of lines.  DIRECTION can
be :UP or :DOWN."
  (declare (type (integer 1 65536) lines)
           (type (member :up :down) direction)
           (optimize (speed 3) (debug 1)))

  #-sbcl (unless (> lines )
           "LINES must be positive")

  (princ (with-output-to-string (str)
           (write-string +ansi-csi+ str)
           (format str "~d" lines)
           (case direction
             (:up (write-char #\S str))
             (:down (write-char #\T str))))
         output)

  nil)

(declaim (ftype (function (T &optional (member :on :off)) null) ansi/aux-port-ctrl)
         (inline ansi/aux-port-ctrl))
(defun ansi/aux-port-ctrl (output &optional (state :on))
  "Enables or disables the AUX port on the OUTPUT stream.  STATE can
be :ON or :OFF."
  (declare (type (member :on :off) state)
           (optimize (speed 3) (debug 1)))

  (princ (with-output-to-string (str)
           (write-string +ansi-csi+ str)
           (format str "~di" (case state
                               (:on 5)
                               (:off 4))))
         output)

  nil)

(declaim (ftype (function (T) null) ansi/save-cursor-pos)
         (inline ansi/save-cursor-pos))
(defun ansi/save-cursor-pos (output)
  "Saves the current cursor position on the OUTPUT stream."
  (declare (optimize (speed 3) (debug 1)))
  (princ (with-output-to-string (str)
           (write-string +ansi-csi+ str)
           (write-char #\s str))
         output)
  nil)

(declaim (ftype (function (T) null) ansi/restore-cursor-pos)
         (inline ansi/restore-cursor-pos))
(defun ansi/restore-cursor-pos (output)
  "Restores the cursor position on the OUTPUT stream to the position that was
saved using ANSI/SAVE-CURSOR-POS."
  (princ (with-output-to-string (str)
           (write-string +ansi-csi+ str)
           (write-char #\u str))
         output)
  nil)

(declaim (ftype (function (T) null) ansi/hide-cursor)
         (inline ansi/hide-cursor))
(defun ansi/hide-cursor (output)
  "Hides the cursor on the OUTPUT stream (VT320-compatible)."
  (declare (optimize (speed 3) (debug 1)))
  (princ (with-output-to-string (str)
           (write-string +ansi-csi+ str)
           (write-string "?25l" str))
         output)
  nil)

(declaim (ftype (function (T) null) ansi/show-cursor)
         (inline ansi/show-cursor))
(defun ansi/show-cursor (output)
  "Unhides the cursor on the OUTPUT stream (VT320-compatible)."
  (declare (optimize (speed 3) (debug 1)))
  (princ (with-output-to-string (str)
           (write-string +ansi-csi+ str)
           (write-string "?25h" str))
         output)
  nil)

(declaim (ftype (function (T) null) ansi/reset-terminal)
         (inline ansi/reset-terminal))
(defun ansi/reset-terminal (output)
  "Resets the terminal (equivalent to calling 'reset' from the prompt)."
  (declare (optimize (speed 3) (debug 1)))
  (princ (with-output-to-string (str)
           (write-string (strings+ #\Escape #\c) str))
         output)
  nil)

(declaim (ftype (function (T) null) ansi/default-font)
         (inline ansi/default-font))
(defun ansi/default-font (output)
  "Changes the font on the OUTPUT stream to the default one."
  (declare (optimize (speed 3) (debug 1)))
  (princ (with-output-to-string (str)
           (write-string +ansi-csi+ str)
           (write-string "10m" str))
         output)
  nil)

(declaim (ftype (function (T (integer 1 9)) null) ansi/set-alternate-font)
         (inline ansi/set-alternate-font))
(defun ansi/set-alternate-font (output font)
  "Changes the font on the OUTPUT stream to the FONT specified.  FONT
can be a number between 1 and 9, inclusive."
  (declare (type (integer 1 9) font)
           (optimize (speed 3) (debug 1)))

  #-sbcl (unless (and (< font 10) (> font 0))
           (error "FONT must be between 1 and 9, inclusive"))

  (princ (with-output-to-string (str)
           (write-string +ansi-csi+ str)
           (format str "~dm" (+ 10 font)))
         output)

  nil)

(declaim (ftype (function (T string) null) xterm/set-window-title)
         (inline xterm/set-window-title))
(defun xterm/set-window-title (output title)
  "Sets the window title to TITLE on the OUTPUT stream."
  (declare (type string title)
           (optimize (speed 3) (debug 1)))

  (progn
    (princ (with-output-to-string (str)
             (write-string (strings+ #\Escape #\]) str)
             (format str "0; ~a" title)
             (write-string (strings+ #\Escape #\\) str))
           output)
    nil))

#|
(defun ansi/get-cursor-pos (output)
  (princ (with-output-to-string (str)
           (format str +ansi-csi+)
           (format str "6n"))
         output)

  (loop with ret = ""
        for c = (read-char output)
        if (char= c #\R)
          do (loop-finish)
        else
          do (setf ret (p36:strings+ ret c))
        finally (progn (terpri output) (print ret))))
|#
