;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/terminal)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Terminology support
;;;

(defmacro with-terminology-escape ((output &optional no-end-of-sequence) &body forms)
  (let ((ret `(progn
                (write-string (strings+ #\Escape #\}) ,output)
                ,@forms)))
    (if no-end-of-sequence
        ret
        `(,@ret (write-string (strings+ #\Null) ,output)))))

(defmacro with-terminology-princ ((final-output var &optional no-end-of-sequence) &body forms)
  `(princ (with-output-to-string (,var)
            (with-terminology-escape (,var ,no-end-of-sequence)
              ,@forms))
          ,final-output))

(deftype t/terminology-media-mode ()
  '(member :centered :filled :stretched))

(defun terminology/show-media (output width height media-path &optional (display-mode :centered))
  "Inserts/shows the given media in the terminal.  The WIDTH and HEIGHT parameters are in cells, and cannot exceed
512.  DISPLAY-MODE is a T/TERMINOLOGY-MEDIA-MODE and defaults to :CENTERED."
  (declare (type (integer 1 512) width height)
           (type simple-string media-path)
           (type t/terminology-media-mode display-mode)
           (optimize (speed 3) (debug 1)))

  (let ((line-str "")
        (cmd ""))
    (declare (type simple-string line-str cmd)
             (dynamic-extent line-str cmd))
    (setf line-str (make-string width :initial-element #\#))
    (setf cmd (ecase display-mode
                (:centered "ic")
                (:filled "if")
                (:stretched "is")))

    (with-terminology-princ (output str t)
      (format str "~a#~d;~d;~a" cmd width height media-path)
      (write-string (strings+ #\Null) str)

      (do ((i 0 (1+ i)))
          ((>= i height) nil)
        (declare (type (integer 0 512) i)
                 (dynamic-extent i))
        (write-string (strings+ #\Escape #\} "ib" #\Null line-str #\Escape #\} "ie" #\Null #\Newline) str))))
  nil)

(defun terminology/show-thumb (output width height media-path &optional link)
  "Inserts/shows a thumbnail/low-res version of the given media in the terminal.  The WIDTH and HEIGHT parameters
are in cells, and cannot exceed 512.  LINK can be a path or URL to open when the thumbnail is clicked, or T if the
MEDIA-PATH should also be used for the link."
  (declare (type (integer 1 512) width height)
           (type simple-string media-path)
           (type (or null simple-string boolean) link)
           (optimize (speed 3) (debug 1)))

  (let ((line-str ""))
    (declare (type simple-string line-str)
             (dynamic-extent line-str))
    (setf line-str (make-string width :initial-element #\#))

    (with-terminology-princ (output str t)
      (cond
        ((eql link t)
         (format str "it#~d;~d;~a~%~a" width height media-path media-path))
        ((stringp link)
         (format str "it#~d;~d;~a~%~a" width height link media-path))
        (t (format str "it#~d;~d;~a" width height media-path)))
      (write-string (strings+ #\Null) str)

      (dotimes (i height)
        (write-string (strings+ #\Escape #\} "ib" #\Null line-str #\Escape #\} "ie" #\Null #\Newline) str))))
  nil)

(defun terminology/query-grid ()
  "Queries the grid and font size for *STANDARD-OUTPUT*, then returns four values:
1. The width of the terminal in characters
2. The height of the terminal in characters
3. The width of one character in pixels
4. The height of one character in pixels

IMPORTANT NOTE: The terminal writes the results to standard input!  This writes the command to *STANDARD-OUTPUT*,
calls (FINISH-OUTPUT *STANDARD-OUTPUT*), then immediately reads a line from *STANDARD-INPUT*.  Keep that in mind."
  (declare (optimize (speed 3) (debug 1)))

  (with-terminology-princ (*standard-output* str)
    (write-string "qs" str))
  (finish-output *standard-output*)
  (let ((ret (read-line *standard-input*)))
    (values-list (mapcar #'parse-integer (split-string ret #\; :as-list t)))))

(defun terminology/set-alpha-state (output enable? &optional make-permanent)
  "Sets the background state on or off depending on if ENABLE is truthy or not, respectively.  If MAKE-PERMANENT is
NIL, the change is a temporary one."
  (with-terminology-princ (output str)
    (if enable?
        (if make-permanent
            (write-string "apon" str)
            (write-string "aton" str))
        (if make-permanent
            (write-string "apoff" str)
            (write-string "atoff" str))))
  nil)

(defun terminology/set-background (output media-path &optional make-permanent)
  "Sets the background to MEDIA-PATH.  If MAKE-PERMANENT is NIL, the change is a temporary one."
  (declare (type simple-string media-path)
           (optimize (speed 3) (debug 1)))
  (with-terminology-princ (output str)
    (if make-permanent
        (format str "bp~a" media-path)
        (format str "bt~a" media-path)))
  nil)

(defun terminology/popup (output media-path &optional enqueue)
  "Pops MEDIA-PATH up in the terminal.  If ENQUEUE is truthy, then the popup is queued up by the terminal,
otherwise it is immediately shown."
  (declare (type simple-string media-path)
           (optimize (speed 3) (debug 1)))
  (with-terminology-princ (output str)
    (if enqueue
        (format str "pq~a" media-path)
        (format str "pn~a" media-path)))
  nil)
