;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/terminal)

(defconst +terminology-expected+ (strings+ #\Escape "P!|7E7E5459" #\Escape #\\))

(defun terminology/running-in-terminology-p (&optional (stream sb-sys:*stdout*))
  "Returns T if STREAM outputs to a Terminology terminal, or NIL otherwise.
This can mess up SWANK and require the Lisp processed to be SIGKILL'd, so be
careful."
  (unless (sb-sys:fd-stream-p stream)
    (error "SB-SYS:FD-STREAM required"))

  ;; Get the current terminal state and a copy of the state.
  (let ((orig-attrs (sb-posix:tcgetattr stream))
        (new-attrs (sb-posix:tcgetattr stream)))

    (unwind-protect
         (progn
           ;; Clear input buffer

           ;; Turn off canonical (buffered) mode and echo
           (setf (sb-posix:termios-lflag new-attrs)
                 (logand (sb-posix:termios-lflag new-attrs)
                         (lognot (logior sb-posix:icanon sb-posix:echo))))

           ;; Minimum of number input read: 1 byte
           (setf (elt (sb-posix:termios-cc new-attrs) sb-posix:vmin) 1)
           (setf (elt (sb-posix:termios-cc new-attrs) sb-posix:vtime) 0)

           ;; Set new terminal attributes
           (sb-posix:tcsetattr stream sb-posix:tcsanow new-attrs)

           ;; Query the device attributes
           (princ (strings+ #\Escape "[=c") stream)
           (finish-output stream)

           ;; Read input
           (with-output-to-string (out)
             (loop for i from 1 to (length +terminology-expected+)
                   for byte = (read-byte sb-sys:*stdin* nil nil)
                   while byte do
                     (print byte)
                     (write-char (code-char byte) out)
                   finally (return (if (and (= i (length +terminology-expected+))
                                            (string= (get-output-stream-string out) +terminology-expected+))
                                       t
                                       nil)))))

      ;; Restore old terminal attributes
      (sb-posix:tcsetattr stream sb-posix:tcsanow orig-attrs))))
