;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;; Copyright 2012-2025 Manas Technology Solutions
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; SIZED-STREAM Class and Methods
;;;;
;;;; This is based on the IO::Sized class in the Crystal standard library.
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-condition sized-stream-closed-error (stream-error)
  ()
  (:documentation "Raised when a SIZED-STREAM is closed, but a read function is attempted on it."))

(deftype t/sized-stream-max ()
  "Equivalent to (AND UNSIGNED-BYTE FIXNUM)"
  't/ufixnum)

(defclass sized-stream (binary-utf-8-input-stream)
  ((stream
    :initarg :stream
    :type stream
    :documentation "The underlying stream.")

   (sync-close?
    :initform nil
    :type boolean
    :reader sized-stream-sync-close-p
    :documentation "When T, then the wrapped underlying stream is closed when this instance is
closed.  Otherwise the underlying wrapped stream will remaining open.")

   (closed?
    :initform nil
    :type boolean
    :reader sized-stream-closed-p
    :documentation "Returns T if this instance has been closed, or NIL otherwise.  Depending on if
SIZED-STREAM-SYNC-CLOSE-P was T at the time of closing, the underlying wrapped
stream may still be open.")

   (read-remaining
    :initform 0
    :type t/sized-stream-max
    :reader sized-stream-read-remaining
    :documentation "The number of remaining bytes to be read."))
  (:documentation "A binary input stream that wraps another stream, setting a limit on the number
of bytes that can be read from it.  You must use MAKE-SIZED-STREAM to create
instances of this class.

This class cannot PEEK-CHAR or UNREAD-CHAR.

The maximum restricted size is T/SIZED-STREAM-MAX."))

(defmethod initialize-instance :after ((stream sized-stream) &key &allow-other-keys)
  (unless (slot-boundp stream 'stream)
    (error "STREAM slot was not bound")))

(defmethod (setf sized-stream-sync-close-p) (value (stream sized-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (setf (slot-value stream 'sync-close?) (if value t nil)))

(defmethod (setf sized-stream-read-remaining) ((value integer) (stream sized-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type value t/sized-stream-max)
  (setf (slot-value stream 'read-remaining) value))

(defun make-sized-stream (stream read-size &optional sync-close)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type stream stream)
  (check-type read-size t/sized-stream-max)
  (let ((ret (make-instance 'sized-stream :stream stream)))
    (setf (slot-value ret 'read-remaining) read-size)
    (setf (slot-value ret 'sync-close?) (if sync-close t nil))
    ret))

(define-typed-fn %sized-stream-check-open ((sized-stream stream))
    (null t)
  (declare (optimize speed (debug 1) (safety 1) (space 0) (compilation-speed 0)))
  (when (slot-value stream 'closed?)
    (error 'sized-stream-closed-error :stream stream)))

(defmethod trivial-gray-streams:stream-read-byte ((stream sized-stream))
  "Reads the next byte from STREAM and returns it.  Or, if there are no more bytes
to read (either because of an end of file, or the limit was reached), this
returns :EOF."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (%sized-stream-check-open stream)

  (with-typed-slots ((t/sized-stream-max read-remaining)
                     (stream (wrapped stream)))
      stream
    (if (plusp read-remaining)
        (let ((ret (read-byte wrapped nil nil)))
          (if ret
              (prog1 ret
                (decf read-remaining))
              :eof))
        :eof)))

(defmethod trivial-gray-streams:stream-read-sequence ((stream sized-stream) (sequence vector)
                                                      (start integer) (end integer)
                                                      &key &allow-other-keys)
  "Reads bytes into SEQUENCE, then returns the first element of SEQUENCE that was
not updated (in other words, it returns the number of bytes read).  This may
return fewer than (LENGTH SEQUENCE) bytes if the end of the stream is reached,
or the set limit is reached.  SEQUENCE must be a T/UINT8-VECTOR."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type sequence t/uint8-vector)
  (%sized-stream-check-open stream)
  (with-typed-slots ((t/sized-stream-max read-remaining)
                     (stream (wrapped stream)))
      stream
    (let* ((count (min (length sequence) read-remaining))
           (ret (read-sequence sequence wrapped :end count)))
      (decf read-remaining ret)
      ret)))

(defmethod trivial-gray-streams:stream-read-char ((stream sized-stream))
  "Reads the next byte from STREAM and returns it.  Or, if there are no more bytes
to read (either because of an end of file, or the limit was reached), this
returns :EOF."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (%sized-stream-check-open stream)
  (with-typed-slots ((t/sized-stream-max read-remaining))
      stream
    (if (plusp read-remaining)
        (call-next-method)
        :eof)))

(defmethod trivial-gray-streams:stream-read-line ((stream sized-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (call-next-method stream))

(defmethod cl:open-stream-p ((stream sized-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (not (slot-value stream 'closed?)))

#-clisp
(defmethod cl:input-stream-p ((stream sized-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  t)

#-clisp
(defmethod cl:output-stream-p ((stream sized-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  nil)

(defmethod cl:stream-element-type ((stream sized-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  '(unsigned-byte 8))

(defmethod cl:close ((stream sized-stream) &key abort)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           (ignore abort))
  (when (and (slot-value stream 'sync-close?)
             (open-stream-p (slot-value stream 'stream)))
    (close (slot-value stream 'stream)))
  (setf (slot-value stream 'closed?) t))
