;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-shlog)

;;;;
;;;; Shlog (Shitty Logging)
;;;;
;;;; This is a very simple alternate logging API based on the condition system
;;;; and the behavior of HANDLER-BIND.
;;;;

(define-condition log-message (simple-condition)
  ((header
    :initarg :header
    :initform ""
    :type string
    :reader log-message-header))
  (:report (lambda (msg out)
             (if (sdm:empty-string-p (log-message-header msg))
                 (format out "~?"
                         (simple-condition-format-control msg)
                         (simple-condition-format-arguments msg))
                 (format out "[~a]: ~?"
                         (log-message-header msg)
                         (simple-condition-format-control msg)
                         (simple-condition-format-arguments msg))))))


(define-condition warning-message (simple-condition)
  ()
  (:default-initargs :header "Warning"))

(define-condition error-message (simple-condition)
  ()
  (:default-initargs :header "Error"))

(define-condition fatal-message (simple-condition)
  ()
  (:default-initargs :header "Fatal"))

(defmacro shlog* ((datum msg &key header) &rest fmt-args)
  (let ((type (etypecase datum
                (list
                 (if (and (= (length datum) 2)
                          (eq (car datum) 'quote))
                     datum
                     (error "DATUM should be a symbol or a quoted symbol")))
                (symbol
                 (list 'quote datum)))))
    `(restart-case
         ,(if header
              `(error ,type :header ,header
                            :format-control ,msg
                            :format-arguments (list ,@fmt-args))
              `(error ,type :format-control ,msg
                            :format-arguments (list ,@fmt-args)))
       (acknowledge-log-msg ()
         :report "Acknowledge the log message and acknowledge-log-msg"))))

(defmacro define-shlog-macro (name type &optional (restart 'acknowledge-log-msg))
  (let ((actual-type (etypecase type
                       (list
                        (if (and (= (length type) 2)
                                 (eq (car type) 'quote))
                            type
                            (error "TYPE should be a symbol or a quoted symbol")))
                       (symbol
                        (list 'quote type)))))
    `(defmacro ,name ((msg &key header) &rest fmt-args)
       (list 'restart-case
             (if header
                 (list 'error ,(list 'quote actual-type) :header header
                                           :format-control msg
                                           :format-arguments (list 'quote fmt-args))
                 (list 'error ,(list 'quote actual-type) :format-control msg
                                           :format-arguments (list 'quote fmt-args)))
             (list ,(list 'quote restart) ()
                   :report "Acknowledge the log message")))))

(define-shlog-macro shlog 'log-message)
(define-shlog-macro shwarn 'warning-message)
(define-shlog-macro sherror 'error-message)
(define-shlog-macro shfatal 'fatal-message)

(defmacro with-logging ((&key
                           (log-dest *standard-output*)
                           (warn-dest *error-output*)
                           (error-dest *error-output*)
                           (fatal-dest *error-output*))
                        &body forms)
  (let ((fatal-handler (gensym "FATAL-HANDLER-"))
        (error-handler (gensym "ERROR-HANDLER-"))
        (warning-handler (gensym "WARNING-HANDLER-"))
        (log-handler (gensym "LOG-HANDLER-")))
    `(flet
         ((,fatal-handler (msg)
            (declare (type fatal-message msg)
                     (optimize speed (debug 0) (space 0) (safety 0) (compilation-speed 0)))
            (format ,fatal-dest "~a" msg))

          (,error-handler (msg)
            (declare (type error-message msg)
                     (optimize speed (debug 0) (space 0) (safety 0) (compilation-speed 0)))
            (format ,error-dest "~a" msg)
            (invoke-restart 'sdm-shlog:acknowledge-log-msg))

          (,warning-handler (msg)
            (declare (type warning-message msg)
                     (optimize speed (debug 0) (space 0) (safety 0) (compilation-speed 0)))
            (format ,warn-dest "~a" msg)
            (invoke-restart 'sdm-shlog:acknowledge-log-msg))

          (,log-handler (msg)
            (declare (type log-message msg)
                     (optimize speed (debug 0) (space 0) (safety 0) (compilation-speed 0)))
            (format ,log-dest "~a" msg)
            (invoke-restart 'sdm-shlog:acknowledge-log-msg)))
       (declare (inline ,fatal-handler ,error-handler ,warning-handler ,log-handler))

       (handler-bind
           ;; Order here is not important
           ((log-message (function ,log-handler))
            (warning-message (function ,warning-handler))
            (error-message (function ,error-handler))
            (fatal-message (function ,fatal-handler)))
         ,@forms))))
