;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Sequence Utilities
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(declaim (ftype (function (function &rest list) list) mapappend)
         (inline mapappend))
(defun mapappend (func &rest lists)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (cond
    ((some #'null lists)
     ())
    (t (append (apply func (mapcar #'car lists))
               (apply #'mapappend func (mapcar #'cdr lists))))))

(define-typed-fn any? ((function predicate) ((or list vector array) sequence))
    (boolean)
  "Returns T if PREDICATE returns a truthy value for any of the items in
SEQUENCE, or NIL otherwise.  PREDICATE is a function that takes one value, an
item from SEQUENCE."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (let ((len (length sequence)))
    (when (> len 0)
      (loop for i fixnum from 0 below len
            for item = (elt sequence i)
            when (funcall predicate item)
              do (return-from any? t)))
    nil))

(declaim (ftype (function ((or list vector) (or list vector) &key (:test function)) boolean) contains-any?))
(defun contains-any? (sequence-1 sequence-2 &key (test #'eq))
  "Returns T if any of the items in SEQUENCE-1 are in SEQUENCE-2, or NIL
otherwise.  TEST is a function that takes two values, an item from SEQUENCE-1
and an item from SEQUENCE-2."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (when (and (> (length sequence-1) 0)
             (> (length sequence-2) 0))
    (loop for i fixnum from 0 below (length sequence-1)
          for item-1 = (elt sequence-1 i)
          when (find item-1 sequence-2 :test test)
            do (return-from contains-any? t))
    nil))

(define-typed-fn shuffle-sequence-inplace (((or list vector array) sequence))
    ((or list vector array))
  "Destructively randomizes the order of SEQUENCE."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (loop with len fixnum = (1- (length sequence))
        for i fixnum from len downto 1
        for j fixnum = (random (1+ i))
        for item1 = (elt sequence i)
        for item2 = (elt sequence j)
        do (setf (elt sequence i) item2)
           (setf (elt sequence j) item1)
        finally (return sequence)))

(define-typed-fn shuffle-vector-inplace ((vector sequence))
    (vector)
  "Destructively randomizes the order of SEQUENCE.  This is the same as
SHUFFLE-SEQUENCE-INPLACE, but is optimized specifically for VECTORs"
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (loop with len fixnum = (1- (length sequence))
        for i fixnum from len downto 1
        for j fixnum = (random (1+ i))
        for item1 = (aref sequence i)
        for item2 = (aref sequence j)
        do (setf (aref sequence i) item2)
           (setf (aref sequence j) item1)
        finally (return sequence)))

(define-typed-fn shuffle-sequence (((or list vector array) sequence))
    ((or list vector array) t)
  "Randomizes the order of SEQUENCE into a new sequence."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (let ((ret (copy-seq sequence)))
    (shuffle-sequence-inplace ret)))

(define-typed-fn shuffle-vector ((vector sequence))
    (vector t)
  "Randomizes the order of SEQUENCE into a new sequence.  This is the same as
SHUFFLE-SEQUENCE, but optimized specifically for VECTORs."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (let ((ret (copy-seq sequence)))
    (shuffle-vector-inplace ret)))

(declaim (ftype (function ((or list vector) &key (:count fixnum)) *) sample))
(defun sample (sequence &key (count 1))
  "Returns COUNT number of elements randomly chosen from SEQUENCE.  If COUNT is
not supplied, a single random element is returned."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (let ((len (length sequence)))
    (when sequence
      (cond
        ((or (not count) (= count 1))
         (elt sequence (random len)))

        ((> count len)
         (shuffle-sequence sequence))

        (t
         (let* ((start (random (the fixnum (- len count 1))))
                (result (copy-seq (subseq sequence
                                          (if (< start 0) 0 start)
                                          (the fixnum (+ start count))))))
           (declare (type fixnum start))

           (loop for i fixnum from count below len
                 for x fixnum = (random (1+ i))
                 when (< x count) do
                   (setf (elt result x) (elt sequence i))
                 finally
                    (progn
                      (shuffle-sequence-inplace result)
                      (return result)))))))))

(defmacro fill-sequence ((sequence idx-var &key (start 0) end) &body forms)
  "Loops from START to END (or START to (LENGTH SEQUENCE) if END is NIL),
binding IDX-VAR to each index, then calls FORMS.  The result of FORMS becomes
the value of SEQUENCE at that index.

FORMS is called within a PROGN."
  `(loop for ,idx-var from ,start below ,(if end end (list 'length sequence))
         do (setf (elt ,sequence ,idx-var)
                  (progn ,@forms))))

(defmacro doseq ((var sequence &optional result) &body forms)
  "Like DOLIST, but for any sequence type."
  (with-gensyms (len idx)
    `(loop with ,len of-type t/ufixnum = (muffling (length ,sequence))
           for ,idx of-type t/ufixnum from 0 below ,len
           for ,var = (muffling (elt ,sequence ,idx))
           do (progn ,@forms)
           finally (return ,result))))

(define-typed-fn shift (((or list vector) seq))
    (T t)
  "Non-destructively removes the first element of SEQUENCE and returns three values.
The first value will be the removed element, or NIL if SEQUENCE is empty.  The
second value is the new sequence after shifting.  The third return value is T if
the sequence was not empty, or NIL if it was.

This function accepts both LISTs and VECTORs.  If you need to work only with
LISTs, then SHIFT-LIST may be a few cycles faster."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (if (> (length seq) 0)
      (values (elt seq 0) (subseq seq 1) t)
      (values nil seq nil)))

(define-typed-fn shift-list ((list list))
    (T t)
  "Non-destructively removes the first element of LIST and returns two values.
The first value will be the removed element, or NIL if LIST is empty.  The
second value is the new sequence after shifting, or NIL if it was empty.

This is basically a version of SHIFT that is optimized to be specific for
lists."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (if list
      (values (elt list 0) (subseq list 1))
      (values nil nil)))

(define-typed-fn swap (((or list vector) sequence) (fixnum idx1 idx2))
    ((or list vector) t)
  "Destructively swaps the elements in SEQUENCE at the indices IDX1 and IDX2,
then returns SEQUENCE."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (let ((val2 (elt sequence idx2)))
    (setf (elt sequence idx2) (elt sequence idx1))
    (setf (elt sequence idx1) val2)
    sequence))

(define-typed-fn seq-last ((sequence sequence))
    (T t)
  "Returns the last element in SEQUENCE.  This is like the built-in LAST
function, but it works for all sequences.  This always returns NIL when SEQUENCE
is empty."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (let ((len 0))
    (declare (type fixnum len)
             (dynamic-extent len))
    (setf len (length sequence))
    (if (plusp len)
        (elt sequence (1- len))
        nil)))

(define-typed-fn partition ((sequence sequence) (fixnum count) &optional copy-elements)
    (sequence)
  "Splits SEQUENCE into a set of smaller sequences of the same type, each
containing up to COUNT elements.  The final element in the returned sequence may
contain fewer than COUNT elements if the length of SEQUENCE does not evenly
divide into COUNT elements.

If COPY-ELEMENTS is truthy, then COPY-SEQ is called on all of the subsequences
when creating the return value."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (when (> count (length sequence))
    (return-from partition
      (etypecase sequence
        (list
         (if copy-elements
             (list (copy-seq sequence))
             (list sequence)))

        (vector
         (make-array (length sequence) :initial-contents (if copy-elements
                                                             (copy-seq sequence)
                                                             sequence))))))

  (flet
      ((copy/copyseq (sub)
         (copy-seq sub))
       (copy/identity (sub)
         sub))

    (loop with seq-size fixnum = (length sequence)
          with ret-size fixnum = (multiple-value-bind (num rem)
                                     (truncate (length sequence) count)
                                   (+ num rem))
          with ret = (etypecase sequence
                       (list
                        (make-list ret-size))
                       (vector
                        (make-array ret-size)))
          with copy-fn function = (if copy-elements #'copy/copyseq #'copy/identity)
          for i fixnum from 0 below (length sequence) by count
          for ret-idx from 0
          do (setf (elt ret ret-idx)
                   (funcall copy-fn (subseq sequence i (min (+ i count)
                                                            (1- seq-size)))))
          finally
             (when (< ret-idx ret-size)
               ;; Copy trailing bits of the sequence into the final element
               (setf (elt ret ret-idx) (funcall copy-fn (subseq sequence (- i count)))))
             (return ret))))

(defmacro every-is-a-p (type thing)
  "Returns T if every element in THING is of type TYPE, or NIL otherwise."
  (with-gensyms (x)
    (let ((type-info (if (symbolp type)
                         (list 'quote type)
                         type)))
      `(every #'(lambda (,x) (typep ,x ,type-info)) ,thing))))
