;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-user)

(defpackage :cl-sdm-semver
  (:use :common-lisp)
  (:nicknames :sdm-semver)
  (:documentation "Semantic Version parsing and comparisons.")
  (:export
   #:semver-error
   #:t/version-core
   #:t/version-core-p
   #:t/prerelease
   #:t/prerelease-p
   #:t/build-string
   #:t/build-string-p
   #:semver
   #:semver-major
   #:semver-minor
   #:semver-revision
   #:semver-prerelease
   #:semver-build
   #:parse-semantic-version
   #:semver-compare
   #:semver=
   #:semver<
   #:semver>
   #:semver<=
   #:semver>=
   #:semver->string
   #:semver-incf-major
   #:semver-incf-minor
   #:semver-incf-revision))

(in-package :cl-sdm-semver)

(define-condition semver-error (simple-error)
  ())

(defun t/version-core-p (thing)
  (typep thing 't/version-core))

(deftype t/version-core ()
  '(integer 0 *))

(defun t/prerelease-p (thing)
  "Checks to see if THING is a T/PRERELEASE, returning T if it is, or NIL
otherwise."
  (and (listp thing)
       (every #'(lambda (x)
                  (typecase x
                    (simple-string
                     (not (every #'digit-char-p x)))
                    (t/version-core t)
                    (otherwise nil)))
              thing)))

(deftype t/prerelease ()
  "Defines a prerelease, which is a list consisting of SIMPLE-STRINGs or T/VERSION-COREs."
  '(satisfies t/prerelease-p))

(defun t/build-string-p (thing)
  (typecase thing
    (null t)
    (string
     (not (zerop (length thing))))
    (otherwise nil)))

(deftype t/build-string ()
  '(satisfies t/build-string-p))

(defclass semver ()
  ((major
    :initarg :major
    :initform 0
    :type t/version-core
    :reader semver-major)

   (minor
    :initarg :minor
    :initform 0
    :type t/version-core
    :reader semver-minor)

   (revision
    :initarg :revision
    :initform 0
    :type t/version-core
    :reader semver-revision)

   (prerelease
    :initarg :prerelease
    :initform ()
    :type t/prerelease
    :reader semver-prerelease)

   (build
    :initform nil
    :type t/build-string
    :reader semver-build))
  (:documentation "A record that stores a semantic version."))

(defmethod initialize-instance :after ((ver semver) &key &allow-other-keys)
  (with-slots (major minor revision prerelease build)
      ver
    (check-type major      t/version-core)
    (check-type minor      t/version-core)
    (check-type revision   t/version-core)
    (check-type prerelease t/prerelease)
    (check-type build      t/build-string)))

(defmethod print-object ((obj semver) out)
  (print-unreadable-object (obj out :type t)
    (format out "Version: ~a" (semver->string obj))))

(defmethod (setf semver-major) ((value integer) (ver semver))
  (check-type value t/version-core)
  (setf (slot-value ver 'major) value))

(defmethod (setf semver-minor) ((value integer) (ver semver))
  (check-type value t/version-core)
  (setf (slot-value ver 'minor) value))

(defmethod (setf semver-revision) ((value integer) (ver semver))
  (check-type value t/version-core)
  (setf (slot-value ver 'revision) value))

(defmethod (setf semver-prerelease) (value (ver semver))
  (check-type value t/prerelease)
  (setf (slot-value ver 'prerelease) value))

(defmethod (setf semver-build) ((value string) (ver semver))
  (setf (slot-value ver 'build) value))

(defmethod (setf semver-build) ((value null) (ver semver))
  (setf (slot-value ver 'build) nil))

(defun parse-semantic-version (string)
  "Parses a new SEMVER out of STRING.  Raises a SEMVER-ERROR if it encounters an
error."
  (declare (optimize speed (debug 1) (safety 1)))
  (check-type string string)

  (labels
      (;; A non-digit character, according to the semver spec.
       (non-digit-p (c)
         (or (char= c #\-)
             (and (char>= c #\a)
                  (char<= c #\z))
             (and (char>= c #\A)
                  (char<= c #\Z))))

       ;; Reads a single prerelease identifier.  Returns the new identifier as
       ;; the first value, and then :DOT if it encounters a dot at the end,
       ;; :BUILD if it encounters a plus at the end, or :DONE if there is
       ;; nothing left to parse.
       (read-pre-release-ident (str)
         (let ((out (make-string-output-stream)))
           (loop for c = (read-char str nil nil)
                 while c do
                   (cond
                     ((or (non-digit-p c)
                          (digit-char-p c))
                      (write-char c out))
                     ((char= c #\.)
                      (return-from read-pre-release-ident (values (get-output-stream-string out) :dot)))
                     ((char= c #\+)
                      (return-from read-pre-release-ident (values (get-output-stream-string out) :build)))
                     (t
                      (error 'semver-error
                             :format-control "Junk in semantic version at position ~a: ~a"
                             :format-arguments (list (file-position str) c)))))
           (values (get-output-stream-string out) :done)))

       ;; Reads a prerelease.  Returns the list of prerelease identifiers as the
       ;; first value, and either :BUILD if a build string should be read next,
       ;; or :DONE if there is nothing left to parse.
       (read-pre-release (str)
         (let ((ret ())
               (next-state nil))
           (loop do
             (multiple-value-bind (ident state)
                 (read-pre-release-ident str)
               (when (string= ident "")
                 (error 'semver-error
                        :format-control "Empty prerelease identifier"))
               (setf ret (append ret (list (block parse-number
                                             (handler-bind
                                                 ((parse-error (lambda (err)
                                                                 (declare (ignore err))
                                                                 (return-from parse-number ident))))
                                               (prog1 (parse-integer ident)
                                                 (when (and (plusp (length ident))
                                                            (char= (elt ident 0) #\0))
                                                   (error 'semver-error
                                                          :format-control "Bad prerelease identifier: ~s"
                                                          :format-arguments (list ident)))))))))
               (ecase state
                 (:dot nil)
                 ((:build :done)
                  (setf next-state state)
                  (loop-finish)))))

           (cond
             ((zerop (length ret))
              (error 'semver-error
                     :format-control "Empty prerelease"))
             ((equal (car ret) "-")
              (error 'semver-error
                     :format-control "Bad prerelease")))
           (values ret next-state)))

       ;; Reads a build string and returns it.
       (read-build (str)
         (let ((ret (with-output-to-string (out)
                      (loop for c = (read-char str nil nil)
                            while c do
                              (cond
                                ((or (non-digit-p c)
                                     (digit-char-p c)
                                     (char= c #\.))
                                 (write-char c out))
                                (t
                                 (error 'semver-error
                                        :format-control "Junk in semantic version at position ~a: ~a"
                                        :format-arguments (list (file-position str) c))))))))
           (cond
             ((zerop (length ret))
              (error 'semver-error
                     :format-control "Empty build string."))
             ((not (or (digit-char-p (elt ret 0))
                       (non-digit-p (elt ret 0))))
              (error 'semver-error
                     :format-control "Invalid build string: ~s"
                     :format-arguments (list ret))))
           ret))

       ;; Reads a single numeric identifier.
       (read-numeric-ident (str)
         (let ((ret (with-output-to-string (out)
                      (loop for c = (read-char str nil nil)
                            while c do
                              (case c
                                ((#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)
                                 (write-char c out))
                                (otherwise
                                 (unread-char c str)
                                 (loop-finish)))))))
           (when (and (> (length ret) 1)
                      (char= (elt ret 0) #\0))
             (error 'semver-error
                    :format-control "Bad numeric identifier: cannot start with 0"))
           ret))

       ;; Reads the major version and returns it.
       (read-major (str)
         (handler-case
             (prog1 (parse-integer (read-numeric-ident str))
               (unless (equal (read-char str nil nil) #\.)
                 (error 'semver-error
                        :format-control "Junk in semantic version at position ~a"
                        :format-arguments (list (file-position str)))))
           (parse-error ()
             (error 'semver-error :format-control "Junk in semantic version major version"))))

       ;; Reads the minor version and returns it.
       (read-minor (str)
         (handler-case
             (prog1 (parse-integer (read-numeric-ident str))
               (unless (equal (read-char str nil nil) #\.)
                 (error 'semver-error
                        :format-control "Junk in semantic version at position ~a"
                        :format-arguments (list (file-position str)))))
           (parse-error ()
             (error 'semver-error :format-control "Junk in semantic version minor version"))))

       ;; Reads the revision.  Returns the revision as the first value, and then
       ;; either :PRE to indicate a prerelease is next, :BUILD to indicate a
       ;; build string is next, or :DONE if there is nothing left to parse.
       (read-rev (str)
         (handler-case
             (let ((val (parse-integer (read-numeric-ident str)))
                   (c (read-char str nil nil)))
               (values val (cond
                             ((null c)
                              :done)
                             (t
                              (case c
                                (#\- :pre)
                                (#\+ :build)
                                (otherwise
                                 (error 'semver-error
                                        :format-control "Junk in semantic version at position ~a"
                                        :format-arguments (list (file-position str)))))))))
           (parse-error ()
             (error 'semver-error :format-control "Junk in semantic version revision")))))

    ;; Store things in some variables for now.
    (let ((major nil)
          (minor nil)
          (rev   nil)
          (pre   nil)
          (build nil)
          (state :major))

      ;; Begin parsing
      (with-input-from-string (in string)
        (loop do
          (ecase state
            (:major
             (setf major (read-major in))
             (setf state :minor))

            (:minor
             (setf minor (read-minor in))
             (setf state :rev))

            (:rev
             (multiple-value-setq (rev state)
               (read-rev in)))

            (:pre
             (multiple-value-setq (pre state)
               (read-pre-release in)))

            (:build
             (setf build (read-build in))
             (loop-finish))

            (:done
             (loop-finish)))))

      ;; Finished, return the new SEMVER instance.
      (make-instance 'semver :major major :minor minor :revision rev :prerelease pre :build build))))

(defun semver-compare (ver1 ver2)
  "Compares VER1 to VER2.  If VER1 is less than VER2, this returns -1.  If VER1
is greater than VER2, this returns 1.  If the two are equal, this returns 0."
  (check-type ver1 semver)
  (check-type ver2 semver)

  (labels
      (;; Compares two T/PRERELEASE instances.  Same semantics as the outer
       ;; function: -1, 0, or 1.
       (semver-compare-prereleases (pre1 pre2)
         (loop for i fixnum from 0 ;; Keep track of how many we compare.
               for ident1 in pre1
               for ident2 in pre2
               do (etypecase ident1
                    (integer
                     ;; Integers are compared as numbers.
                     (if (integerp ident2)
                         (cond
                           ((< ident1 ident2)
                            (return-from semver-compare-prereleases -1))
                           ((> ident1 ident2)
                            (return-from semver-compare-prereleases 1)))
                         (return-from semver-compare-prereleases -1)))

                    (string
                     ;; Strings are compared lexically.
                     (if (integerp ident2)
                         (return-from semver-compare-prereleases 1)
                         (cond
                           ((string< ident1 ident2)
                            (return-from semver-compare-prereleases -1))
                           ((string> ident1 ident2)
                            (return-from semver-compare-prereleases 1))))))
               finally
                  (return
                    (if (/= (length pre1) (length pre2))
                        ;; When two prereleases have had equal identifiers
                        ;; up until now, but one is longer than the other,
                        ;; then the longer one is "greater".
                        (cond
                          ((= i (length pre1))
                           -1)
                          ((= i (length pre2))
                           1)
                          (t ;; Shouldn't happen?
                           0))

                        ;; The prereleases are equal
                        0)))))
    (declare (ftype (function (list list) (integer -1 1)) semver-compare-prereleases))

    ;; Check major version first
    (if (< (semver-major ver1) (semver-major ver2))
        (return-from semver-compare -1)
        (when (> (semver-major ver1) (semver-major ver2))
          (return-from semver-compare 1)))

    ;; Check minor version
    (if (< (semver-minor ver1) (semver-minor ver2))
        (return-from semver-compare -1)
        (when (> (semver-minor ver1) (semver-minor ver2))
          (return-from semver-compare 1)))

    ;; Check revision
    (if (< (semver-revision ver1) (semver-revision ver2))
        (return-from semver-compare -1)
        (when (> (semver-revision ver1) (semver-revision ver2))
          (return-from semver-compare 1)))

    ;; Check prerelease, if any.
    (cond
      ((semver-prerelease ver1)
       (if (semver-prerelease ver2)
           (semver-compare-prereleases (semver-prerelease ver1) (semver-prerelease ver2))
           -1))
      ((semver-prerelease ver2)
       ;; We already know that there's no prerelease on ver1
       1)

      (t ;; The two are equal.
       0))))

(defun semver= (ver1 ver2)
  "Returns T if VER1 is equal to VER2."
  (zerop (semver-compare ver1 ver2)))

(defun semver< (ver1 ver2)
  "Returns T if VER1 is strictly less than VER2."
  (< (semver-compare ver1 ver2) 0))

(defun semver> (ver1 ver2)
  "Returns T if VER1 is strictly greater than VER2."
  (> (semver-compare ver1 ver2) 0))

(defun semver<= (ver1 ver2)
  "Returns T if VER1 is less than or equal to VER2."
  (<= (semver-compare ver1 ver2) 0))

(defun semver>= (ver1 ver2)
  "Returns T if VER1 is greater than or equal to VER2."
  (>= (semver-compare ver1 ver2) 0))

(defun semver->string (ver)
  "Converts VER to a string."
  (with-output-to-string (out)
    (format out "~a.~a.~a" (semver-major ver) (semver-minor ver) (semver-revision ver))
    (when (semver-prerelease ver)
      (write-char #\- out)
      (loop with len fixnum = (length (semver-prerelease ver))
            for i fixnum from 0
            for pre in (semver-prerelease ver)
            do (format out "~a" pre)
               (unless (= i (1- len))
                 (write-char #\. out))))

    (when (semver-build ver)
      (write-char #\+ out)
      (write-string (semver-build ver) out))))

(defun semver-incf-major (ver)
  "Creates a new SEMVER that has the same minor and revision as VER, but a major
that is increased by 1.  The new SEMVER has no prerelease or build string."
  (make-instance 'semver
                 :major (1+ (semver-major ver))
                 :minor (semver-minor ver)
                 :revision (semver-revision ver)))

(defun semver-incf-minor (ver)
  "Creates a new SEMVER that has the same major and revision as VER, but a minor
that is increased by 1.  The new SEMVER has no prerelease or build string."
  (make-instance 'semver
                 :major (semver-major ver)
                 :minor (1+ (semver-minor ver))
                 :revision (semver-revision ver)))

(defun semver-incf-revision (ver)
  "Creates a new SEMVER that has the same major and minor as VER, but a revision
that is increased by 1.  The new SEMVER has no prerelease or build string."
  (make-instance 'semver
                 :major (semver-major ver)
                 :minor (semver-minor ver)
                 :revision (1+ (semver-revision ver))))
