;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-rsconf)

(defgeneric rsconf-write (data dest &key &allow-other-keys)
  (:documentation "Writes parsed RSConf data in DATA to DEST."))

(defgeneric rsconf-write-to-string (data &optional writer)
  (:documentation "Writes parsed RSConf data in DATA to a new string.  Options may be specified
using using WRITER."))

(defclass rsconf-writer ()
  ((indent-size
    :initarg :indent-size
    :initform 2
    :type t/ufixnum
    :reader rsconf-writer-indent-size
    :documentation "How many spaces to increase the indent by for each new level.")

   (always-quote-keys?
    :initarg :always-quote-keys
    :initform nil
    :type boolean
    :reader rsconf-writer-always-quote-keys-p
    :documentation "When T, then key names are always quoted even when they don't need to be.")

   (always-use-scientific-notation?
    :initarg :always-use-scientific-notation
    :initform nil
    :type boolean
    :reader rsconf-writer-always-use-scientific-notation-p
    :documentation "When T, then float values are always printed in scientific notation.")

   (explicit-root-object?
    :initarg :explicit-root-object
    :initform nil
    :type boolean
    :reader rsconf-writer-explicit-root-object-p
    :documentation "When T, then { and } braces will explicitly be written in the root.")

   (comma-after-values?
    :initarg :comma-after-values
    :initform nil
    :type boolean
    :reader rsconf-writer-comma-after-values-p
    :documentation "When T, then a comma will be written after each value.")

   (extra-newline-between-toplevel-keys?
    :initarg :extra-newline-between-toplevel-keys
    :initform nil
    :type boolean
    :reader rsconf-writer-extra-newline-between-toplevel-keys-p
    :documentation "When T, then an extra newline will be printed between each toplevel key.")

   (stream
    :type stream)

   (indent
    :initform 0
    :type t/ufixnum))
  (:documentation "Used to write RSConf data.  This stores various options for writing the data,
and also stores various bits of internal state used while writing."))

(defmethod (setf rsconf-writer-indent-size) ((value integer) (writer rsconf-writer))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type value t/ufixnum)
  (setf (slot-value writer 'indent-size) value))

(defmethod (setf rsconf-writer-always-quote-keys-p) (value (writer rsconf-writer))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (setf (slot-value writer 'always-quote-keys?) (if value t nil)))

(defmethod (setf rsconf-writer-always-use-scientific-notation-p) (value (writer rsconf-writer))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (setf (slot-value writer 'always-use-scientific-notation?) (if value t nil)))

(defmethod (setf rsconf-writer-explicit-root-object-p) (value (writer rsconf-writer))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (setf (slot-value writer 'explicit-root-object?) (if value t nil)))

(defmethod (setf rsconf-writer-comma-after-values-p) (value (writer rsconf-writer))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (setf (slot-value writer 'comma-after-values?) (if value t nil)))

(defmethod (setf rsconf-writer-extra-newline-between-toplevel-keys-p) (value (writer rsconf-writer))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (setf (slot-value writer 'extra-newline-between-toplevel-keys?) (if value t nil)))

(define-typed-fn write-indent ((stream dest) (t/ufixnum size))
    (null t)
  "Writes an indent to DEST of SIZE spaces."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (dotimes (i size)
    (write-char #\Space dest)))

(define-typed-fn write-quoted-string ((stream dest) (string str))
    (null t)
  "Writes a double-quote, then writes STR, then writes another double-quote.  Any
double-quotes in STR are escaped according to RSConf's specs."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (write-char #\" dest)
  (write-string (string-replace str "\"" "\\\"") dest)
  (write-char #\" dest)
  nil)

(define-typed-fn write-unquoted-string ((stream dest) (string str))
    (null t)
  "Writes STR without any quotes.  Nothing is escaped in STR."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (write-string str dest)
  nil)

(define-typed-fn write-value ((t/value val) (rsconf-writer writer))
    (null)
  "Writes VAL, followed by a newline if needed.  The cursor will be in column 0
when this returns."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((stream stream)
                     (t/ufixnum indent indent-size)
                     (boolean comma-after-values?))
      writer
    (etypecase val
      (hash-table
       (write-object val writer))

      (t/string
       (write-quoted-string stream val)
       (when comma-after-values?
         (write-char #\, stream)))

      (float
       (if (slot-value writer 'always-use-scientific-notation?)
           (format stream "~e" (coerce val 'double-float))
           (format stream "~f" (coerce val 'double-float)))
       (when comma-after-values?
         (write-char #\, stream)))

      (integer
       (format stream "~d" val)
       (when comma-after-values?
         (write-char #\, stream)))

      (t/array
       (write-array val writer))

      (t/bool
       (ecase val
         (:true (write-string "true" stream))
         (:false (write-string "false" stream)))
       (when comma-after-values?
         (write-char #\, stream)))

      (t/null
       (write-string "nil" stream)
       (when comma-after-values?
         (write-char #\, stream))))
    (fresh-line stream))
  nil)

(define-typed-fn write-array ((t/array val) (rsconf-writer writer))
    (null)
  "Writes a [, then writes each value on new lines, indenting as it goes.  This
then writes ] and a newline.  The cursor will be in column 0 when this returns."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((stream stream)
                     (t/ufixnum indent indent-size)
                     (boolean comma-after-values?))
      writer
    (format stream "[~%")
    (incf indent indent-size)
    (muffling
      (doseq (element val)
        (unless (typep val 't/value)
          (rsconf-error () "Cannot serialize value of type ~a" (type-of val)))
        (write-indent stream indent)
        (write-value element writer)))
    (setf indent (max 0 (- indent indent-size)))
    (write-indent stream indent)
    (if comma-after-values?
        (format stream "],~%")
        (format stream "]~%")))
  nil)

(define-typed-fn write-object-pairs ((hash-table data) (rsconf-writer writer) &optional extra-newline?)
    (null)
  "Writes the key/value pairs in DATA."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((stream stream)
                     (t/ufixnum indent indent-size)
                     (boolean always-quote-keys? explicit-root-object?))
      writer
    (loop for key being the hash-keys in data
            using (hash-value val)
          do (unless (stringp key)
               (rsconf-error () "All keys must be strings"))
             (unless (typep val 't/value)
               (rsconf-error () "Cannot serialize value of type ~a" (type-of val)))

             (write-indent stream indent)

             ;; Do we need quotes?
             (if (or always-quote-keys?
                     (any? #'(lambda (x)
                               (or (char= x #\:)
                                   (char= x #\[)
                                   (char= x #\])
                                   (char= x #\{)
                                   (char= x #\})
                                   (char= x #\")
                                   (whitespace-char-p x)))
                           key))
                 (write-quoted-string stream key)
                 (write-unquoted-string stream key))

             (write-string ": " stream)
             (write-value val writer)

             (when extra-newline?
               (write-char #\Newline stream))))
  nil)

(define-typed-fn write-object ((hash-table obj) (rsconf-writer writer))
    (null)
  "Writes a {, then writes each pair in OBJ on its own line, indenting as it goes.
This then writes } followed by a newline.  The cursor will be in column 0 when
this returns."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((stream stream)
                     (t/ufixnum indent indent-size)
                     (boolean always-quote-keys? comma-after-values?))
      writer
    (format stream "{~%")
    (incf indent indent-size)
    (write-object-pairs obj writer)
    (setf indent (max 0 (- indent indent-size)))
    (write-indent stream indent)

    (if comma-after-values?
        (format stream "},~%")
        (format stream "}~%")))
  nil)

(defmethod rsconf-write ((data hash-table) (dest rsconf-writer) &key &allow-other-keys)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((stream stream)
                     (t/ufixnum indent indent-size)
                     (boolean always-quote-keys? explicit-root-object?))
      dest
    (unless (or (eql (stream-external-format stream) :utf-8)
                (null (stream-external-format stream))) ;; This may be null if it's a string stream.
      (rsconf-encoding-error () "Unsupported encoding: ~a" (stream-external-format stream)))

    (when explicit-root-object?
      (format stream "{~%")
      (incf indent indent-size))

    (write-object-pairs data dest (slot-value dest 'extra-newline-between-toplevel-keys?))

    (when explicit-root-object?
      (fresh-line stream)
      (format stream "}"))

    (fresh-line stream))
  t)

(defmethod rsconf-write ((data hash-table) (dest stream) &key &allow-other-keys)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (let ((state (make-instance 'rsconf-writer)))
    (setf (slot-value state 'stream) dest)
    (rsconf-write data state)))

(defmethod rsconf-write ((data hash-table) (dest pathname) &key (if-exists :error) (if-does-not-exist :create)
                         &allow-other-keys)
  "Writes DATA to a file.  The IF-EXISTS and IF-DOES-NOT-EXIST are the same as for
OPEN and default to :ERROR and :CREATE, respectively."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-open-file (out dest :if-exists if-exists
                            :if-does-not-exist if-does-not-exist
                            :external-format :utf-8)
    (rsconf-write data out)))

(defmethod rsconf-write ((data hash-table) (dest string) &key (if-exists :error) (if-does-not-exist :create)
                         &allow-other-keys)
  "Writes DATA to a file.  The IF-EXISTS and IF-DOES-NOT-EXIST are the same as for
OPEN and default to :ERROR and :CREATE, respectively."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (rsconf-write data (uiop:parse-native-namestring dest) :if-exists if-exists :if-does-not-exist if-does-not-exist))

(defmethod rsconf-write-to-string ((data hash-table) &optional writer)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-output-to-string (out)
    (if writer
        (if (typep writer 'rsconf-writer)
            (if (slot-boundp writer 'stream)
                (rsconf-error () "WRITER seems to already be used for a different stream.")
                (progn
                  (setf (slot-value writer 'stream) out)
                  (rsconf-write data writer)))
            (rsconf-error () "WRITER must be NIL or a RSCONF-WRITER"))
        (rsconf-write data out))))
