;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-rsconf)

(define-simple-error rsconf-error nil ()
  ()
  (:documentation "Base class for all RSConf errors."))

(define-simple-error rsconf-encoding-error rsconf-error ()
  ()
  (:documentation "Represents a stream encoding error."))

(define-simple-error rsconf-bom-error rsconf-encoding-error ()
  ()
  (:documentation "Raised when a Byte-Order-Mark is detected."))

(defining-consts
  (+whitespace-except-newline+
   (list #\Space #\Tab #\Zero_Width_Space #\No-Break_Space #\Figure_Space
         #-ccl #\Ogham_Space_Mark
         #-ccl #\En_Quad
         #-ccl #\Em_Quad
         #-ccl #\En_Space
         #-ccl #\Em_Space
         #-ccl #\Three-Per-Em_Space
         #-ccl #\Four-Per-Em_Space
         #-ccl #\Six-Per-Em_Space
         #-ccl #\Punctuation_Space
         #-ccl #\Thin_Space
         #-ccl #\Hair_Space
         #-ccl #\Narrow_No-Break_Space
         #-ccl #\Medium_Mathematical_Space
         #-ccl #\Ideographic_Space)))

(defmacro rsconf-parse-error/return-found (parser)
  "Shorthand for raising an error about finding #\Return"
  `(rsconf-parse-error ,parser "Illegal character: 0x~x (\\r, Return)"
     #.(char-code #\Return)))

(deftype t/object ()
  'hash-table)

(deftype t/number ()
  '(or integer double-float))

(deftype t/bool ()
  '(member :true :false))

(deftype t/null ()
  '(eql :null))

(deftype t/string ()
  'string)

(deftype t/array ()
  '(vector t *))

(deftype t/value ()
  '(or t/object t/number t/bool t/null t/string t/array))

(define-typed-fn whitespace-char-but-not-newline-p ((character char))
    (boolean t)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (if (find char +whitespace-except-newline+ :test #'char=) t nil))
