;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Pseudo-Enums
;;;;
;;;; Pseudo-Enums are a way to map keywords to/from scalar values.
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-condition pseudo-enum-error (simple-condition)
  ())

(defmacro %pseudo-enum-define-type (type-name values &optional docstring)
  "Expands into the DEFTYPE for TYPE-NAME."
  (let ((tname (intern type-name)))
    `(progn
       (deftype ,tname ()
         ,(or docstring (format nil "~a" tname))
         '(member ,@(loop for val in values collecting (car val)))))))

(defmacro %pseudo-enum-define-predicate (type-name)
  "Expands into the DEFUN for the VALID-x-P function."
  (let ((thing (gensym "THING"))
        (fname (intern (string-upcase (format nil "VALID-~a-P" type-name)))))
    `(progn
       (declaim (ftype (function (T) boolean) ,fname)
                (inline ,fname))
       (defun ,fname (,thing)
         (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0))
                  #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
         (typep ,thing ',(intern type-name))))))

(defmacro %pseudo-enum-define-from-mapper (type-name fn-name values inline?)
  "Expands into a function that maps a pseudo-enum to its associated value (e.g. powerup-type->int)."
  (let ((thing (gensym "THING")))
    `(progn
       ,(when inline?
          `(declaim (inline ,(intern fn-name))))

       (defun ,(intern fn-name) (,thing)
         (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0))
                  #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
         (case ,thing
           ,@(loop for val in values
                   collect (list (car val)
                                 (if (listp (cdr val))
                                     (cadr val)
                                     (cdr val))))
           (otherwise (error 'pseudo-enum-error
                             :format-control ,(format nil "Not a ~a: ~~a" type-name)
                             :format-arguments (list ,thing))))))))

(defmacro %pseudo-enum-define-from?-mapper (type-name fn-name values inline?)
  "Expands into a function that maps a pseudo-enum to its associated value, or nil (e.g. powerup-type->int?)"
  (declare (ignorable type-name))
  (let ((thing (gensym "THING")))
    `(progn
       ,(when inline?
          `(declaim (inline ,(intern fn-name))))
       (defun ,(intern fn-name) (,thing)
         (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0))
                  #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
         (case ,thing
           ,@(loop for val in values
                   collect (list (car val)
                                 (if (listp (cdr val))
                                     (cadr val)
                                     (cdr val))))
           (otherwise nil))))))

(defmacro %pseudo-enum-define-to-mapper (type-name mapping-type fn-name values inline?)
  "Defines a function that maps from a value to a pseudo-enum (e.g. int->powerup-type)."
  (let ((thing (gensym "THING")))
    `(progn
       ,(when inline?
          `(declaim (inline ,(intern fn-name))))
       (defun ,(intern fn-name) (,thing)
         (declare (type ,mapping-type ,thing)
                  (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0))
                  #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

         ;; If MAPPING-TYPE is an integer, then we can use CASE
         ,(ecase mapping-type
            ;; NOTE: this is temporarily disabled due to SBCL
            ;; complaining about the jump tables that get produced for
            ;; some pseudo-enums.
            ;;
            (integer
             `(cond
                ,@(loop for val in values
                        collect (list (list '= thing (if (listp (cdr val))
                                                         (cadr val)
                                                         (cdr val)))
                                      (car val)))
                (t (error 'pseudo-enum-error
                          :format-control ,(format nil "Cannot convert ~~a to a ~a" type-name)
                          :format-arguments (list ,thing)))))

            ;; Use a COND
            (string
             `(cond
                ,@(loop for val in values
                        collect (list (list 'string= ;; (if (eq mapping-type 'string) 'string= '=)
                                            thing
                                            (if (listp (cdr val))
                                                (cadr val)
                                                (cdr val)))
                                      (car val)))
                (t (error 'pseudo-enum-error
                          :format-control ,(format nil "Cannot convert ~~a to a ~a" type-name)
                          :format-arguments (list ,thing))))))))))


(defmacro %pseudo-enum-define-to?-mapper (mapping-type fn-name values inline?)
  "Defines a function that maps from a value to a pseudo-enum or nil (e.g. int->powerup-type?)."
  (let ((thing (gensym "THING")))
    `(progn
       ,(when inline?
          `(declaim (inline ,(intern fn-name))))
       (defun ,(intern fn-name) (,thing)
         (declare (type ,mapping-type ,thing)
                  (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0))
                  #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

         ;; If MAPPING-TYPE is an integer, then we can use CASE
         ,(ecase mapping-type
            ;; NOTE: this is temporarily disabled due to SBCL
            ;; complaining about the jump tables that get produced for
            ;; some pseudo-enums.
            ;;
            (integer
             ;; Use a CASE
             `(cond
                ,@(loop for val in values
                        collect (list (list '= thing (if (listp (cdr val))
                                                         (cadr val)
                                                         (cdr val)))
                                      (car val)))
                (t nil)))

            ;; Use a COND
            (string
             `(cond
                ,@(loop for val in values
                        collect (list (list 'string=  ;;(if (eq mapping-type 'string) 'string= '=)
                                            thing
                                            (if (listp (cdr val))
                                                (cadr val)
                                                (cdr val)))
                                      (car val)))
                (t nil))))))))

(defmacro define-pseudo-enum (name mapping-type &body values)
  "Defines a 'pseudo-enum', where a keyword is mapped to a scalar type.  Only
certain types are supported.  This sets up a type named NAME, and five
functions:

 * scalar-type->type-name
 * type-name->scalar-type
 * type-name-MEMBERS
 * type-name-VALUES
 * VALID-type-name-P

VALUES is one or more CONS cells or LISTs where the CAR is the keyword, and the
CDR (or CADR) is the value that keyword should map to.  The very first item in
VALUES can be :INLINE-ALL, in which case all of the conversion functions are
declared INLINE.

VALUES can also contain a single string as its very first item.  If this is the
case, that string becomes the type's documentation string.

The scalar types that are supported are:
* string
* integer
* float"
  (let* ((docstring (if (stringp (car values))
                        (prog1 (car values)
                          (setf values (cdr values)))
                        nil))
         (type-name (string-upcase (string name)))
         (mtype-name (ecase mapping-type
                       (string  "STR")
                       (integer "INT")
                       (float   "FLOAT")))
         (fname-to    (format nil "~a->~a" mtype-name type-name))   ;; Mapping type -> pseudo-enum
         (fname-to?   (format nil "~a->~a?" mtype-name type-name))  ;; Mapping type -> pseudo-enum (or nil)
         (fname-from  (format nil "~a->~a" type-name mtype-name))   ;; pseudo-enum -> Mapping type
         (fname-from? (format nil "~a->~a?" type-name mtype-name))  ;; pseudo-enum -> Mapping type (or nil)
         (inline? (if (eq  (car values) :inline-all)
                      (prog1 t
                        (setf values (cdr values)))
                      nil)))
    `(progn
       ;; Define the type
       (%pseudo-enum-define-type ,type-name ,values ,docstring)

       ;; Define the predicate
       (%pseudo-enum-define-predicate ,type-name)

       ;; Define the mapping functions
       (%pseudo-enum-define-from-mapper ,type-name ,fname-from ,values ,inline?)
       (%pseudo-enum-define-from?-mapper ,type-name ,fname-from? ,values ,inline?)
       (%pseudo-enum-define-to-mapper ,type-name ,mapping-type ,fname-to ,values ,inline?)
       (%pseudo-enum-define-to?-mapper ,mapping-type ,fname-to? ,values ,inline?)

       ;; Define the function to get the members of the pseudo-enum
       (defun ,(intern (string-upcase (format nil "~a-MEMBERS" type-name))) ()
         (loop for val in ,(list 'quote values) collect (car val)))

       ;; Define the function to get the values of the pseudo-enum
       (defun ,(intern (string-upcase (format nil "~a-VALUES" type-name))) ()
         (loop for val in ,(list 'quote values) collect (if (listp (cdr val))
                                                            (cadr val)
                                                            (cdr val)))))))
