;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;
;;;; This code is based on the PARSE-FLOAT package by Sumant Oemrawsingh, which
;;;; was released into the public domain.
;;;;
;;;; https://github.com/soemraws/parse-float
;;;;

(define-condition float-parse-error (simple-error parse-error)
  ())

(deftype t/valid-radix ()
  "A valid Common Lisp radix."
  '(integer 2 36))

(defmacro get-char (string index)
  `(locally
       (declare #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
     (char ,string ,index)))

(declaim (type simple-base-string +whitespace-characters+))
(defconst +whitespace-characters+
  (coerce '(#\Space #\Tab #\Return #\Newline #\Linefeed #\Page) 'simple-base-string)
  :documentation "List of whitespace characters")

(define-typed-fn parse-float/sign-char-p ((character char))
    (boolean t)
  "Predicate for testing if CHARACTER is a sign character (i.e. #\+ or #\-)."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (or (char= #\+ char)
      (char= #\- char)))

(define-typed-fn parse-float/whitespace-char-p ((character char))
    (boolean t)
  "Predicate for testing if CHARACTER is a whitespace character."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (if (some #'(lambda (c)
                (char= c char))
            +whitespace-characters+)
      t
      nil))


(declaim (ftype (function (string &key (:start fixnum) (:end (or null fixnum))) fixnum) parse-float/skip-whitespaces)
         (inline skip-whitespaces))
(defun parse-float/skip-whitespaces (string &key (start 0) end)
  "For the substring in STRING delimited by START and END, skip all the
whitespace at the beginning and return the index of the first non-whitespace
character, or END if no non-whitespace characters were found."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let ((real-end (or end (length string))))
    (loop for index fixnum from start upto real-end
          while (and (< index end)
                     (parse-float/whitespace-char-p (get-char string index)))
          finally (return index))))

(declaim (ftype (function (string &key (:start fixnum) (:end fixnum) (:radix t/valid-radix) (:allow-sign T))
                          (values (or null integer) fixnum boolean))
                parse-float/parse-integer-only))
(defun parse-float/parse-integer-only (string &key (start 0) (end (length string))
                                    (radix 10) (allow-sign t))
  "Parse an integer from a string, without skipping whitespaces.
Returns three values: the integer, the position in the string that ended the
parsing, and a boolean which is T if the parsing ended due to a whitespace or
end of the string, and NIL otherwise.  If allow-sign is NIL (T by default), also
signs are not allowed in the string (i.e. cannot start with #\+ or #\-)."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let ((index start))
    (if (>= index end)
        (values nil index t)
        (let ((char (get-char string index)))
          (if (or (and (not allow-sign) (parse-float/sign-char-p char))
                  (parse-float/whitespace-char-p char))
              (values nil index t)
              (multiple-value-bind (value position)
                  (parse-integer string
                                 :start index
                                 :end end
                                 :junk-allowed t
                                 :radix radix)
                (if (or (= position end)
                        (parse-float/whitespace-char-p (get-char string position)))
                    (values value position t)
                    (values value position nil))))))))

(declaim (ftype (function (string &key (:start fixnum) (:end fixnum) (:radix t/valid-radix)
                                  (:junk-allowed T) (:decimal-character character)
                                  (:exponent-character character) (:type symbol))
                          (values float (or null fixnum)))
                parse-float))
(defun parse-float (string &key (start 0) (end (length string))
                             (radix 10) (junk-allowed nil)
                             (decimal-character #\.) (exponent-character #\e)
                             (type *READ-DEFAULT-FLOAT-FORMAT*))
  "Similar to PARSE-INTEGER, but parses a floating point value and returns the
value as the specified TYPE (by default *READ-DEFAULT-FLOAT-FORMAT*). The
DECIMAL-CHARACTER (by default #\.)  specifies the separator between the integer
and decimal parts, and the EXPONENT-CHARACTER (by default #\e, case insensitive)
specifies the character before the exponent. Note that the exponent is only
parsed if RADIX is 10."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let* (;; sign of the float
         (sign 1)

         ;; number of decimal digits
         (digits 0)

         ;; index walking through string
         (index (parse-float/skip-whitespaces string :start start :end end))

         ;; parts of the value
         (integer-part nil)
         (decimal-part 0)
         (exponent-part 0)

         ;; final result
         (result nil))
    (declare (type fixnum index digits sign decimal-part exponent-part)
             (type (or null fixnum) integer-part)
             (type (or null float) result))

    (labels ((parse-sign ()
               (if (= index end)
                   #'parse-finish
                   (let ((char (get-char string index)))
                     (cond
                       ((char= #\- char)
                        (if (>= (incf index) end)
                            #'parse-finish
                            (progn
                              (setf sign -1)
                              #'parse-integer-part)))
                       ((char= #\+ char)
                        (if (>= (incf index) end)
                            #'parse-finish
                            #'parse-integer-part))
                       (t #'parse-integer-part)))))

             (parse-integer-part ()
               (multiple-value-bind (value position finished)
                   (parse-float/parse-integer-only string
                                                   :start index
                                                   :end end
                                                   :radix radix
                                                   :allow-sign nil)
                 (setf integer-part value
                       index position)
                 (if finished
                     #'parse-finish
                     (let ((char (get-char string index)))
                       (cond
                         ((char= char decimal-character)
                          (incf index)
                          #'parse-decimal-part)
                         ((null integer-part)
                          #'parse-finish)
                         ((and (char= char exponent-character)
                               (= radix 10))
                          (setf index (+ 1 index)
                                decimal-part 0)
                          #'parse-exponent-part)
                         (t #'parse-finish))))))

             (parse-decimal-part ()
               (multiple-value-bind (value position finished)
                   (parse-float/parse-integer-only string
                                                   :start index
                                                   :end end
                                                   :radix radix
                                                   :allow-sign nil)
                 (setf decimal-part (or value 0)
                       digits (- position index)
                       index position)
                 (when (and decimal-part
                            (null integer-part))
                   (setf integer-part 0))
                 (if finished
                     #'parse-finish
                     (progn
                       ;;(unless decimal-part
                       ;;  (setf decimal-part 0))
                       (if (and (= radix 10)
                                (char= (get-char string index) exponent-character))
                           (progn
                             (incf index)
                             #'parse-exponent-part)
                           #'parse-finish)))))

             (parse-exponent-part ()
               (multiple-value-bind (value position)
                   (parse-integer string
                                  :start index
                                  :end end
                                  :junk-allowed t)
                 (setf exponent-part (or value 0)
                       index position)
                 #'parse-finish))

             (parse-finish ()
               (unless junk-allowed
                 (setf index (parse-float/skip-whitespaces string :start index :end end)))
               (if integer-part
                   (if (or (= index end)
                           junk-allowed)
                       (locally
                           (declare #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
                         (setf result (let ((mantissa
                                              (* sign (+ (coerce integer-part type)
                                                         (coerce (* decimal-part
                                                                    (expt radix (- digits)))
                                                                 type)))))
                                        (if (minusp exponent-part)
                                            (/ mantissa (expt 10 (- exponent-part)))
                                            (* mantissa (expt 10 exponent-part))))))
                       (error 'float-parse-error
                              :format-control "junk in string ~S."
                              :format-arguments (list string)))
                   (unless junk-allowed
                     (error 'float-parse-error
                            :format-control "junk in string ~S."
                            :format-arguments (list string))))
               nil))
      (declare (dynamic-extent #'parse-sign
                               #'parse-integer-part
                               #'parse-decimal-part
                               #'parse-exponent-part
                               #'parse-finish))

      (loop with parser = #'parse-sign
            while parser
            do (setf parser (funcall (the function parser)))
            finally (return (values result index))))))
