;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/mmapped-file)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Conditions
;;;

(define-condition mmapped-file-error (simple-error stream-error)
  ()
  (:documentation "Indicates an error with an MMAPED-FILE."))

(define-condition mmapped-file-seek-error (mmapped-file-error)
  ()
  (:documentation "Indicates an error where a seek failed for an MMAPED-FILE."))

(define-condition mmapped-file-eof-error (end-of-file mmapped-file-seek-error)
  ()
  (:documentation "Indicates an error where a read failed for an MMAPED-FILE
because it is past the end of the underlying file."))

(defmacro mmapped-file-error ((&optional (type 'mmapped-file-error) stream) msg &rest fmt-args)
  ;; This check doesn't seem to work in ClozureCL or Clisp, and I'm not sure
  ;; why.
  #-(or ccl clisp)
  (unless (subtypep type 'mmapped-file-error)
    (error "~a is not a recognizable subtype of MMAPPED-FILE-ERROR" type))

  `(error ,(if (listp type)
               (if (eq (car type) 'quote)
                   type
                   (error "Invalid type: ~a" type))
               (list 'quote type))
          :format-control ,msg
          :format-arguments (list ,@fmt-args)
          :stream ,stream))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; MMAPPED-FILE
;;;

(deftype t/mmapped-file-mode ()
  '(member :read :write :read-write))

(defclass mmapped-file ()
  ((addr
    :initarg :addr
    :initform nil
    :reader mmapped-file-addr)
   (fd
    :initarg :fd
    :initform nil
    :reader mmapped-file-fd)
   (file-size
    :initarg :file-size
    :initform 0
    :type fixnum
    :reader mmapped-file-file-size)
   (size
    :initarg :size
    :initform 0
    :type fixnum
    :reader mmapped-file-size)
   (cur-offset
    :initform 0
    :type fixnum
    :reader mmapped-file-cur-offset))
  (:documentation "Simple wrapper for mmap()'ed files."))

(defgeneric (setf mmapped-file-cur-offset) (value file))

(defmethod (setf mmapped-file-cur-offset) ((value integer) (file mmapped-file))
  (check-type value fixnum)
  (setf (slot-value file 'cur-offset) value))

(defun make-mmapped-file (filename &key (mode :read) (mapping '(:private)) size (offset 0)
                                    (if-does-not-exist :error) (if-exists :open))
  "Creates a new MMAPPED-FILE from FILENAME.  The returned MMAPPED-FILE must be
freed with CLOSE-MMAPPED-FILE when you are done with it.

MODE must be a T/MMAPPED-FILE-MODE and is used to indicate the desired behavior.
MAPPING is the flags that are passed when mmap()'ing the file.

If the file exists and mode is :READ, then SIZE is optional.  If SIZE is
provided, then it must be less than or equal to the file's actual size.

OFFSET must be between 0 and one less than either SIZE (if provided), or one
less than the file's actual size.

IF-DOES-NOT-EXIST is the behavior taken if the file does not yet exist.  It can
be one of the following:

* :ERROR - If the file does not exist, raise an MMAPPED-FILE-ERROR.
* :CREATE - If the file does not exist, create it and fill it with zeros.  SIZE
  is then required.

IF-EXISTS is the behavior taken if the file already exists, and MODE is :WRITE
or :READ-WRITE.  If MODE is :READ, then IF-EXISTS is ignored.  It can be one of
the following:

* :ERROR - If the file already exists, raise an MMAPPED-FILE-ERROR.
* :ZERO - If the file already exists, overwrite it with zeros and then open it.
  SIZE is then required.  If the existing file is of a different size, it will
  be adjusted.
* :OPEN - If the file already exists, open it but do not zero it out.  If SIZE
  is provided, then the file's actual size must be greater than or equal to
  SIZE.  Otherwise if SIZE is NIL, the file is simply opened."
  (check-type mode t/mmapped-file-mode)
  (check-type offset fixnum)
  (check-type size (or null fixnum))
  (check-type filename (or pathname string))

  ;; Does the file exist?
  (if (probe-file filename)
      (ecase mode
        ((:write :read-write)
         ;; File exists, write or read/write requested
         (ecase if-exists
           (:error
            (mmapped-file-error () "File already exists"))
           (:zero
            (when (null size)
              (error "SIZE cannot be null when the file exists, writing is requested, and MODE is :ZERO"))
            (create-empty-file filename size :if-exists :supersede))
           (:open
            (if (null size)
                (setf size (sdm-file:file-size filename)) ;; Set to the actual size of the file.
                (unless (<= size (sdm-file:file-size filename))
                  (error "Writing is requested and MODE is :OPEN, but the file's actual size is not greater than or equal to SIZE."))))))

        (:read
         ;; File exists, reading requested
         (when size
           (unless (<= size (sdm-file:file-size filename))
             (error "SIZE is not less than or equal to the file's actual size")))))

      ;; File does not exist
      (ecase if-does-not-exist
        (:error (mmapped-file-error () "File does not exist: ~a" filename))
        (:create
         (if (null size)
             (error "SIZE must be provided if the file does not exist.")
             (create-empty-file filename size)))))

  (when (and (null size) (probe-file filename))
    (setf size (sdm-file:file-size filename)))

  (assert (not (null size)))
  (when (or (< offset 0)
            (> offset (1- size)))
    (error "OFFSET must be between zero and one less than the size, inclusive"))

  (let ((open-flags (ecase mode
                      (:read '(:read))
                      (:write '(:write))
                      (:read-write '(:read :write)))))
    (multiple-value-bind (addr fd size)
        (mmap:mmap filename :open open-flags :protection open-flags :mmap mapping :size size :offset offset)
      (make-instance (ecase mode
                       (:read 'cl-sdm/mmapped-file-gray:mmapped-input-stream)
                       (:write 'cl-sdm/mmapped-file-gray:mmapped-output-stream)
                       (:read-write 'mmapped-stream))
                     :addr addr
                     :fd fd
                     :size (- size offset)
                     :file-size size))))

(defun close-mmapped-file (file)
  "Frees the underlying resources associated with the given MMAPPED-FILE."
  (check-type file mmapped-file)
  (mmap:munmap (mmapped-file-addr file) (mmapped-file-fd file) (mmapped-file-file-size file))
  (setf (slot-value file 'addr) nil)
  t)

(defun mmapped-file-open-p (file)
  (check-type file mmapped-file)
  (not (null (mmapped-file-addr file))))

(defmacro with-mmapped-file ((var filename
                              &key (mode :read)
                                (mapping '(:private) mapping-provided-p)
                                size
                                (offset 0)
                                (if-does-not-exist :error)
                                (if-exists :open))
                             &body forms)
  "Opens FILENAME and creates a new MMAPPED-FILE bound for VAR for it.  This
then executes FORMS.  This will ensure CLOSE-MMAPPED-FILE is called at the end."
  `(let ((,var (make-mmapped-file ,filename :mode ,mode :mapping ,(if mapping-provided-p
                                                                      mapping
                                                                      (list 'quote mapping))
                                            :size ,size :offset ,offset
                                            :if-does-not-exist ,if-does-not-exist :if-exists ,if-exists)))
     (unwind-protect
          (progn ,@forms)
       (close-mmapped-file ,var))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Reading functions
;;;

(defmacro define-mmapped-file-read-fn (name type cffi-type size-in-bytes swap-endian?)
  `(define-typed-fn ,name ((mmapped-file file))
       (,type t)
     (declare (optimize speed (debug 1))
              #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
     (with-typed-slots ((fixnum cur-offset)
                        (fixnum size)
                        addr)
         file
       (if (<= cur-offset (- size ,size-in-bytes))
           (prog1 ,(if swap-endian?
                       `(swap-endianness
                         (the ,type (cffi:mem-ref addr ,cffi-type cur-offset))
                         ,(* size-in-bytes 8))
                       `(the ,type (cffi:mem-ref addr ,cffi-type cur-offset)))
             (incf cur-offset ,size-in-bytes))
           (mmapped-file-error (mmapped-file-eof-error) ,(format nil "Cannot read ~a: No more bytes" type))))))

(defmacro define-mmapped-file-get-fn (name type cffi-type size-in-bytes swap-endian?)
  `(define-typed-fn ,name ((mmapped-file file) (fixnum offset))
       (,type t)
     (declare (optimize speed (debug 1))
              #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
     (with-typed-slots ((fixnum size)
                        addr)
         file
       (if (and (>= offset 0) (< offset (- size ,size-in-bytes)))
           ,(if swap-endian?
                `(swap-endianness
                  (the ,type (cffi:mem-ref addr ,cffi-type offset)) ,(* size-in-bytes 8))
                `(the ,type (cffi:mem-ref addr ,cffi-type offset)))
           (mmapped-file-error (mmapped-file-seek-error)
                               ,(format nil "Cannot read ~a from ~~a: invalid offset" type)
                               offset)))))

(define-mmapped-file-read-fn mmapped-file-read-byte   t/uint8  :uint8  1 nil)
(define-mmapped-file-read-fn mmapped-file-read-uint16 t/uint16 :uint16 2 nil)
(define-mmapped-file-read-fn mmapped-file-read-uint32 t/uint32 :uint32 4 nil)
(define-mmapped-file-read-fn mmapped-file-read-uint64 t/uint64 :uint64 8 nil)

(define-mmapped-file-read-fn mmapped-file-read-uint16/be t/uint16 :uint16 2 t)
(define-mmapped-file-read-fn mmapped-file-read-uint32/be t/uint32 :uint32 4 t)
(define-mmapped-file-read-fn mmapped-file-read-uint64/be t/uint64 :uint64 8 t)

(define-mmapped-file-read-fn mmapped-file-read-int8  t/int8  :int8  1 nil)
(define-mmapped-file-read-fn mmapped-file-read-int16 t/int16 :int16 2 nil)
(define-mmapped-file-read-fn mmapped-file-read-int32 t/int32 :int32 4 nil)
(define-mmapped-file-read-fn mmapped-file-read-int64 t/int64 :int64 8 nil)

(define-mmapped-file-read-fn mmapped-file-read-int16/be t/int16 :int16 2 t)
(define-mmapped-file-read-fn mmapped-file-read-int32/be t/int32 :int32 4 t)
(define-mmapped-file-read-fn mmapped-file-read-int64/be t/int64 :int64 8 t)

(define-mmapped-file-get-fn mmapped-file-get-byte   t/uint8  :uint8  1 nil)
(define-mmapped-file-get-fn mmapped-file-get-uint16 t/uint16 :uint16 2 nil)
(define-mmapped-file-get-fn mmapped-file-get-uint32 t/uint32 :uint32 4 nil)
(define-mmapped-file-get-fn mmapped-file-get-uint64 t/uint64 :uint64 8 nil)

(define-mmapped-file-get-fn mmapped-file-get-uint16/be t/uint16 :uint16 2 t)
(define-mmapped-file-get-fn mmapped-file-get-uint32/be t/uint32 :uint32 4 t)
(define-mmapped-file-get-fn mmapped-file-get-uint64/be t/uint64 :uint64 8 t)

(define-mmapped-file-get-fn mmapped-file-get-int8  t/int8  :int8  1 nil)
(define-mmapped-file-get-fn mmapped-file-get-int16 t/int16 :int16 2 nil)
(define-mmapped-file-get-fn mmapped-file-get-int32 t/int32 :int32 4 nil)
(define-mmapped-file-get-fn mmapped-file-get-int64 t/int64 :int64 8 nil)

(define-mmapped-file-get-fn mmapped-file-get-int16/be t/int16 :int16 2 t)
(define-mmapped-file-get-fn mmapped-file-get-int32/be t/int32 :int32 4 t)
(define-mmapped-file-get-fn mmapped-file-get-int64/be t/int64 :int64 8 t)

(define-typed-fn mmapped-file-read-float32 ((mmapped-file file))
    (single-float t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (uint32->sfloat (mmapped-file-read-uint32 file)))

(define-typed-fn mmapped-file-read-float32/be ((mmapped-file file))
    (single-float t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (uint32->sfloat (mmapped-file-read-uint32/be file)))

(define-typed-fn mmapped-file-read-float64 ((mmapped-file file))
    (double-float t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (uint64->dfloat (mmapped-file-read-uint64 file)))

(define-typed-fn mmapped-file-read-float64/be ((mmapped-file file))
    (double-float t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (uint64->dfloat (mmapped-file-read-uint64/be file)))

(define-typed-fn mmapped-file-get-float32 ((mmapped-file file) (fixnum offset))
    (single-float t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (uint32->sfloat (mmapped-file-get-uint32 file offset)))

(define-typed-fn mmapped-file-get-float32/be ((mmapped-file file) (fixnum offset))
    (single-float t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (uint32->sfloat (mmapped-file-get-uint32/be file offset)))

(define-typed-fn mmapped-file-get-float64 ((mmapped-file file) (fixnum offset))
    (double-float t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (uint64->dfloat (mmapped-file-get-uint64 file offset)))

(define-typed-fn mmapped-file-get-float64/be ((mmapped-file file) (fixnum offset))
    (double-float t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (uint64->dfloat (mmapped-file-get-uint64/be file offset)))

(define-typed-fn mmapped-file-read-string ((mmapped-file file) (fixnum length))
    (simple-string t)
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((fixnum cur-offset)
                     (fixnum size)
                     addr)
      file
    (if (< cur-offset (- size length))
        (muffling
          (prog1 (coerce (cffi:foreign-string-to-lisp addr :offset cur-offset :count length)
                         'simple-string)
            (incf cur-offset length)))
        (mmapped-file-error (mmapped-file-eof-error) "Not enough bytes to read a string of length ~d" length))))

(define-typed-fn mmapped-file-get-string ((mmapped-file file) (fixnum length offset))
    (simple-string t)
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((fixnum size)
                     addr)
      file
    (cond
      ((< offset 0)
       (mmapped-file-error (mmapped-file-error) "Bad offset: ~a" offset))

      ((< offset (- size length))
       (muffling
         (coerce (cffi:foreign-string-to-lisp addr :offset offset :count length)
                 'simple-string)))

      (t
       (mmapped-file-error (mmapped-file-eof-error) "Not enough bytes to read a string of length ~d" length)))))

(define-typed-fn mmapped-file-read-bytes ((mmapped-file file) (fixnum count))
    ((simple-array t/uint8) t)
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((fixnum cur-offset)
                     (fixnum size)
                     addr)
      file
    (if (< cur-offset(- size count))
        (let ((ret (new-array count t/uint8)))
          (dotimes (i count)
            (setf (aref ret i) (cffi:mem-ref addr :uint8 (+ cur-offset i))))
          (incf cur-offset count)
          ret)
        (mmapped-file-error (mmapped-file-eof-error)
                            "Not enough bytes to read an array of bytes of length ~d"
                            count))))

(define-typed-fn mmapped-file-get-bytes ((mmapped-file file) (fixnum count offset))
    ((simple-array t/uint8) t)
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((fixnum size)
                     addr)
      file
    (cond
      ((< offset 0)
       (mmapped-file-error () "Bad offset: ~a" offset))

      ((< offset (- size count))
       (let ((ret (new-array count t/uint8)))
         (dotimes (i count)
           (setf (aref ret i) (cffi:mem-ref addr :uint8 (+ offset i))))
         ret))

      (t
       (mmapped-file-error (mmapped-file-eof-error)
                           "Not enough bytes to read an array of bytes of length ~d"
                           count)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Writing functions
;;;

(defmacro define-mmapped-file-write-fn (name type cffi-type size-in-bytes swap-endian?)
  `(define-typed-fn ,name ((mmapped-file file) (,type value))
       (null t)
     (declare (optimize speed (debug 1))
              #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
     (with-typed-slots ((fixnum cur-offset)
                        (fixnum size)
                        addr)
         file
       (if (<= cur-offset (- size ,size-in-bytes))
           (progn
             (setf (cffi:mem-ref addr ,cffi-type cur-offset)
                   ,(if swap-endian?
                        `(swap-endianness value ,(* size-in-bytes 8))
                        `value))
             (incf cur-offset ,size-in-bytes))
           (mmapped-file-error (mmapped-file-eof-error) ,(format nil "Cannot write ~a: No more space" type))))
     nil))

(defmacro define-mmapped-file-put-fn (name type cffi-type size-in-bytes swap-endian?)
  `(define-typed-fn ,name ((mmapped-file file) (fixnum offset) (,type value))
       (null t)
     (declare (optimize speed (debug 1))
              #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
     (with-typed-slots ((fixnum size)
                        addr)
         file
       (if (<= offset (- size ,size-in-bytes))
           (setf (cffi:mem-ref addr ,cffi-type offset)
                 ,(if swap-endian?
                      `(swap-endianness value ,(* size-in-bytes 8))
                      `value))
           (mmapped-file-error (mmapped-file-eof-error) ,(format nil "Cannot write ~a: No more space" type))))
     nil))

(define-mmapped-file-write-fn mmapped-file-write-byte   t/uint8  :uint8  1 nil)
(define-mmapped-file-write-fn mmapped-file-write-uint16 t/uint16 :uint16 2 nil)
(define-mmapped-file-write-fn mmapped-file-write-uint32 t/uint32 :uint32 4 nil)
(define-mmapped-file-write-fn mmapped-file-write-uint64 t/uint64 :uint64 8 nil)
(define-mmapped-file-write-fn mmapped-file-write-uint16/be t/uint16 :uint16 2 t)
(define-mmapped-file-write-fn mmapped-file-write-uint32/be t/uint32 :uint32 4 t)
(define-mmapped-file-write-fn mmapped-file-write-uint64/be t/uint64 :uint64 8 t)

(define-mmapped-file-write-fn mmapped-file-write-int8  t/int8  :int8  1 nil)
(define-mmapped-file-write-fn mmapped-file-write-int16 t/int16 :int16 2 nil)
(define-mmapped-file-write-fn mmapped-file-write-int32 t/int32 :int32 4 nil)
(define-mmapped-file-write-fn mmapped-file-write-int64 t/int64 :int64 8 nil)
(define-mmapped-file-write-fn mmapped-file-write-int16/be t/int16 :int16 2 t)
(define-mmapped-file-write-fn mmapped-file-write-int32/be t/int32 :int32 4 t)
(define-mmapped-file-write-fn mmapped-file-write-int64/be t/int64 :int64 8 t)

(define-mmapped-file-put-fn mmapped-file-put-byte   t/uint8  :uint8  1 nil)
(define-mmapped-file-put-fn mmapped-file-put-uint16 t/uint16 :uint16 2 nil)
(define-mmapped-file-put-fn mmapped-file-put-uint32 t/uint32 :uint32 4 nil)
(define-mmapped-file-put-fn mmapped-file-put-uint64 t/uint64 :uint64 8 nil)
(define-mmapped-file-put-fn mmapped-file-put-uint16/be t/uint16 :uint16 2 t)
(define-mmapped-file-put-fn mmapped-file-put-uint32/be t/uint32 :uint32 4 t)
(define-mmapped-file-put-fn mmapped-file-put-uint64/be t/uint64 :uint64 8 t)

(define-mmapped-file-put-fn mmapped-file-put-int8  t/int8  :int8  1 nil)
(define-mmapped-file-put-fn mmapped-file-put-int16 t/int16 :int16 2 nil)
(define-mmapped-file-put-fn mmapped-file-put-int32 t/int32 :int32 4 nil)
(define-mmapped-file-put-fn mmapped-file-put-int64 t/int64 :int64 8 nil)
(define-mmapped-file-put-fn mmapped-file-put-int16/be t/int16 :int16 2 t)
(define-mmapped-file-put-fn mmapped-file-put-int32/be t/int32 :int32 4 t)
(define-mmapped-file-put-fn mmapped-file-put-int64/be t/int64 :int64 8 t)

(define-typed-fn mmapped-file-write-float32 ((mmapped-file file) (single-float value))
    (null t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (mmapped-file-write-uint32 file (sfloat->uint32 value)))

(define-typed-fn mmapped-file-write-float32/be ((mmapped-file file) (single-float value))
    (null t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (mmapped-file-write-uint32/be file (sfloat->uint32 value)))

(define-typed-fn mmapped-file-write-float64 ((mmapped-file file) (double-float value))
    (null t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (mmapped-file-write-uint64 file (dfloat->uint64 value)))

(define-typed-fn mmapped-file-write-float64/be ((mmapped-file file) (double-float value))
    (null t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (mmapped-file-write-uint64/be file (dfloat->uint64 value)))

(define-typed-fn mmapped-file-put-float32 ((mmapped-file file) (fixnum offset) (single-float value))
    (null t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (mmapped-file-put-uint32 file offset (sfloat->uint32 value)))

(define-typed-fn mmapped-file-put-float32/be ((mmapped-file file) (fixnum offset) (single-float value))
    (null t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (mmapped-file-put-uint32/be file offset (sfloat->uint32 value)))

(define-typed-fn mmapped-file-put-float64 ((mmapped-file file) (fixnum offset) (double-float value))
    (null t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (mmapped-file-put-uint64 file offset (dfloat->uint64 value)))

(define-typed-fn mmapped-file-put-float64/be ((mmapped-file file) (fixnum offset) (double-float value))
    (null t)
  (declare (optimize speed (debug 1))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (mmapped-file-put-uint64/be file offset (dfloat->uint64 value)))

(define-typed-fn mmapped-file-write-bytes ((mmapped-file file) (sequence value))
    (null t)
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((fixnum cur-offset)
                     (fixnum size)
                     addr)
      file
    (let ((len (muffling (length value))))
      (declare (type fixnum len))
      (if (<= cur-offset (- size len))
          (muffling
            (loop for i fixnum from 0 below len
                  do (setf (cffi:mem-ref addr :uint8 (+ i cur-offset))
                           (elt value i))
                  finally (incf cur-offset len)))
          (mmapped-file-error (mmapped-file-eof-error) "Not enough space to write ~d bytes" len))))
  nil)

(define-typed-fn mmapped-file-put-bytes ((mmapped-file file) (fixnum offset) (sequence value))
    (null t)
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((fixnum size)
                     addr)
      file
    (let ((len (muffling (length value))))
      (declare (type fixnum len))
      (if (<= offset (- size len))
          (muffling
            (loop for i fixnum from 0 below len
                  do (setf (cffi:mem-ref addr :uint8 (+ i offset))
                           (elt value i))))
          (mmapped-file-error (mmapped-file-eof-error) "Not enough space to put ~d bytes" len))))
  nil)

(define-typed-fn mmapped-file-write-string ((mmapped-file file) (string value))
    (null t)
  (declare (optimize speed (debug 1)))
  (mmapped-file-write-bytes file (babel:string-to-octets value)))

(define-typed-fn mmapped-file-put-string ((mmapped-file file) (fixnum offset) (string value))
    (null t)
  (declare (optimize speed (debug 1)))
  (mmapped-file-put-bytes file offset (babel:string-to-octets value)))
