;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/mmapped-file-gray)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Gray Stream Interface
;;;

(defclass mmapped-input-stream (mmapped-file trivial-gray-streams:fundamental-binary-input-stream)
  ())

(defclass mmapped-output-stream (mmapped-file trivial-gray-streams:fundamental-binary-output-stream)
  ())

(defclass mmapped-stream (mmapped-input-stream mmapped-output-stream)
  ())

(defun open-mmapped-stream (filename &key (direction :input)
                                       if-exists
                                       if-does-not-exist)
  "An equivalent to OPEN, but backed by an mmap()'ed file."
  (make-mmapped-file filename :mode (ecase direction
                                      (:input :read)
                                      (:output :write)
                                      (:io :read-write))
                              :if-does-not-exist (if if-does-not-exist
                                                     if-does-not-exist
                                                     (ecase direction
                                                       (:input :error)
                                                       ((:output :io)
                                                        (case if-exists
                                                          ((:overwrite :append)
                                                           :create)
                                                          (otherwise :nil)))))
                              :if-exists (if (eq (pathname-version (sdm-file:native-pathname filename))
                                                 :newest)
                                             :create
                                             :error)))

(defmacro with-mmapped-stream ((in filename &key (direction :input) if-exists if-does-not-exist) &body forms)
  "An equivalent to WITH-OPEN-FILE, but backed by an mmap()'ed file."
  `(let ((,in (open-mmapped-stream ,filename :direction ,direction :if-exists ,if-exists
                                             :if-does-not-exist ,if-does-not-exist)))
     (unwind-protect
          (progn ,@forms)
       (close ,in))))

(defmethod cl:open-stream-p ((stream mmapped-file))
  (declare (optimize speed (debug 1)))
  (with-slots ((sdm-mmapped-file::addr sdm-mmapped-file::addr))
      stream
    (not (null sdm-mmapped-file::addr))))

(defmacro ensuring-open-stream ((stream) &body forms)
  `(if (open-stream-p ,stream)
       (progn ,@forms)
       (error "Stream is closed")))

(defmethod cl:input-stream-p ((stream mmapped-input-stream))
  t)

(defmethod cl:output-stream-p ((stream mmapped-output-stream))
  t)

(defmethod cl:input-stream-p ((stream mmapped-file))
  nil)

(defmethod cl:stream-element-type ((stream mmapped-file))
  '(unsigned-byte 8))

(defmethod cl:close ((stream mmapped-file) &key abort)
  (declare (optimize speed (debug 1)))
  (declare (ignore abort))
  (close-mmapped-file stream))

(defmethod trivial-gray-streams:stream-listen ((stream mmapped-input-stream))
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    (with-typed-slots ((fixnum cl-sdm/mmapped-file::cur-offset cl-sdm/mmapped-file::size))
        stream
      (< cl-sdm/mmapped-file::cur-offset cl-sdm/mmapped-file::size))))

(defmethod trivial-gray-streams:stream-file-position ((stream mmapped-file))
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((fixnum sdm-mmapped-file::cur-offset))
      stream
    cl-sdm/mmapped-file::cur-offset))

(defmethod trivial-gray-streams:stream-write-string ((stream mmapped-file) (string string) &optional start end)
  (declare (optimize speed (debug 1)))
  (mmapped-file-write-string stream (subseq string (or start 0) end)))

(defmethod (setf trivial-gray-streams:stream-file-position) (new-offset (stream mmapped-file))
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((fixnum cl-sdm/mmapped-file::cur-offset cl-sdm/mmapped-file::size))
      stream
    (case new-offset
      (:start
       (setf cl-sdm/mmapped-file::cur-offset 0))
      (:end
       (setf cl-sdm/mmapped-file::cur-offset cl-sdm/mmapped-file::size))
      (otherwise
       (unless (and (typep new-offset 'fixnum)
                    (>= new-offset 0))
         (mmapped-file-error (mmapped-file-seek-error stream)
                             "Bad file position: ~a"
                             new-offset))

       (unless (<= new-offset cl-sdm/mmapped-file::size)
         (mmapped-file-error (mmapped-file-seek-error stream)
                             "File position is past the end of the file: ~a is past ~a"
                             new-offset cl-sdm/mmapped-file::size))

       (setf cl-sdm/mmapped-file::cur-offset new-offset)))
    cl-sdm/mmapped-file::cur-offset))

(defmethod trivial-gray-streams:stream-read-byte ((stream mmapped-input-stream))
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    (with-typed-slots ((fixnum cl-sdm/mmapped-file::cur-offset cl-sdm/mmapped-file::size)
                       sdm-mmapped-file::addr)
        stream
      (if (<= cl-sdm/mmapped-file::cur-offset (1- cl-sdm/mmapped-file::size))
          (prog1 (the t/uint8 (cffi:mem-ref sdm-mmapped-file::addr :uint8 cl-sdm/mmapped-file::cur-offset))
            (incf (the fixnum cl-sdm/mmapped-file::cur-offset)))
          :eof))))

(defmethod trivial-gray-streams:stream-read-sequence ((stream mmapped-input-stream) sequence start end &key)
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    (with-typed-slots ((fixnum cl-sdm/mmapped-file::cur-offset cl-sdm/mmapped-file::size)
                       sdm-mmapped-file::addr)
        stream
      (loop for dest-idx fixnum from start below end
            while (<= (+ cl-sdm/mmapped-file::cur-offset dest-idx) cl-sdm/mmapped-file::size)
            do (muffling (setf (elt sequence dest-idx)
                               (the t/uint8 (cffi:mem-ref sdm-mmapped-file::addr
                                                          :uint8
                                                          cl-sdm/mmapped-file::cur-offset))))
               (incf (the fixnum cl-sdm/mmapped-file::cur-offset))
            finally (return dest-idx)))))

(defmethod trivial-gray-streams:stream-write-byte ((stream mmapped-output-stream) integer)
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    (unless (typep integer 't/uint8)
      (error "Integer is not a byte"))

    (with-typed-slots ((fixnum cl-sdm/mmapped-file::cur-offset cl-sdm/mmapped-file::size)
                       cl-sdm/mmapped-file::addr)
        stream
      (if (<= cl-sdm/mmapped-file::cur-offset (1- cl-sdm/mmapped-file::size))
          (prog1 integer
            (setf (cffi:mem-ref cl-sdm/mmapped-file::addr :uint8 cl-sdm/mmapped-file::cur-offset) integer)
            (incf (the fixnum cl-sdm/mmapped-file::cur-offset)))
          (error 'end-of-file :stream stream)))))

(defmethod trivial-gray-streams:stream-write-sequence ((stream mmapped-output-stream) sequence start end &key)
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    (with-typed-slots ((fixnum cl-sdm/mmapped-file::cur-offset cl-sdm/mmapped-file::size)
                       cl-sdm/mmapped-file::addr)
        stream
      (loop for src-idx fixnum from start below end
            for element = (muffling (elt sequence src-idx))
            while (< sdm-mmapped-file::cur-offset sdm-mmapped-file::size)
            do (if (typep element 't/uint8)
                   (progn
                     (setf (cffi:mem-ref sdm-mmapped-file::addr :uint8 sdm-mmapped-file::cur-offset) element)
                     (incf (the fixnum sdm-mmapped-file::cur-offset)))
                   (mmapped-file-error () "Data at index ~a is not a byte" src-idx)))))
  sequence)

(defmethod trivial-gray-streams:stream-finish-output ((stream mmapped-output-stream))
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    (with-typed-slots ((fixnum sdm-mmapped-file::size) sdm-mmapped-file::addr sdm-mmapped-file::fd)
        stream
      (muffling (mmap:msync sdm-mmapped-file::addr sdm-mmapped-file::fd sdm-mmapped-file::size))))
  t)

(defmethod trivial-gray-streams:stream-clear-output ((stream mmapped-output-stream))
  (declare (optimize speed (debug 1)))
  (finish-output stream))

(defmethod trivial-gray-streams:stream-force-output ((stream mmapped-output-stream))
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    (with-typed-slots ((fixnum sdm-mmapped-file::size) sdm-mmapped-file::addr sdm-mmapped-file::fd)
        stream
      (muffling (mmap:msync sdm-mmapped-file::addr sdm-mmapped-file::fd sdm-mmapped-file::size
                            :flags '(:async)))))
  t)

(defmethod trivial-gray-streams:stream-read-char ((stream mmapped-input-stream))
  (declare (optimize speed (debug 1)))
  (let ((byte (trivial-gray-streams:stream-read-byte stream)))
    (if (eq byte :eof)
        :eof
        (code-char byte))))

(defmethod trivial-gray-streams:stream-read-line ((stream mmapped-input-stream))
  (declare (optimize speed (debug 1)))
  (values
   (with-output-to-string (out)
     (loop for char = (trivial-gray-streams:stream-read-char stream) do
       (case char
         (:eof
          (return-from trivial-gray-streams:stream-read-line
            (values (get-output-stream-string out) t)))
         (otherwise
          (if (not (char= char #\Newline))
              (write-char char out)
              (loop-finish))))))
   nil))

(defmethod trivial-gray-streams:stream-write-char ((stream mmapped-output-stream) (character character))
  (declare (optimize speed (debug 1)))
  (let* ((code (char-code character))
         (data (uint->bytes (int->uint/2c code (* 8 (num-8bit-bytes-needed code))))))
    (declare (type list data)
             (dynamic-extent data))
    (with-typed-slots ((fixnum sdm-mmapped-file::cur-offset sdm-mmapped-file::size))
        stream
      (unless (<= (+ sdm-mmapped-file::cur-offset (length data)) sdm-mmapped-file::size)
        (error 'end-of-file :stream stream)))
    (write-sequence data stream))
  character)
