;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;
;;;; MEMORY-UTF-8-STREAM - An in-memory binary I/O stream that supports reading
;;;; UTF-8 character data.
;;;;

(defclass memory-utf-8-stream (memory-stream
                               binary-utf-8-output-stream
                               binary-utf-8-input-stream)
  ((incomplete-char-bytes
    :initform 0
    :type t/utf-8-codepoint-byte))
  (:documentation "The MEMORY-UTF-8-STREAM class is very similar to a MEMORY-STREAM, but the
character/string methods properly work on UTF-8 character data."))

(defmethod memory-stream-reset :after ((stream memory-utf-8-stream))
  (with-typed-slots ((t/utf-8-codepoint-byte incomplete-char-bytes))
      stream
    (setf incomplete-char-bytes 0))
  (binary-utf-8-stream-reset stream))

(defmethod read-line* ((stream memory-utf-8-stream) (limit integer) &optional recursive-p)
  "Same as READ-LINE, but only reads up to LIMIT characters from STREAM.  The
newline, if encountered, is read from the stream, but is not counted towards and
is not included in the returned value.

This ignores the RECURSIVE-P argument."
  (declare (ignore recursive-p)
           (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (assert (typep limit 't/ufixnum))

  (multiple-value-bind (ret incomplete)
      (binary-utf-8-input-stream-do-read-line* stream limit)
    (setf (slot-value stream 'incomplete-char-bytes) incomplete)
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Gray Stream Interface - Others
;;;

(defmethod trivial-gray-streams:stream-file-position ((stream memory-utf-8-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (- (the fixnum (slot-value stream 'pos))
     (the t/utf-8-codepoint-byte (slot-value stream 'incomplete-char-bytes))))

(defmethod (setf trivial-gray-streams:stream-file-position) (new-offset (stream memory-utf-8-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((fixnum pos)
                     (t/memory-stream-buffer buffer))
      stream
    (setf (slot-value stream 'did-unread?) nil)
    (case new-offset
      (:start
       (setf (slot-value stream 'input-buffer) (new-vector character))
       (setf (slot-value stream 'input-buffer-sizes) (new-vector (integer 0 4)))
       (setf pos 0))

      (:end
       (setf (slot-value stream 'input-buffer) (new-vector character))
       (setf (slot-value stream 'input-buffer-sizes) (new-vector (integer 0 4)))
       (setf pos (muffling (length buffer))))

      (otherwise
       (let ((new-pos new-offset))
         (cond
           ((and (typep new-pos 'fixnum)
                 (>= new-pos 0)
                 (<= new-pos (muffling (length buffer))))
            ;; We'll need to account for any unread UTF-8 characters.
            (with-typed-slots ((t/utf-8-stream-back-buffer input-buffer)
                               (t/utf-8-stream-back-size-buffer input-buffer-sizes))
                stream
              (let ((bytes-in-buffer (binary-utf-8-input-stream-back-buffer-byte-length stream)))
                (cond
                  ((<= new-pos bytes-in-buffer)
                   ;; Pop off the required number of characters in the back
                   ;; buffers, then seek normally.
                   (loop for idx of-type t/ufixnum from 0 below (length input-buffer-sizes)
                         for size of-type (integer 0 4) = (muffling (aref input-buffer-sizes idx))
                         repeat (- bytes-in-buffer new-pos)
                         do (muffling (decf new-pos size))
                         finally (dotimes (i idx)
                                   (vector-pop input-buffer)
                                   (vector-pop input-buffer-sizes)))

                   (setf pos new-pos))

                  (t
                   ;; Clear input buffers, then seek normally.
                   (setf (slot-value stream 'input-buffer) (new-vector character))
                   (setf (slot-value stream 'input-buffer-sizes) (new-vector (integer 0 4)))
                   (setf pos new-pos)))))
            t)
           (t nil)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Gray Stream Interface - Input
;;;

;; Reimplement so since a CALL-NEXT-METHOD won't call the correct method.
(defmethod trivial-gray-streams:stream-read-line ((stream memory-utf-8-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (binary-utf-8-input-stream-do-read-line stream))

(defmethod trivial-gray-streams:stream-read-char ((stream memory-utf-8-stream))
  (setf (slot-value stream 'did-unread?) nil)
  (let ((chars-in-buffer (binary-utf-8-input-stream-back-buffer-length stream)))
    (if (plusp chars-in-buffer)
        ;; This won't actually call STREAM-READ-BYTE, so we have to adjust the
        ;; position ourselves.
        (multiple-value-bind (ret num-bytes)
            (binary-utf-8-input-stream-do-read-char stream)
          (with-typed-slots ((fixnum pos)
                             ((integer 0 4) incomplete-char-bytes))
              stream
            (incf pos num-bytes)
            (setf incomplete-char-bytes 0)
            ret))

        ;; STREAM-READ-BYTE will be called normally, so POS will already be
        ;; updated.
        (nth-value 0 (binary-utf-8-input-stream-do-read-char stream)))))

(defmethod trivial-gray-streams:stream-unread-char ((stream memory-utf-8-stream) (character character))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (when (slot-value stream 'did-unread?)
    (%memory-stream-error stream "Cannot UNREAD-CHAR twice"))
  (binary-utf-8-input-stream-do-unread-char stream character)
  (with-typed-slots ((fixnum pos)
                     (t/utf-8-codepoint-byte incomplete-char-bytes))
      stream
    (decf pos (+ (the (integer 0 #.char-code-limit)
                      (round (the (integer 0 #.(* 8 char-code-limit))
                                  (integer-length (char->utf-8 character)))
                             8))
                 incomplete-char-bytes))
    (setf incomplete-char-bytes 0)
    (binary-utf-8-stream-reset stream))
  (setf (slot-value stream 'did-unread?) t)
  nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Gray Stream Interface - Outut
;;;

(defmethod trivial-gray-streams:stream-write-string ((stream memory-utf-8-stream) (string string) &optional start end)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (assert (typep start '(or null integer)))
  (assert (typep end '(or null integer)))
  (setf (slot-value stream 'did-unread?) nil)
  (binary-utf-8-output-stream-do-write-string stream string (or start 0) end))

(defmethod trivial-gray-streams:stream-write-char ((stream memory-utf-8-stream) (character character))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (binary-utf-8-output-stream-do-write-char stream character))
