;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;
;;;; MEMORY-STREAM - An in-memory I/O stream.
;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Conditions
;;;

(define-simple-error memory-stream-error nil (stream-error))
(define-simple-error memory-stream-read-only-error memory-stream-error ())

(defmacro %memory-stream-error (strm fmt &rest fmt-args)
  `(error 'memory-stream-error :stream ,strm
                               :format-control ,fmt
                               :format-args (list ,@fmt-args)))

(defmacro %read-only-error (strm)
  `(%memory-stream-error ,strm "Stream is read-only, cannot write"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Generic Functions
;;;

(defgeneric memory-stream-buffer (stream)
  (:documentation "Returns a copy the internal data buffer."))

(defgeneric memory-stream-buffer! (stream)
  (:documentation "Returns the internal data buffer.

The returned value is NOT a copy of the internal data and should not be
changed."))

(defgeneric memory-stream-string (stream)
  (:documentation "Returns a copy of the internal data buffer as a string."))

(defgeneric memory-stream-read-only-p (stream)
  (:documentation "Returns T if STREAM can only be read from, or NIL otherwise."))

(defgeneric (setf memory-stream-read-only-p) (value stream)
  (:documentation "Sets STREAM to be a read-only stream if VALUE is truthy.  If
VALUE is NIL, then STREAM will be set as a read-write stream."))

(defgeneric memory-stream-size (stream)
  (:documentation "Returns the size of the stream in bytes."))

(defgeneric memory-stream-reset (stream)
  (:documentation "Clears the internal buffer and resets the position back to 0.  If the
MEMORY-STREAM was created with an already-existing buffer, this instead creates
a new internal buffer of the same length filled with zeros."))

(defgeneric memory-stream-length (stream)
  (:documentation "Returns the length of the internal buffer of the stream."))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; MEMORY-STREAM class and methods
;;;

(deftype t/memory-stream-buffer ()
  '(vector t/uint8 *))

(defclass memory-stream (trivial-gray-streams:fundamental-binary-input-stream
                         trivial-gray-streams:fundamental-binary-output-stream)
  ((buffer
    :initarg :buffer
    :type t/memory-stream-buffer
    :reader memory-stream-buffer!)

   (initial-size
    :initarg :initial-size
    :initform nil
    :type (or null fixnum))

   (pos
    :initform 0
    :type fixnum)

   (closed?
    :initform nil
    :type boolean)

   (did-unread?
    :initform nil
    :type boolean)

   (read-only?
    :initarg :read-only
    :initform nil
    :type boolean
    :reader memory-stream-read-only-p))
  (:documentation "The MEMORY-STREAM class provides an in-memory binary stream
that can be read from and written to using normal stream functions.  This also
allows the reading and writing of string/character data, though it is limited to
ASCII.

A new MEMORY-STREAM can be created using MAKE-INSTANCE.  The default will be a
read-write stream backed by a growable (VECTOR (UNSIGNED-BYTE 8)).

The :INITIAL-SIZE will parameter can be used to create an initial buffer filled
with that many zero bytes.  When provided, INITIAL-SIZE must be a FIXNUM, and
must be positive.

If you use the :BUFFER parameter, then the stream will instead be backed by that
buffer.  :BUFFER must then be a T/MEMORY-STREAM-BUFFER.  Be aware that the write
functions will attempt to call VECTOR-PUSH-EXTEND when writing past the end of
the buffer.

:INITIAL-SIZE and :BUFFER cannot be used together.  When neither are specified,
then an empty internal buffer is created.

The stream's internal read/write cursor will always be positioned at 0."))

(defmethod initialize-instance :after ((obj memory-stream) &key &allow-other-keys)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-slots (initial-size read-only?)
      obj
    ;; Normalize to a boolean
    (setf read-only? (if read-only? t nil))

    ;; Can't use both of these at the same time.
    (when (and initial-size
               (and (slot-boundp obj 'buffer)
                    (slot-value obj 'buffer)))
      (error ":INITIAL-SIZE cannot be used together with :BUFFER"))

    ;; This can't be zero.
    (if initial-size
        (when (<= initial-size 0)
          (error ":INITIAL-SIZE cannot be negative"))
        (setf initial-size (if (and (slot-boundp obj 'buffer)
                                    (slot-value obj 'buffer))
                               (length (slot-value obj 'buffer))
                               0)))

    (assert (typep initial-size '(integer 0 *)))
    (unless (and (slot-boundp obj 'buffer)
                 (slot-value obj 'buffer))
      (setf (slot-value obj 'buffer)
            (make-array initial-size
                        :element-type 't/uint8
                        :initial-element 0
                        :adjustable t
                        :fill-pointer (if (plusp initial-size)
                                          initial-size
                                          0))))

    (unless (typep (slot-value obj 'buffer) 't/memory-stream-buffer)
      (error "BUFFER must be an ARRAY with an element type of (UNSIGNED-BYTE 8)"))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Macros and Non-Gray Stream Functions/Methods
;;;

(defmacro with-memory-stream ((var &key initial-size (buffer nil buffer-supplied-p) read-only return)
                              &body forms)
  "Binds VAR to a new MEMORY-STREAM.  INITIAL-SIZE, BUFFER, and READ-ONLY are
used when creating the new MEMORY-STREAM.  This then executes FORMS.

The return value will depend on :RETURN:

  * NIL - Returns the last value returned by FORMS.
  * :BUFFER - Returns the underlying buffer.
  * :STRING - Returns the underlying buffer converted to a string."
  (with-gensyms (ret)
    `(let ((,var ,(if buffer-supplied-p
                      `(make-instance 'memory-stream
                                      :initial-size ,initial-size
                                      :buffer ,buffer
                                      :read-only ,read-only)
                      `(make-instance 'memory-stream
                                      :initial-size ,initial-size
                                      :read-only ,read-only)))
           (,ret nil))
       (declare (ignorable ,ret))
       (unwind-protect
            (setf ,ret (multiple-value-list (progn ,@forms)))
         (close ,var))

       ,(ecase return
          ('nil `(values-list ,ret))
          (:buffer `(memory-stream-buffer! ,var))
          (:string `(bytes->string (memory-stream-buffer! ,var)))))))

(defmethod memory-stream-string ((stream memory-stream))
  (bytes->string (memory-stream-buffer stream)))

(defmethod memory-stream-buffer ((stream memory-stream))
  (declare (optimize speed (debug 1)))
  (muffling (copy-seq (the t/memory-stream-buffer (slot-value stream 'buffer)))))

(defmethod memory-stream-length ((stream memory-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (length (the t/memory-stream-buffer (slot-value stream 'buffer))))

(defmacro ensuring-open-stream ((stream) &body forms)
  `(if (open-stream-p ,stream)
       (progn ,@forms)
       (%memory-stream-error ,stream "Stream is closed")))

(defmethod memory-stream-size ((stream memory-stream))
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((t/memory-stream-buffer buffer))
      stream
    (muffling (length buffer))))

(defmethod (setf memory-stream-read-only-p) (value (stream memory-stream))
  (declare (optimize speed (debug 1)))
  (setf (slot-value stream 'read-only?) (if value t nil)))

(defmethod memory-stream-reset ((stream memory-stream))
  (with-typed-slots ((boolean closed? read-only?)
                     (t/memory-stream-buffer buffer)
                     (fixnum pos)
                     ((or null fixnum) initial-size))
      stream
    (assert (typep initial-size '(integer 0 *)))
    (when closed?
      (%memory-stream-error stream "Stream is closed."))
    (when read-only?
      (%read-only-error stream))
    (setf (slot-value stream 'did-unread?) nil)
    (setf pos 0)
    (setf buffer (make-array initial-size
                             :element-type 't/uint8
                             :initial-element 0
                             :adjustable t
                             :fill-pointer (if (plusp initial-size)
                                               (1- (or initial-size 1))
                                               0))))
  t)

(defmethod trivial-gray-streams:stream-read-line ((stream memory-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (loop with out = (make-string-output-stream)
        for byte of-type (or t/uint8 (eql :eof)) = (trivial-gray-streams:stream-read-byte stream)
        until (or (eq byte :eof)
                  (= byte #.(char-code #\Newline)))
        do (write-char (code-char byte) out)
        finally (return (values (get-output-stream-string out)
                                (eq byte :eof)))))

(defmethod read-line* ((stream memory-stream) (limit integer) &optional recursive-p)
  "Same as READ-LINE, but only reads up to LIMIT bytes from STREAM as a string.
The newline, if encountered, is read from the stream, but is not counted towards
and is not included in the returned value."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           (ignore recursive-p))
  (assert (typep limit 't/ufixnum))
  (with-output-to-string (out)
    (loop with remaining of-type (integer -1 #.most-positive-fixnum) = limit
          for byte of-type (or t/uint8 (eql :eof)) = (trivial-gray-streams:stream-read-byte stream)
          until (eq byte :eof)
          do (when (= byte #.(char-code #\Newline))
               (loop-finish))
             (write-char (code-char byte) out)
             (when (zerop (decf remaining))
               (loop-finish)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Gray Stream Interface - Others
;;;

(defmethod cl:open-stream-p ((stream memory-stream))
  (declare (optimize speed (debug 1)))
  (not (slot-value stream 'closed?)))

#-clisp
(defmethod cl:input-stream-p ((stream memory-stream))
  t)

#-clisp
(defmethod cl:output-stream-p ((stream memory-stream))
  (not (slot-value stream 'read-only?)))

(defmethod cl:stream-element-type ((stream memory-stream))
  '(unsigned-byte 8))

(defmethod cl:close ((stream memory-stream) &key abort)
  (declare (optimize speed (debug 1)))
  (declare (ignore abort))
  (setf (slot-value stream 'closed?) t))

(defmethod trivial-gray-streams:stream-file-position ((stream memory-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (slot-value stream 'pos))

(defmethod (setf trivial-gray-streams:stream-file-position) (new-offset (stream memory-stream))
  (declare (optimize speed (debug 1)))
  (with-typed-slots ((fixnum pos)
                     (t/memory-stream-buffer buffer))
      stream
    (setf (slot-value stream 'did-unread?) nil)
    (case new-offset
      (:start
       (setf pos 0))
      (:end
       (setf pos (muffling (length buffer))))
      (otherwise
       (cond
         ((and (typep new-offset 'fixnum)
               (>= new-offset 0)
               (<= new-offset (muffling (length buffer))))
          (setf pos new-offset)
          t)
         (t nil))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Gray Stream Interface - Input
;;;

(defmethod trivial-gray-streams:stream-listen ((stream memory-stream))
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    (with-typed-slots ((fixnum pos)
                       (t/memory-stream-buffer buffer))
        stream
      (< pos (muffling (length buffer))))))

(defmethod trivial-gray-streams:stream-read-byte ((stream memory-stream))
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    (with-typed-slots ((fixnum pos)
                       (t/memory-stream-buffer buffer))
        stream
      (if (< pos (muffling (length buffer)))
          (prog1 (muffling (elt buffer pos))
            (setf (slot-value stream 'did-unread?) nil)
            (incf (the fixnum pos)))
          :eof))))

(defmethod trivial-gray-streams:stream-read-sequence ((stream memory-stream) sequence start end &key)
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    (with-typed-slots ((fixnum pos)
                       (t/memory-stream-buffer buffer))
        stream
      (loop with len fixnum = (muffling (length buffer))
            for dest-idx fixnum from start below end
            while (< pos len)
            do (muffling (setf (elt sequence dest-idx) (elt buffer pos)))
               (incf (the fixnum pos))
            finally
               (setf (slot-value stream 'did-unread?) nil)
               (return dest-idx)))))

(defmethod trivial-gray-streams:stream-read-char ((stream memory-stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (let ((ch (trivial-gray-streams:stream-read-byte stream)))
    (declare (type (or t/uint8 (eql :eof)) ch))
    (setf (slot-value stream 'did-unread?) nil)
    (if (eq ch :eof)
        :eof
        (code-char ch))))

(defmethod trivial-gray-streams:stream-unread-char ((stream memory-stream) (character character))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((fixnum pos)
                     (boolean did-unread?))
      stream
    (cond
      (did-unread?
       (%memory-stream-error stream "Cannot UNREAD-CHAR twice"))
      ((not (zerop pos))
       (decf pos)
       (setf did-unread? t))))
  nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Gray Stream Interface - Outut
;;;

(defmethod trivial-gray-streams:stream-write-byte ((stream memory-stream) integer)
  (declare (optimize speed (debug 1)))
  (when (slot-value stream 'read-only?)
    (%read-only-error stream))
  (ensuring-open-stream (stream)
    (unless (typep integer 't/uint8)
      (error "Integer is not a byte"))

    (with-typed-slots ((fixnum pos)
                       (t/memory-stream-buffer buffer))
        stream
      (prog1 integer
        (setf (slot-value stream 'did-unread?) nil)
        (if (< pos (length buffer))
            (muffling (setf (elt buffer pos) integer))
            (vector-push-extend integer buffer))
        (incf (the fixnum pos))))))

(defmethod trivial-gray-streams:stream-write-char ((stream memory-stream) (character character))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (when (slot-value stream 'read-only?)
    (%read-only-error stream))
  (ensuring-open-stream (stream)
    (let ((code (char-code character)))
      (unless (< code 128)
        (%memory-stream-error stream "Cannot write non-ASCII characters to this stream"))
      (setf (slot-value stream 'did-unread?) nil)
      (trivial-gray-streams:stream-write-byte stream code)))
  character)

(defmethod trivial-gray-streams:stream-write-sequence ((stream memory-stream) sequence start end &key)
  (declare (optimize speed (debug 1)))

  #+clisp ;; Clisp will call this for WRITE-STRING
  (when (stringp sequence)
    (return-from trivial-gray-streams:stream-write-sequence
      (trivial-gray-streams:stream-write-string stream sequence start end)))

  (when (slot-value stream 'read-only?)
    (%read-only-error stream))
  (ensuring-open-stream (stream)
    (with-typed-slots ((fixnum pos)
                       (t/memory-stream-buffer buffer))
        stream
      (let ((real-start (or start 0))
            (real-end (or end (muffling (length sequence))))
            (buf-len (muffling (length buffer))))
        (check-type real-start fixnum)
        (check-type real-end fixnum)

        (let ((total-bytes (- real-end real-start)))
          (if (muffling (< (+ pos total-bytes) buf-len))
              ;; Can write without worrying about going off the end, which means
              ;; we can use a simpler loop.  This is a tad bit faster because it
              ;; lacks an extra check.
              (loop for src-idx fixnum from real-start below real-end
                    for element = (muffling (elt sequence src-idx))
                    do (muffling (setf (elt buffer pos) element))
                       (incf (the fixnum pos)))

              ;; We'll eventually go over, so check for that in this loop.  This
              ;; is a tad bit slower because of the extra check.
              (loop for src-idx fixnum from real-start below real-end
                    for element = (muffling (elt sequence src-idx))
                    do (if (< pos buf-len)
                           (muffling (setf (elt buffer pos) element))
                           (vector-push-extend element buffer))
                       (incf (the fixnum pos))))))))
  (setf (slot-value stream 'did-unread?) nil)
  sequence)

(defmethod trivial-gray-streams:stream-write-string ((stream memory-stream) (string string) &optional start end)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (muffling
    (loop for byte of-type t/uint8
            across (babel:string-to-octets (subseq string (or start 0) end)
                                           :encoding :ascii)
          do (cl:write-byte byte stream))
    (setf (slot-value stream 'did-unread?) nil))
  string)

(defmethod trivial-gray-streams:stream-finish-output ((stream memory-stream))
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    t))

(defmethod trivial-gray-streams:stream-clear-output ((stream memory-stream))
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    t))

(defmethod trivial-gray-streams:stream-force-output ((stream memory-stream))
  (declare (optimize speed (debug 1)))
  (ensuring-open-stream (stream)
    t))
