;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/logging)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Main Logging Macros
;;;;
;;;; There are six basic categories of logging, named after their macros:
;;;;  - LOG (normal logging)
;;;;  - VLOG (verbise logging)
;;;;  - DLOG (debug logging)
;;;;  - WARN (warning logging)
;;;;  - ERROR (error logging)
;;;;  - FATAL (error + exit or raise)
;;;;
;;;; Each of these macros take a tag (either a keyword or string), a message,
;;;; and some formatting arguments for the message.
;;;;
;;;; Except for FATAL, there are a few different variations of each of macros,
;;;; denoted by their suffix (e.g. LOG* or DLOG*!)
;;;;  - * (take a LOGGER as an argument)
;;;;  - ! (does not have a tag argument, never muffled)
;;;;  - *! (combines both * and ! together)
;;;;
;;;; FATAL never takes a tag, and has these variations instead:
;;;;  - * (take a LOGGER as an argument)
;;;;  - $ (takes an exit code as well as the message)
;;;;  - *$ (combines both * and $ together)
;;;;
;;;; The tags can be used to muffle certain groups of messages.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defparameter *logger* (make-instance 'logger))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Internal macros
;;;

(defmacro do-log (logger priority tag msg &rest fmt-args)
  "The base macro for all logging.  A message is formatted, then the LOGGER
prints it for the given PRIORITY.  No tag or priority filtering is ever done."
  `(print-message ,logger ,priority ,tag (format nil ,msg ,@fmt-args)))

(defmacro priority-muffled (logger priority)
  `(contains-any? (list ,priority) (slot-value ,logger 'muffle)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Debug Logging
;;;

(defmacro dlog* (logger tag msg &rest fmt-args)
  "Outputs a DEBUG log message unless TAG or DEBUG messages are muffled."
  `(unless (or (find ,tag (the list (slot-value ,logger 'muffle)) :test #'eq)
               (priority-muffled ,logger :debug))
     (do-log ,logger :debug ,tag ,msg ,@fmt-args)))

(defmacro dlog*! (logger msg &rest fmt-args)
  "Outputs a DEBUG log message.  This never muffles a message."
  `(do-log ,logger :debug nil ,msg ,@fmt-args))

(defmacro dlog (tag msg &rest fmt-args)
  "Outputs a DEBUG log message using CL-REMI-LOG:*LOGGER* unless TAG or DEBUG
levels are muffled."
  `(dlog* *logger* ,tag ,msg ,@fmt-args))

(defmacro dlog! (msg &rest fmt-args)
  "Outputs a DEBUG log message using CL-REMI-LOG:*LOGGER*.  This never muffles a
message."
  `(dlog*! *logger* ,msg ,@fmt-args))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Verbose Logging
;;;

(defmacro vlog* (logger tag msg &rest fmt-args)
  "Outputs a VERBOSE log message unless TAG or VERBOSE messages are muffled."
  `(unless (or (find ,tag (the list (slot-value ,logger 'muffle)) :test #'eq)
               (priority-muffled ,logger :verbose))
     (do-log ,logger :verbose ,tag ,msg ,@fmt-args)))

(defmacro vlog*! (logger msg &rest fmt-args)
  "Outputs a VERBOSE log message.  This never muffles a message."
  `(do-log ,logger :verbose nil ,msg ,@fmt-args))

(defmacro vlog (tag msg &rest fmt-args)
  "Outputs a VERBOSE log message using CL-REMI-LOG:*LOGGER* unless TAG or VERBOSE
levels are muffled."
  `(vlog* *logger* ,tag ,msg ,@fmt-args))

(defmacro vlog! (msg &rest fmt-args)
  "Outputs a VERBOSE log message using CL-REMI-LOG:*LOGGER*.  This never muffles a
message."
  `(vlog*! *logger* ,msg ,@fmt-args))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Normal Logging
;;;

(defmacro log* (logger tag msg &rest fmt-args)
  "Outputs an INFO log message unless TAG or INFO messages are muffled."
  `(unless (find ,tag (the list (slot-value ,logger 'muffle)) :test #'eq)
     (do-log ,logger :info ,tag ,msg ,@fmt-args)))

(defmacro log*! (logger msg &rest fmt-args)
  "Outputs an INFO log message.  This never muffles a message."
  `(do-log ,logger :info nil ,msg ,@fmt-args))

(defmacro log (tag msg &rest fmt-args)
  "Outputs an INFO log message using CL-REMI-LOG:*LOGGER* unless TAG or INFO
levels are muffled."
  `(log* *logger* ,tag ,msg ,@fmt-args))

(defmacro log! (msg &rest fmt-args)
  "Outputs an INFO log message using CL-REMI-LOG:*LOGGER*.  This never muffles a
message."
  `(log*! *logger* ,msg ,@fmt-args))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Warning Logging
;;;

(defmacro warn* (logger tag msg &rest fmt-args)
  "Outputs a WARN log message unless TAG or WARN messages are muffled."
  `(unless (find ,tag (the list (slot-value ,logger 'muffle)) :test #'eq)
     (do-log ,logger :warn ,tag ,msg ,@fmt-args)))

(defmacro warn*! (logger msg &rest fmt-args)
  "Outputs a WARN log message.  This never muffles a message."
  `(do-log ,logger :warn nil ,msg ,@fmt-args))

(defmacro warn (tag msg &rest fmt-args)
  "Outputs a WARN log message using CL-REMI-LOG:*LOGGER* unless TAG or WARN
levels are muffled."
  `(warn* *logger* ,tag ,msg ,@fmt-args))

(defmacro warn! (msg &rest fmt-args)
  "Outputs a WARN log message using CL-REMI-LOG:*LOGGER*.  This never muffles a
message."
  `(warn*! *logger* ,msg ,@fmt-args))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Error Logging
;;;

(defmacro error* (logger tag msg &rest fmt-args)
  "Outputs an ERROR log message unless TAG or DEBUG messages are muffled."
  `(unless (find ,tag (the list (slot-value ,logger 'muffle)) :test #'eq)
     (do-log ,logger :error ,tag ,msg ,@fmt-args)))

(defmacro error*! (logger msg &rest fmt-args)
  "Outputs an ERROR log message.  This never muffles a message."
  `(do-log ,logger :error nil ,msg ,@fmt-args))

(defmacro error (tag msg &rest fmt-args)
  "Outputs an ERROR log message using CL-REMI-LOG:*LOGGER* unless TAG or DEBUG
levels are muffled."
  `(error* *logger* ,tag ,msg ,@fmt-args))

(defmacro error! (msg &rest fmt-args)
  "Outputs an ERROR log message using CL-REMI-LOG:*LOGGER*.  This never muffles a
message."
  `(error*! *logger* ,msg ,@fmt-args))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Fatal Logging
;;;

(defmacro fatal*$ (logger exit-code msg &rest fmt-args)
  "Outputs a FATAL log message.  Messages are never muffled.  This then either
raises a LOGGER-FATAL-CONDITION error with the same formatted message and the
given EXIT-CODE, or calls SDM:EXIT with the given exit code, depending on
the value of LOGGER-FATAL-ERROR-P."
  (with-gensyms (str)
    `(let ((,str (do-log ,logger :fatal nil ,msg ,@fmt-args)))
       (if (slot-value ,logger 'fatal-error?)
           (cl:error 'logger-fatal-condition :text ,str :code ,exit-code)
           (exit ,exit-code)))))

(defmacro fatal$ (exit-code msg &rest fmt-args)
  "Outputs a FATAL log message using CL-REMI-LOG:*LOGGER*.  Messages are never
muffled.  This then either raises a LOGGER-FATAL-CONDITION error with the same
formatted message and the given EXIT-CODE, or calls SDM:EXIT with the given
exit code, depending on the value of LOGGER-FATAL-ERROR-P."
  `(fatal*$ *logger* ,exit-code ,msg ,@fmt-args))

(defmacro fatal* (logger msg &rest fmt-args)
  "Outputs a FATAL log message.  Messages are never muffled.  This then either
raises a LOGGER-FATAL-CONDITION error with the same formatted message and an
exit code of 1, or calls SDM:EXIT with an exit code of 1, depending on the
value of LOGGER-FATAL-ERROR-P."
  `(fatal*$ ,logger 1 ,msg ,@fmt-args))

(defmacro fatal (msg &rest fmt-args)
  "Outputs a FATAL log message using CL-REMI-LOG:*LOGGER*.  Messages are never
muffled.  This then either raises a LOGGER-FATAL-CONDITION error with the same
formatted message and an exit code of 1, or calls SDM:EXIT with an exit code
of 1, depending on the value of LOGGER-FATAL-ERROR-P."
  `(fatal$ 1 ,msg ,@fmt-args))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Utilities
;;;

(defmacro indenting ((logger &key (amount 1)) &body forms)
  "Increases the current indent level of LOGGER by AMOUNT temporarily, then
executes FORMS.  After FORMS, the indent level is returned to its previous
value.  Note that AMOUNT means AMOUNT times the INDENT-SIZE of the LOGGER."
  `(unwind-protect
        (progn
          (incf (slot-value ,logger 'indent-level) ,amount)
          ,@forms)
     (decf (slot-value ,logger 'indent-level) ,amount)))

(defmacro muffling ((muffle-these &key (append-these t) (logger *logger*)) &body forms)
  "Temporarily changes which messages are muffled for LOGGER in FORMS.
MUFFLE-THESE must be a list of things things to muffle (specifically a
T/MUFFLE-LIST).  If APPEND-THESE is truthy, then MUFFLE-THESE are appended onto
the current muffled tags/priorities.  Otherwise the current tags/priorities are
replaced.

The old muffled tags/priorities are restored after FORMS."
  (with-gensyms (old-muffles)
    `(let ((,old-muffles (copy-seq (slot-value ,logger 'muffle))))
       (check-type ,muffle-these t/muffle-list)
       (setf (slot-value ,logger 'muffle)
             (if ,append-these
                 (append (slot-value ,logger 'muffle) ,muffle-these)
                 ,muffle-these))
       ,@forms
       (setf (slot-value ,logger 'muffle) ,old-muffles))))

;; (defun as-wrapped (logger str)
;;   "This wraps a string using the LOGGER's indent level, size, and character,
;; then returns the newly wrapped string."
;;   (with-output-to-string (out)
;;     (print-indented-string str (* (slot-value logger 'indent-level)
;;                                   (slot-value logger 'indent-size))
;;                            :stream out
;;                            :indent-char (logger-indent-char logger))))
