;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/logging)

(defgeneric print-header (dest logger priority tag)
  (:documentation "Creates a header string appropriate for the given slot and
writes it to DEST.  This will be called by PRINT-MESSAGE when using the built-in
logging classes.  Returns NIL."))

(defgeneric build-message-string (logger priority tag msg)
  (:documentation "Creates a STRING with the final output message and returns
it.  This will call PRINT-HEADER when used with the built-in logging classes."))

(defgeneric print-message (logger priority tag msg)
  (:documentation "Builds a string containing the appropriate header and MSG for
the given PRIORITY.  This string is then output to the appropriate stream.  This
returns NIL.

When using the built-in classes, this will use BUILD-MESSAGE-STRING to build the
message"))

(defclass logger ()
  (;;;
   ;;; Headers
   ;;;

   (fatal-error?
    :initarg :fatal-error
    :initform t
    :type boolean
    :accessor logger-fatal-error-p
    :documentation "When T, FATAL will raise a LOGGER-FATAL-CONDITION.  When
NIL, calling FATAL will instead call CL-SDM:EXIT.")

   (fatal-header
    :initarg :fatal-header
    :initform "Fatal"
    :type string
    :accessor logger-header/fatal
    :documentation "The header tag that is printed on FATAL messages.")

   (error-header
    :initarg :error-header
    :initform "Error"
    :type string
    :accessor logger-header/error
    :documentation "The header tag that is printed on ERROR messages.")

   (warn-header
    :initarg :warn-header
    :initform "Warning"
    :type string
    :accessor logger-header/warn
    :documentation "The header tag that is printed on WARNING messages.")

   (log-header
    :initarg :log-header
    :initform ""
    :type string
    :accessor logger-header/log
    :documentation "The header tag that is printed on normal LOG messages.")

   (verbose-header
    :initarg :verbose-header
    :initform "Info"
    :type string
    :accessor logger-header/verbose
    :documentation "The header tag that is printed on VERBOSE messages.")

   (debug-header
    :initarg :debug-header
    :initform "Debug"
    :type string
    :accessor logger-header/debug
    :documentation "The header tag that is printed on DEBUG messages.")

   (header-date-format
    :initarg :header-date-format
    :initform nil
    :type (or null string)
    :accessor logger-header-date-format
    :documentation "When non-NIL, all headers will include a timestamp that uses
this as a format string.  Headers will have this format: [<tag> <timestamp>]

See SDM-TIME:DT->STRING for the format of this string.")

   (indent-headers?
    :initarg :indent-headers
    :initform t
    :type boolean
    :accessor logger-indent-headers-p
    :documentation "When truthy, all logging functions will indent the header
when needed.  When NIL, the header is not indented, but the message content
still is.")

   ;;;
   ;;; Streams
   ;;;

   (fatal-stream
    :initarg :fatal-stream
    :initform *error-output*
    :type stream
    :accessor logger-fatal-stream
    :documentation "The output stream for FATAL log messages.")

   (error-stream
    :initarg :error-stream
    :initform *error-output*
    :type stream
    :accessor logger-error-stream
    :documentation "The output stream for ERROR log messages.")

   (warn-stream
    :initarg :warn-stream
    :initform *error-output*
    :type stream
    :accessor logger-warn-stream
    :documentation "The output stream for WARN log messages.")

   (log-stream
    :initarg :log-stream
    :initform *standard-output*
    :type stream
    :accessor logger-log-stream
    :documentation "The output stream for normal log messages.")

   (verbose-stream
    :initarg :verbose-stream
    :initform *standard-output*
    :type stream
    :accessor logger-verbose-stream
    :documentation "The output stream for VERBOSE log messages.")

   (debug-stream
    :initarg :debug-stream
    :initform *error-output*
    :type stream
    :accessor logger-debug-stream
    :documentation "The output stream for DEBUG log messages.")

   ;;;
   ;;; Other Slots
   ;;;

   (ensure-newline?
    :initarg :ensure-newline
    :initform t
    :type boolean
    :accessor logger-ensure-newline-p
    :documentation "When truthy, all logging functions will ensure a newline is
printed after every log message.")

   (muffle
    :initarg :muffle
    :initform '()
    :type t/muffle-list
    :accessor logger-muffle
    :documentation "A T/MUFFLE-LIST of tags that will not be printed by any
logging functions.")

   (show-tag-in-header?
    :initarg :show-tag-in-header
    :initform nil
    :type boolean
    :accessor logger-show-tag-in-header-p
    :documentation "When truthy, headers (when printed) will include the tag.")

   (show-empty-headers?
    :initarg :show-empty-headers
    :initform nil
    :type boolean
    :accessor logger-show-empty-headers-p
    :documentation "When truthy, headers will always be printed, even if the
corresponding header tag is an empty string.")

   (indent-size
    :initarg :indent-size
    :initform 2
    :type (integer 1 *)
    :accessor logger-indent-size
    :documentation "The number of LOGGER-INDENT-CHARs to print when indenting.")

   (indent-char
    :initarg :indent-char
    :initform #\Space
    :type character
    :accessor logger-indent-char
    :documentation "The character to print when indenting.")

   (sanitize?
    :initarg :sanitize
    :initform nil
    :type boolean
    :accessor logger-sanitize-p
    :documentation "Whether or not to call CL-SDM:STRING-SANITIZE on each message
before printing it.")

   (indent-level
    :initform 0
    :type (integer 0 *))

   (sync?
    :initarg :synchronized
    :initform nil
    :type boolean
    :reader logger-synchronized-p))
  (:documentation "The base class for all loggers."))

(defclass color-logger (logger)
  ((fatal-color
    :initarg :fatal-color
    :initform :reg
    :type keyword
    :accessor logger-header-color/fatal
    :documentation "The color of header tag that is printed on FATAL messages.")

   (error-color
    :initarg :error-color
    :initform :red
    :type keyword
    :accessor logger-header-color/error
    :documentation "The color of header tag that is printed on ERROR messages.")

   (warn-color
    :initarg :warn-color
    :initform :yellow
    :type keyword
    :accessor logger-header-color/warn
    :documentation "The color of header tag that is printed on WARNING messages.")

   (log-color
    :initarg :log-color
    :initform :default
    :type keyword
    :accessor logger-header-color/log
    :documentation "The color of header tag that is printed on normal LOG messages.")

   (verbose-color
    :initarg :verbose-color
    :initform :cyan
    :type keyword
    :accessor logger-header-color/verbose
    :documentation "The color of header tag that is printed on VERBOSE messages.")

   (debug-color
    :initarg :debug-color
    :initform :green
    :type keyword
    :accessor logger-header-color/debug
    :documentation "The color of header tag that is printed on DEBUG messages."))
  (:documentation "This is similar to the normal LOGGER class, except that
headers may have colors associated with them.  Header colors are printed using
ANSI escape codes."))

(defclass multi-logger (logger)
  ((extra-streams
    :initarg :extra-streams
    :initform '()
    :type list
    :accessor logger-extra-streams
    :documentation "Additional streams to print output to.  All non-muffled log
    messages will be printed to these streams."))
  (:documentation "This is the same as a normal LOGGER, except that all
non-muffled log messages are also printed to LOGGER-EXTRA-STREAMS."))

(defclass multi-color-logger (color-logger multi-logger)
  ()
  (:documentation "This class effectively combines the COLOR-LOGGER and
MULTI-LOGGER classes.  Note that the ANSI escape codes for the color information
will also be printed to LOGGER-EXTRA-STREAMS."))

(defgeneric (setf logger-synchronized-p) (value logger))
(defmethod (setf logger-synchronized-p) (value (logger logger))
  (setf (slot-value logger 'sync?) (if value t nil)))
