;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/logging)

(defmethod print-header ((dest stream) (logger logger) (priority symbol) (tag string))
  (declare (optimize speed (space 0) (safety 0) (compilation-speed 0)))
  (check-type priority t/log-priority)

  ;; A header is printed if either of these two conditions are true:
  ;;  * One of these are true:
  ;;    - There is not an empty string for the header name
  ;;    - There's an empty string, but show-empty-headers? is truthy
  ;;  * Timestamp is not empty
  (let* ((header-name (slot-value (the logger logger)
                                  (case priority
                                    (:fatal   'fatal-header)
                                    (:error   'error-header)
                                    (:warn    'warn-header)
                                    (:info    'log-header)
                                    (:verbose 'verbose-header)
                                    (:debug   'debug-header))))
         (tag-empty? (empty-string-p tag))
         (header-empty? (empty-string-p header-name)))
    (declare (type string header-name))

    (with-slots ((show-empty show-empty-headers?)
                 (show-tag show-tag-in-header?)
                 (timestamp header-date-format)
                 sanitize?
                 indent-headers? indent-size indent-char indent-level)
        logger
      (declare (type (or null simple-string) timestamp)
               (type fixnum indent-size indent-level)
               (type character indent-char))

      ;; Only output a header when-needed
      (when (or (not header-empty?)
                show-empty
                timestamp)
        ;; Handle header indenting
        (when indent-headers?
          (write-string (make-string (the fixnum (* indent-size indent-level)) :initial-element indent-char)
                        dest))

        (write-char #\[ dest)
        (unless header-empty?
          (write-string (if sanitize?
                            (string-sanitize header-name t)
                            header-name)
                        dest))

        ;; Write the "tag" if we need
        (when (and show-tag (not tag-empty?))
          (unless header-empty?
            (write-char #\- dest))
          (write-string (if sanitize?
                            (string-sanitize tag t)
                            tag)
                        dest))

        (when timestamp
          (when (or (not header-empty?) (and show-tag (not tag-empty?)))
            (write-char #\Space dest))
          (let ((ts-str (sdm-time:dt->string (sdm-time:make-now-datetime
                                              (car (last (multiple-value-list
                                                          (decode-universal-time (get-universal-time))))))
                                             timestamp)))
            (write-string (if sanitize?
                              (string-sanitize ts-str t)
                              ts-str)
                          dest)))

        (write-string "]: " dest)

        ;; Header indenting disabled?
        (unless indent-headers?
          (write-string (make-string (the fixnum (* indent-size indent-level)) :initial-element indent-char)
                        dest)))))
  nil)

(defmethod print-header :around (dest (logger color-logger) (priority symbol) (tag string))
  (declare (optimize speed (space 0) (safety 0) (compilation-speed 0)))
  (check-type priority t/log-priority)
  (let ((color-slot (case priority
                      (:fatal   'fatal-color)
                      (:error   'error-color)
                      (:warn    'warn-color)
                      (:info    'log-color)
                      (:verbose 'verbose-color)
                      (:debug   'debug-color))))
  (sdm-term:ansi/with-color (dest (slot-value (the color-logger logger) color-slot))
    (call-next-method)))
  nil)

(defmethod build-message-string ((logger logger) (priority symbol) tag (msg string))
  (declare (optimize speed (space 0) (safety 0) (compilation-speed 0)))
  (check-type priority t/log-priority)
  (check-type tag (or string symbol null))

  (with-slots (ensure-newline? sanitize?)
      logger
    (with-output-to-string (out)
      (let ((has-newline (string-ends-with msg (string #\Newline))))
        (print-header out logger priority (if tag (write-to-string tag :escape nil) ""))
        (if sanitize?
            (if has-newline
                (write-string (string-sanitize (subseq msg 0 (1- (length msg))) t) out)
                (write-string (string-sanitize msg t) out))
            (write-string msg out))
        (when ensure-newline?
          (fresh-line out))))))

(defmethod print-message ((logger logger) (priority symbol) tag (msg string))
  (declare (optimize speed (space 0) (safety 0) (compilation-speed 0)))
  (check-type priority t/log-priority)
  (check-type tag (or string symbol null))

  (let ((final-str (build-message-string logger priority tag msg))
        (dest (slot-value (the logger logger)
                          (case priority
                            (:fatal   'fatal-stream)
                            (:error   'error-stream)
                            (:warn    'warn-stream)
                            (:info    'log-stream)
                            (:verbose 'verbose-stream)
                            (:debug   'debug-stream)))))
    (declare (type simple-string final-str)
             (type stream dest))
    (write-string final-str dest)
    nil))

(defmethod print-message ((logger multi-logger) priority tag (msg string))
  (declare (optimize speed (space 0) (safety 0) (compilation-speed 0)))
  (check-type priority t/log-priority)
  (check-type tag (or string symbol null))

  (let ((final-str (build-message-string logger priority tag msg))
        (dest (slot-value (the logger logger)
                          (case priority
                            (:fatal   'fatal-stream)
                            (:error   'error-stream)
                            (:warn    'warn-stream)
                            (:info    'log-stream)
                            (:verbose 'verbose-stream)
                            (:debug   'debug-stream)))))
    (declare (type simple-string final-str)
             (type stream dest))
    (write-string final-str dest)

    (dolist (strm (slot-value logger 'extra-streams))
      (write-string final-str strm)))
  nil)

(defmethod print-message :after ((logger logger) priority tag msg)
  (declare (ignore priority tag msg))
  (when (logger-synchronized-p logger)
    (logger-flush logger)))

(defmethod logger-flush ((logger logger))
  (declare (optimize (speed 3) (debug 1) (compilation-speed 0) (safety 1)))
  (macrolet ((flush (thing)
               `(when (open-stream-p ,thing)
                  (finish-output ,thing))))
    (flush (logger-fatal-stream logger))
    (flush (logger-error-stream logger))
    (flush (logger-warn-stream logger))
    (flush (logger-log-stream logger))
    (flush (logger-verbose-stream logger))
    (flush (logger-debug-stream logger)))
  t)

(defmethod logger-flush :around ((logger multi-logger))
  (declare (optimize (speed 3) (debug 1) (compilation-speed 0) (safety 1)))
  (when (next-method-p)
    (call-next-method))
  (dolist (strm (logger-extra-streams logger))
    (when (open-stream-p strm)
      (finish-output strm))))
