;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2023 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or
;;;; modify it under the terms of the GNU Affero General Public
;;;; License as published by the Free Software Foundation, either
;;;; version 3 of the License, or (at your option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful,
;;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;; Affero General Public License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/logging)

;; Export new symbols
(eval-when (:compile-toplevel :load-toplevel)
  (dolist (sym '(concurrent-logger
                 concurrent-color-logger
                 concurrent-multi-logger
                 concurrent-multi-color-logger
                 start-logger
                 stop-logger
                 with-concurrent-logger
                 logger-running-p
                 logger-drain))
    (export sym :cl-sdm/logging)))

(deftype t/concurrent-logger-state ()
  '(member :stopped :drain :running :stopping))

(defclass concurrent-logger (logger)
  ((logger-queue
    :initform (lparallel.queue:make-queue)
    :type lparallel.queue:queue)

   (paused-message-queue
    :initform (lparallel.queue:make-queue)
    :type lparallel.queue:queue)

   (logging-thread
    :initform nil)

   (state
    :initform :stopped
    :type t/concurrent-logger-state))
  (:documentation "This is the same as a normal LOGGER, but the actual message
assembly and printing is handled concurrently."))

(defclass concurrent-color-logger (concurrent-logger color-logger)
  ()
  (:documentation "This is the same as a normal COLOR-LOGGER, but the actual
message assembly and printing is handled concurrently."))

(defclass concurrent-multi-logger (concurrent-logger multi-logger)
  ((extra-streams-mutex
    :initform (bt:make-lock "Concurrent logger extra streams mutex")
    :type bt:lock))
  (:documentation "This is the same as a normal MULTI-LOGGER, but the actual
message assembly and printing is handled concurrently."))

(defclass concurrent-multi-color-logger (color-logger concurrent-multi-logger)
  ()
  (:documentation "This is the same as a normal MULTI-COLOR-LOGGER, but the
actual message assembly and printing is handled concurrently."))

(defgeneric start-logger (logger)
  (:documentation "Starts a concurrent logger.  If this is not called, no output
from the logger gets printed."))

(defgeneric stop-logger (logger)
  (:documentation "Stops a concurrent logger.  Once called, printing messages
with it will be ignored and not queued until started again.  Messages that are
currently queued are flushed when this is called."))

(defgeneric logger-running-p (logger)
  (:documentation "Returns T if the given concurrent logger is running, or NIL
otherwise."))

(defmethod logger-running-p ((logger concurrent-logger))
  (declare (optimize (speed 3) (debug 1)))
  (or (eq (slot-value logger 'state) :running)
      (eq (slot-value logger 'state) :drain)))

(defmethod print-message :around ((logger concurrent-logger) (priority symbol) tag (msg string))
  (declare (optimize (speed 3) (debug 1)))
  (check-type priority t/log-priority)
  (check-type tag (or null keyword string))
  (with-typed-slots ((lparallel.queue:queue logger-queue paused-message-queue)
                     (t/concurrent-logger-state state)
                     logging-thread)
      logger
    (case state
      (:running
       (when logging-thread
         (lparallel.queue:push-queue (list :priority priority
                                           :tag tag
                                           :msg msg)
                                     logger-queue)
         (when (logger-synchronized-p logger)
           (lparallel.queue:push-queue :sync logger-queue))))
      (:drain
       (when logging-thread
         (lparallel.queue:push-queue (list :priority priority
                                           :tag tag
                                           :msg msg)
                                     paused-message-queue)
         (when (logger-synchronized-p logger)
           (lparallel.queue:push-queue :sync paused-message-queue))))))
  nil)

(defun %concurrent-logger-dispatch-cmd (logger cmd)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-typed-slots ((lparallel.queue:queue logger-queue paused-message-queue)
                     (t/concurrent-logger-state state))
      logger
    (etypecase cmd
      (keyword
       (ecase cmd
         (:drain
          ;; Drain the logger queue until nothing else is left.
          (loop for cmd = (lparallel.queue:try-pop-queue logger-queue)
                while cmd
                do (when (%concurrent-logger-dispatch-cmd logger cmd)
                     (return-from %concurrent-logger-dispatch-cmd t)))

          ;; Now copy things from the paused queue back to the normal queue.
          (loop for cmd = (lparallel.queue:try-pop-queue paused-message-queue)
                while cmd do (lparallel.queue:push-queue cmd logger-queue)))

         (:stop-logger (return-from %concurrent-logger-dispatch-cmd t))
         (:sync (logger-flush logger))))

      (list
       (let ((priority (getf cmd :priority))
             (tag (getf cmd :tag))
             (msg (getf cmd :msg)))
         (if (and (typep priority 't/log-priority)
                  (typep tag '(or keyword symbol))
                  (typep msg 'string))
             (let ((final-str (build-message-string logger priority tag msg))
                   (dest (slot-value (the logger logger)
                                     (case priority
                                       (:fatal   'fatal-stream)
                                       (:error   'error-stream)
                                       (:warn    'warn-stream)
                                       (:info    'log-stream)
                                       (:verbose 'verbose-stream)
                                       (:debug   'debug-stream)))))
               (declare (type simple-string final-str)
                        (type stream dest))
               (write-string final-str dest)

               (when (typep logger 'concurrent-multi-logger)
                 (bt:with-lock-held ((slot-value logger 'extra-streams-mutex))
                   (dolist (strm (slot-value logger 'extra-streams))
                     (when (open-stream-p strm)
                       (write-string final-str strm))
                     (when (and (logger-synchronized-p logger)
                                (open-stream-p strm))
                       (finish-output strm))))))

             ;; At least one of priority/tag/message was a bad type
             (cl:warn "A CONCURRENT-LOGGER received a bad message"))))))
  nil)

(defun concurrent-logger-check-loop (logger)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type logger concurrent-logger)
  (with-typed-slots ((lparallel.queue:queue logger-queue)) logger
    (loop for cmd = (lparallel.queue:pop-queue logger-queue)
          do (when (%concurrent-logger-dispatch-cmd logger cmd)
               (loop-finish))))
  nil)

(defmethod logger-flush :around ((logger concurrent-multi-logger))
  (declare (optimize (speed 3) (debug 1) (compilation-speed 0) (safety 1)))
  (when (next-method-p)
    (bt:with-lock-held ((slot-value logger 'extra-streams-mutex))
      (call-next-method))))

(defmethod start-logger ((logger concurrent-logger))
  (with-typed-slots ((lparallel.queue:queue logger-queue)
                     (t/concurrent-logger-state state)
                     logging-thread)
      logger
    (when logging-thread
      (cl:error "Logger already started"))

    (setf state :running)
    (setf logging-thread (bt:make-thread
                          #'(lambda ()
                              (concurrent-logger-check-loop logger))
                          :name "Concurrent logger thread")))
  t)

(defmethod stop-logger ((logger concurrent-logger))
  (with-slots (logger-queue logging-thread state)
      logger
    (assert (bt:threadp logging-thread))
    (setf state :stopping)
    (lparallel.queue:push-queue :drain logger-queue)
    (lparallel.queue:push-queue :stop-logger logger-queue)
    (loop while (bt:thread-alive-p logging-thread)
          do (sleep #.(/ 1 100.0)))
    (setf state :stopped)
    (logger-flush logger)
    (setf logging-thread nil)
    t))

(defmacro with-concurrent-logger ((logger) &body forms)
  `(unwind-protect
        (progn
          (start-logger ,logger)
          ,@forms)
     (when (and (typep ,logger 'concurrent-logger)
                (eq (slot-value ,logger 'state) :running))
       (stop-logger ,logger))))

(defmethod logger-drain ((logger concurrent-logger))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (case (slot-value logger 'state)
    (:running
     (lparallel.queue:push-queue :drain (slot-value logger 'logger-queue))
     (loop until (lparallel.queue:queue-empty-p
                  (slot-value logger 'paused-message-queue))
           do (sleep 0.01))))
  nil)
