;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; I/O, String, and Byte Utilities
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defgeneric read-line* (stream byte-limit &optional recursive-p)
  (:documentation "Same as READ-LINE, but only reads up to LIMIT elements of data from STREAM."))

(defmacro with-stream-excursion ((stream go-to) &body forms)
  (with-gensyms (return-to)
    `(let ((,return-to (file-position ,stream)))
       (unwind-protect
            (progn
              (file-position ,stream ,go-to)
              ,@forms)
         (file-position ,stream ,return-to)))))

(define-typed-fn %bytes->string/list ((list byte-list))
    (simple-string t)
  "Converts a LIST of bytes to a string, optionally keeping any null
characters."
  (declare (optimize (speed 3) (safety 0) (debug 0) (compilation-speed 0)))
  (coerce (with-output-to-string (out)
            (dolist (byte byte-list)
              (write-char (code-char byte) out)))
          'simple-string))

(define-typed-fn %bytes->string/vector ((vector byte-list))
    (simple-string t)
  "Converts a LIST of bytes to a string, optionally keeping any null
characters."
  (declare (optimize (speed 3) (safety 0) (debug 0) (compilation-speed 0)))
  (coerce (the string (babel:octets-to-string byte-list :errorp nil))
          'simple-string))

(declaim (ftype (function ((or list vector) &optional t) simple-string) bytes->string))
(defun bytes->string (byte-list &optional keep-nulls)
  "Converts a LIST or VECTOR of bytes to a string, optionally keeping any null
characters."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
    (let ((ret (etypecase byte-list
                 (list (%bytes->string/list byte-list))
                 (vector (%bytes->string/vector byte-list)))))
      (if keep-nulls
          ret
          (coerce (remove #\Nul ret :test 'char=) 'simple-string))))

(define-typed-fn bytes->uint ((sequence bytes))
    ((or t/uint64 null))
  "Converts a set of bytes stored in a LIST or VECTOR into an unsigned 64-bit
integer.  The byte order is assumed to be little endian."
  (declare (optimize (speed 3) (safety 1) (debug 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (when (= (length bytes) 0)
    (return-from bytes->uint nil))

  (let ((ret 0) (byte 0) (pos 0) (len 0))
    (declare (type t/uint64 len ret)
             (type t/uint8 byte)
             (dynamic-extent byte pos len))
    (setf len (length bytes))

    (dotimes (i len)
      (setf byte (elt bytes i))
      (setf ret (dpb byte (byte 8 pos) ret))
      (incf pos 8))

    ret))

(define-typed-fn bytes->uint! ((sequence bytes))
    ((or integer null))
  "Converts a set of bytes stored in a LIST or VECTOR into an unsigned integer.
The byte order is assumed to be little endian.  Unlike BYTES->UINT, this version
will return a BIGNUM if needed."
  (declare (optimize speed (safety 1) (debug 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (when (= (length bytes) 0)
    (return-from bytes->uint! nil))

  (let ((ret 0) (byte 0) (pos 0) (len 0))
    (declare (type integer len ret)
             (type t/uint8 byte))
    (setf len (length bytes))
    (dotimes (i len)
      (setf byte (elt bytes i))
      (setf ret (dpb byte (byte 8 pos) ret))
      (incf pos 8))

    ret))

(declaim (ftype (function ((integer 0 *) &optional (or null fixnum)) list) uint->bytes))
(defun uint->bytes (int &optional num-bytes)
  "Converts an unsigned integer into a set of bytes."
  (declare (type (integer 0 *) int)
           (type (or null fixnum) num-bytes)
           (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (if (= int 0)
      (loop for i fixnum from 1 to (or num-bytes 1) collect 0)
      (loop for pos fixnum from 0 by 8
            for i fixnum from 1 to (or num-bytes (num-8bit-bytes-needed int))
            collect (ldb (byte 8 pos) int))))

(define-typed-fn %uint->byte-vector/num-bytes-is-fixnum (((integer 0 *) int) (t/ufixnum num-bytes))
    (t/uint8-array t)
  "Variation opf UINT->BYTE-VECTOR where NUM-BYTES is known to be a T/UFIXNUM, but
INT may be a bignum."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (loop for pos of-type (integer 0 *) from 0 by 8
        for i of-type t/ufixnum from 0 below num-bytes
        with ret of-type t/uint8-array = (make-array num-bytes :element-type 't/uint8
                                                               :initial-element 0)
        do (setf (aref ret i) (ldb (byte 8 pos) int))
        finally (return ret)))

(define-typed-fn %uint->byte-vector/int-is-ufixnum ((t/ufixnum int) ((or null t/ufixnum) num-bytes))
    (t/uint8-array t)
  "Variation opf UINT->BYTE-VECTOR where INT is known to be a T/UFIXNUM."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (if (= int 0)
      (make-array (or num-bytes 1) :element-type 't/uint8 :initial-element 0)

      (let ((bytes-needed (or num-bytes (num-8bit-bytes-needed int))))
        (declare (type t/ufixnum bytes-needed))

        (loop for pos of-type t/ufixnum from 0 by 8
              for i of-type t/ufixnum from 0 below bytes-needed
              with ret of-type t/uint8-array = (make-array bytes-needed :element-type 't/uint8
                                                                        :initial-element 0)
              do (setf (aref ret i) (ldb (byte 8 pos) int))
              finally (return ret)))))

(define-compiler-macro uint->byte-vector (&whole form int &optional num-bytes &environment env)
  (declare (ignorable int num-bytes env))
  (cond
    #+sbcl
    ((or (and (constantp int)
              (typep int 't/ufixnum))
         (and (symbolp int)
              (%var-is-type env int t/ufixnum)))
     `(%uint->byte-vector/int-is-ufixnum ,int ,num-bytes))

    ((and (constantp num-bytes)
          (typep num-bytes 't/ufixnum))
     `(%uint->byte-vector/num-bytes-is-fixnum ,int ,num-bytes))

    (t form)))

(declaim (ftype (function ((integer 0 *) &optional (or null fixnum)) t/uint8-vector) uint->byte-vector))
(defun uint->byte-vector (int &optional num-bytes)
  "Converts an unsigned integer into a set of bytes."
  (declare (type (integer 0 *) int)
           (type (or null t/ufixnum) num-bytes)
           (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (if (= int 0)
      (make-array (or num-bytes 1) :element-type 't/uint8 :initial-element 0)

      (let ((bytes-needed 0))
        (setf bytes-needed (or num-bytes (num-8bit-bytes-needed int)))

        (loop for pos of-type (integer 0 *) from 0 by 8
              for i of-type (integer 0 *) from 0 below bytes-needed
              with ret of-type t/uint8-array = (make-array bytes-needed :element-type 't/uint8
                                                                        :initial-element 0)
              do (setf (aref ret i) (ldb (byte 8 pos) int))
              finally (return ret)))))

(define-typed-fn bytes->dfloat ((sequence bytes))
    ((or double-float t/special-floats) t)
  "Converts BYTES into a DOUBLE-FLOAT or a T/SPECIAL-FLOATS.  The length of
BYTES must be exactly 8."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (unless (= (muffling (length bytes)) 8)
    (error "BYTES->DFLOAT requires a sequence of exactly 8 bytes."))
  (uint64->dfloat (bytes->uint bytes)))

(define-typed-fn bytes->sfloat ((sequence bytes))
    ((or single-float t/special-floats) t)
  "Converts BYTES into a SINGLE-FLOAT or a T/SPECIAL-FLOATS.  The length of
BYTES must be exactly 4."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (unless (= (muffling (length bytes)) 4)
    (error "BYTES->SFLOAT requires a sequence of exactly 4 bytes."))
  (uint32->sfloat (bytes->uint bytes)))

(define-typed-fn bytes->float ((sequence bytes))
    ((or double-float single-float t/special-floats) t)
  "Converts BYTES into a SINGLE-FLOAT, DOUBLE-FLOAT, or a T/SPECIAL-FLOATS,
depending on its length.  The length of BYTES must be exactly 4 or 8."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (case (muffling (length bytes))
    (4 (uint32->sfloat (bytes->uint bytes)))
    (8 (uint64->dfloat (bytes->uint bytes)))
    (otherwise (error "BYTES->FLOAT requires a sequence of exactly 4 or 8 bytes."))))

(define-typed-fn float->bytes (((or double-float single-float) value))
    (list t)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  "Converts VALUE into a LIST of bytes.  VALUE can be a DOUBLE-FLOAT or a
SINGLE-FLOAT.  The length of the returned list will be either 8 or 4 bytes,
depending on the type of VALUE."
  (etypecase value
    (double-float (muffling (uint->bytes (dfloat->uint64 value) 8)))
    (single-float (uint->bytes (sfloat->uint32 value) 4))))

(define-typed-fn float->byte-vector (((or double-float single-float) value))
    (t/uint8-array t)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  "Converts VALUE into a LIST of bytes.  VALUE can be a DOUBLE-FLOAT or a
SINGLE-FLOAT.  The length of the returned vector will be either 8 or 4 bytes,
depending on the type of VALUE."
  (etypecase value
    (double-float (muffling (uint->byte-vector (dfloat->uint64 value) 8)))
    (single-float (uint->byte-vector (sfloat->uint32 value) 4))))

(define-typed-fn sfloat->bytes (((or single-float t/special-floats) value))
    (list t)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  "Converts VALUE into a LIST of exactly 4 bytes.  VALUE can be a SINGLE-FLOAT or a
T/SPECIAL-FLOATS."
  (uint->bytes (sfloat->uint32 value) 4))

(define-typed-fn dfloat->bytes (((or double-float t/special-floats) value))
    (list t)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  "Converts VALUE into a LIST of exactly 8 bytes.  VALUE can be a DOUBLE-FLOAT or a
T/SPECIAL-FLOATS."
  (uint->bytes (dfloat->uint64 value) 8))

(define-typed-fn sfloat->byte-vector (((or single-float t/special-floats) value))
    ((t/uint8-array 4) t)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  "Converts VALUE into a T/UINT8-VECTOR of exactly 4 bytes.  VALUE can be a
SINGLE-FLOAT or a T/SPECIAL-FLOATS."
  (uint->byte-vector (sfloat->uint32 value) 4))

(define-typed-fn dfloat->byte-vector (((or double-float t/special-floats) value))
    ((t/uint8-array 8) t)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  "Converts VALUE into a T/UINT8-VECTOR of exactly 8 bytes.  VALUE can be a
DOUBLE-FLOAT or a T/SPECIAL-FLOATS."
  (uint->byte-vector (dfloat->uint64 value) 8))

(declaim (ftype (function (string &key (:as-vector t) (:pad-length (or null t/uint64))
                                  (:pad-char character))
                          (or list vector))
                string->bytes))
(defun string->bytes (str &key as-vector pad-length (pad-char #\Nul))
  "Converts a string stored in STR to a set of bytes.  If AS-VECTOR is non-NIL,
then the result is returned as a vector.  Otherwise, the result is returned as a
list.

If PAD-LENGTH is non-NIL, then STR is first expanded to a length of PAD-LENGTH,
adding PAD-CHAR as necessary.  If PAD-STRING is non-NIL, and STR is already
PAD-LENGTH characters or longer, no padding occurs."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))

  (let* ((ret-type (if as-vector 'vector 'list))
         (ret (if as-vector
                  (locally
                      (declare #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
                    (map 'vector #'char-code str))
                  (map 'list #'char-code str)))
         (ret-len 0))
    (declare (dynamic-extent ret-type ret-len))
    (locally
        (declare #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
      (setf ret-len (length ret)))
    (if  (and pad-length (< ret-len pad-length))
         (concatenate ret-type
                      ret
                      (loop for i from ret-len below pad-length
                            collect (char-code pad-char)))
         ret)))

(defun read-bytes (stream num-bytes-to-read &key as-list (buffer-size *io-buffer-size*))
  "Reads the specified number of bytes from STREAM, and returns a new sequence
of bytes.  The return value is a vector unless AS-LIST is non-NIL, in which case
this returns a list instead.

The returned sequence may be smaller than NUM-BYTES-TO-READ if there are not
enough bytes left in the stream."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type num-bytes-to-read fixnum)
  (check-type buffer-size fixnum)

  (let* ((buf-len (if (< num-bytes-to-read buffer-size)
                      num-bytes-to-read
                      buffer-size))
         (buf (new-array buf-len t/uint8))
         (ret (new-array 0 t/uint8)))
    (declare (type (vector (unsigned-byte 8) *) buf ret))

    (loop with num-read fixnum = 0
          for pos fixnum = (read-sequence buf stream)
          for i fixnum from 1
          while (and (plusp pos)
                     (< num-read num-bytes-to-read))
          do (setf ret (concatenate '(vector (unsigned-byte 8))
                                    ret
                                    (subseq buf 0 pos)))
             (incf num-read pos)

             ;; Will we read too far the next loop?
             (when (> (+ num-read buf-len) num-bytes-to-read)
               (setf buf (subseq buf 0 (muffling (- num-bytes-to-read num-read))))
               (setf pos (read-sequence buf stream))
               (setf ret (concatenate '(vector (unsigned-byte 8))
                                      ret
                                      (subseq buf 0 pos)))
               (loop-finish)))

    (if as-list
        (coerce ret 'list)
        (coerce ret 't/uint8-array))))

(define-compiler-macro read-bytes (&whole form stream num-bytes-to-read &key as-list (buffer-size cl-sdm:*io-buffer-size*))
  (if (and (constantp num-bytes-to-read)
           (typep num-bytes-to-read '(and unsigned-byte fixnum)))
      (let ((%buf-len nil)
            (%buffer-size nil))
        ;; Can we determine the buffer size ahead of time?
        (when (and (constantp buffer-size)
                   (typep buffer-size '(and unsigned-byte fixnum)))
          ;; We can!
          (setf %buffer-size buffer-size)
          (setf %buf-len (if (< num-bytes-to-read %buffer-size) num-bytes-to-read %buffer-size)))

        ;; Possibly emit a single READ-SEQUENCE if we have a constant
        ;; %BUFFER-SIZE and the number of bytes to read can fit within a single
        ;; buffer.
        (if (and %buffer-size
                 (<= num-bytes-to-read %buffer-size))
            `(let* ((ret (new-array ,%buf-len t/uint8))
                    (pos (read-sequence ret ,stream)))
               (declare (type t/uint8-array ret)
                        (type fixnum pos))
               (when (< pos ,%buf-len)
                 (setf ret (coerce (subseq ret pos) 't/uint8-array)))

               ;; We may be able to skip a check here if AS-LIST is a constant.
               ,(if (and (constantp as-list) as-list)
                    `(coerce ret 'list)
                    ;; Emit the normal check
                    `(if ,as-list
                         (coerce ret 'list)
                         ret)))

            ;; We have too much to read for a single buffer, but we can
            ;; probably still emit some optimized code.
            `(let* (,@(let ((forms ()))
                        ;; Did we determine a buffer size ahead of time?  If we did,
                        ;; then we don't emit a BUF-LEN variable here.
                        (unless %buf-len
                          ;; Emit the normal BUF-LEN variable
                          (setf forms (append forms (list `(buf-len (if (< ,num-bytes-to-read ,buffer-size)
                                                                        ,num-bytes-to-read
                                                                        ,buffer-size))))))

                        ;; If we determined the buffer size ahead of time, use that size.
                        (setf forms (append forms (list `(buf (new-array ,(or %buf-len `buf-len) t/uint8)))))
                        (setf forms (append forms (list `(ret (new-array 0 t/uint8)))))

                        forms))
               (declare (type (vector (unsigned-byte 8) *) buf ret))

               (loop with num-read fixnum = 0
                     for pos fixnum = (read-sequence buf ,stream)
                     for i fixnum from 1
                     while (and (plusp pos) (< num-read ,num-bytes-to-read))
                     do (setf ret (concatenate '(vector (unsigned-byte 8)) ret (subseq buf 0 pos)))
                        (incf num-read pos)

                        ;; Will we read too far the next loop?
                        (when (> (+ num-read ,(or %buf-len `buf-len)) ,num-bytes-to-read)
                          (setf buf (subseq buf 0 (muffling (- ,num-bytes-to-read num-read))))
                          (setf pos (read-sequence buf ,stream))
                          (setf ret (concatenate '(vector (unsigned-byte 8)) ret (subseq buf 0 pos)))
                          (loop-finish)))

               ;; We may be able to skip a check here if AS-LIST is a constant.
               ,(if (and (constantp as-list) as-list)
                    `(coerce ret 'list)
                    ;; Emit the normal check
                    `(if ,as-list
                         (coerce ret 'list)
                         (coerce ret 't/uint8-array))))))

      ;; Emit the entire original form
      form))

(defun read-string (stream num-bytes-to-read &key trim-nulls (buffer-size *io-buffer-size*))
  "Reads the specified number of bytes from STREAM, and returns a string where
each byte is converted into a character.  If TRIM-NULLS is non-NIL, then the
null bytes in the string are trimmed off the end before returning it.
Additionally, TRIM-NULLS can be :FROM-FIRST, which will trim all data starting
from the first null character found.

The returned string may be smaller than NUM-BYTES-TO-READ if there are not
enough bytes left in the stream."
  (declare (type fixnum num-bytes-to-read buffer-size)
           (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (let ((vec (read-bytes stream num-bytes-to-read :buffer-size buffer-size)))
    (cond
      ((not trim-nulls)
       (bytes->string vec t))

      ((not (equal trim-nulls :from-first))
       (bytes->string vec nil))

      (t
       (let* ((ret (bytes->string vec t))
              (pos nil))
         (declare (dynamic-extent pos))
         (setf pos (the (or fixnum null) (position #\Nul (the simple-string ret) :test #'char=)))
         (if pos
             (subseq ret 0 pos)
             ret))))))

(defun read-null-term-string (stream &optional max-bytes-to-read)
  "Reads the specified number of bytes from STREAM, and returns a
string where each byte is converted into a character.  If TRIM-NULLS
is non-NIL, then the null bytes in the string are trimmed off the end
before returning it.  Additionally, TRIM-NULLS can be :FROM-FIRST,
which will trim all data starting from the first null character
found."
  (declare (type (or null fixnum) max-bytes-to-read)
           (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (loop for byte = (read-byte stream nil nil)
        for num-read fixnum from 1
        with ret simple-string = "" do
          (cond
            ((not byte)
             (loop-finish))

            ((and max-bytes-to-read (= num-read max-bytes-to-read))
             (loop-finish))

            ((= byte 0)
             (loop-finish))

            (t (setf ret (strings+ ret (code-char byte)))))
        finally (return ret)))

(defmacro if-read-string ((stream string-to-expect actual &key case-insensitive)
                          if-found if-not-found)
  "Reads a string from STREAM that is the same length as STRING-TO-EXPECT and
binds ACTUAL to that string.  If the string that was read is equal to
STRING-TO-EXPECT, then the IF-FOUND form is executed, otherwise IF-NOT-FOUND is
executed.  The comparison is case-sensitive unless CASE-INSENSITIVE is truthy."
  `(let ((,actual (read-string ,stream (length ,string-to-expect))))
     (if (or (and ,case-insensitive
                  (caseless-string= ,actual ,string-to-expect))
             (string= ,actual ,string-to-expect))
         ,if-found
         ,if-not-found)))

(trivial-indent:define-indentation if-read-string (4 2 2))

(defmacro expect-string ((stream string-to-expect actual &key case-insensitive) &body forms)
  "Reads a string from STREAM that is the same length as STRING-TO-EXPECT and
binds ACTUAL to that string.  If the string that was read is *NOT* equal to
STRING-TO-EXPECT, then FORMS are executed and the last value of FORMS is
returned, otherwise this does nothing except return T."
  `(if-read-string (,stream ,string-to-expect ,actual :case-insensitive ,case-insensitive)
     t
     (progn ,@forms)))

(defmethod read-line* ((stream stream) (limit integer) &optional recursive-p)
  "Same as READ-LINE, but only reads up to LIMIT characters from STREAM.

The newline, if encountered, is read from the stream, but is not counted towards
and is not included in the returned value."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (assert (typep limit 't/ufixnum))

  (with-output-to-string (out)
    (let ((remaining limit)
          (char nil))
      (declare (type (integer -1 #.most-positive-fixnum) remaining)
               (type (or null character) char))
      (loop do
        (when (setf char (read-char stream nil nil recursive-p))
          (when (char= char #\Newline)
            (loop-finish))
          (write-char char out)
          (when (zerop (decf remaining))
            (loop-finish)))))))

(define-typed-fn %read-bytes-into-uint (stream (fixnum num-bytes-to-read buffer-size))
    (unsigned-byte t)
  "Reads NUM-BYTES-TO-READ bytes from STREAM, then converts this into an
unsigned integer."
  (declare (optimize (speed 3) (debug 0) (safety 0) (compilation-speed 0)))
  (bytes->uint (read-bytes stream num-bytes-to-read :buffer-size buffer-size)))

(define-typed-fn %read-bytes-into-int (stream (fixnum num-bytes-to-read buffer-size))
    (integer t)
  "Reads NUM-BYTES-TO-READ bytes from STREAM, then converts this into a signed
integer."
  (declare (optimize (speed 3) (debug 0) (safety 0) (compilation-speed 0)))
  (uint->int/2c (bytes->uint (read-bytes stream num-bytes-to-read :buffer-size buffer-size))
                (muffling (* 8 num-bytes-to-read))))

(declaim (ftype (function (T fixnum &key (:unsigned T) (:buffer-size fixnum)) integer) read-bytes-into-int)
         (inline read-bytes-into-int))
(defun read-bytes-into-int (stream num-bytes-to-read &key unsigned (buffer-size *io-buffer-size*))
  "Reads NUM-BYTES-TO-READ bytes from STREAM, then converts this into a signed
integer (or an unsigned integer of UNSIGNED is non-NIL) and returns the number."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0) (space 0)))
  (if unsigned
      (%read-bytes-into-uint stream num-bytes-to-read buffer-size)
      (%read-bytes-into-int stream num-bytes-to-read buffer-size)))

(declaim (ftype (function (T sequence &optional (or null fixnum) fixnum) null) write-bytes))
(defun write-bytes (stream bytes &optional max-num-bytes (buffer-size *io-buffer-size*))
  "Writes a sequence of bytes to STREAM, buffering the output using
BUFFER-SIZE.

When MAX-NUM-BYTES is non-NIL, then at most that many bytes will be written to
the stream."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))

  (let ((blen 0)
        (real-bytes bytes))
    (declare (type fixnum blen)
             (dynamic-extent blen))
    (setf blen (length bytes))

    ;; Cut off any bytes that are beyond MAX-NUM-BYTES
    (when (and max-num-bytes (> blen max-num-bytes))
      (setf real-bytes (subseq real-bytes 0 (1- max-num-bytes))))

    (cond
      ;; If the number of real-bytes is less than BUFFER-SIZE, just write them
      ;; out all at once.
      ((<= blen buffer-size)
       (write-sequence real-bytes stream))

      ;; Otherwise, write it out in bulk blocks
      (t
       (loop for i fixnum from 0 below blen by buffer-size
             for buf = (subseq real-bytes i
                               ;; This makes sure we don't go past the end of the buffer
                               (if  (<= blen (+ i buffer-size))
                                    blen
                                    (+ i buffer-size)))
             do (write-sequence buf stream)))))
  nil)

(declaim (ftype (function (T string &key
                             (:pad-length (or null t/uint64))
                             (:pad-char character)
                             (:buffer-size fixnum))
                          null)
                write-string)
         (inline write-string))
(defun write-string (stream str &key pad-length (pad-char #\Nul) (buffer-size *io-buffer-size*))
  "Writes a string stored in STR out to STREAM.  If PAD-LENGTH is non-NIL, then
STR is first expanded to a length of PAD-LENGTH, adding PAD-CHAR as necessary.
If PAD-STRING is non-NIL, and STR is already PAD-LENGTH characters or longer, no
padding occurs.  Output is buffered using BUFFER-SIZE."
  (declare (optimize speed (debug 1) (safety 1) (space 0) (compilation-speed 0)))
  (let ((bytes (string->bytes str :pad-length pad-length
                                  :pad-char pad-char)))
    (declare (type list bytes))
    (write-bytes stream bytes buffer-size))
  nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Int reading/writing
;;;

(declaim (ftype (function (T integer &key (:max-width fixnum) (:min-width fixnum) (:buffer-size fixnum)) null)
                write-uint)
         (inline write-uint))
(defun write-uint (stream num &key max-width min-width
                                (buffer-size *io-buffer-size*))
  "Writes an INTEGER stored in NUM out to STREAM.  The number is always written
as an unsigned integer.  If MIN-WIDTH is non-NIL, then additional null bytes are
written as necessary.  Output is buffered using BUFFER-SIZE."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (write-bytes stream (uint->bytes num min-width) max-width buffer-size))

(define-typed-fn read-uint8 (stream)
    ((unsigned-byte 8) t)
  "Reads a byte from STREAM and returns it as an unsigned 8-bit integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (read-byte stream))

(define-typed-fn read-int8 (stream)
    (t/int8 t)
  "Reads a byte from STREAM and returns it as a signed 8-bit integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let ((byte (read-byte stream)))
    (declare (type t/uint8 byte))
    (coerce-to-int8 (uint->int/2c byte 8))))

(define-typed-fn write-uint8 (stream ((unsigned-byte 8) byte))
    (null t)
  "Writes the given 8-bit integer to STREAM."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-byte byte stream)
  nil)

(define-typed-fn write-int8 (stream (t/int8 byte))
    (null t)
  "Writes the given signed 8-bit integer to STREAM."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-byte (int->uint/2c byte 8) stream)
  nil)



(define-typed-fn read-uint16 (stream)
    (t/uint16 t)
  "Reads two bytes from STREAM, then converts them to an unsigned 16-bit
integer, treating the order they're read as big endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let* ((b1 (read-byte stream))
         (b2 (read-byte stream)))
    (declare (type t/uint8 b1 b2))
    (logior (the fixnum (ash b2 8)) b1)))

(define-typed-fn read-uint16/be (stream)
    (t/uint16 t)
  "Reads two bytes from STREAM, then converts them to an unsigned 16-bit
integer, treating the order they're read as big endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let* ((b2 (read-byte stream))
         (b1 (read-byte stream)))
    (declare (type t/uint8 b1 b2))
    (logior (the fixnum (ash b2 8)) b1)))

(define-typed-fn read-int16 (stream)
    (t/int16 t)
  "Reads two bytes from STREAM, then converts them to a signed 16-bit integer,
treating the order they're read as big endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let ((word (read-uint16 stream)))
    (declare (type t/uint16 word))
    (coerce-to-int16 (uint->int/2c word 16))))

(define-typed-fn read-int16/be (stream)
    (t/int16 t)
  "Reads two bytes from STREAM, then converts them to a signed 16-bit integer,
treating the order they're read as big endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let ((word (read-uint16/be stream)))
    (declare (type t/uint16 word))
    (coerce-to-int16 (uint->int/2c word 16))))

(define-typed-fn write-uint16 (stream (t/uint16 num))
    (null t)
  "Writes NUM to STREAM as an unsigned 16-bit little endian integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-byte (logand num #x00FF) stream)
  (write-byte (ash (logand num #xFF00) -8) stream)
  nil)

(define-typed-fn write-uint16/be (stream (t/uint16 num))
    (null t)
  "Writes NUM to STREAM as an unsigned 16-bit big endian integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-byte (ash (logand num #xFF00) -8) stream)
  (write-byte (logand num #x00FF) stream)
  nil)

(define-typed-fn write-int16 (stream (t/int16 num))
    (null t)
  "Writes NUM to STREAM as a signed 16-bit little endian integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-uint16 stream (int->uint/2c num 16))
  nil)

(define-typed-fn write-int16/be (stream (t/int16 num))
    (null t)
  "Writes NUM to STREAM as a signed 16-bit little big integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-uint16/be stream (int->uint/2c num 16))
  nil)




(define-typed-fn read-uint24 (stream)
    (t/uint24 t)
  "Reads three bytes from STREAM, then converts them to an unsigned 24-bit integer,
treating the order they're read as little endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let* ((b1 (read-byte stream))
         (b2 (read-byte stream))
         (b3 (read-byte stream)))
    (declare (type t/uint8 b1 b2 b3)
             (optimize (speed 3) (debug 0) (safety 0) (space 0)))
    (logior (the fixnum (ash b3 16)) (the fixnum (ash b2 8)) b1)))

(define-typed-fn read-uint24/be (stream)
    (t/uint24 t)
  "Reads three bytes from STREAM, then converts them to an unsigned 24-bit integer,
treating the order they're read as big endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let* ((b3 (read-byte stream))
         (b2 (read-byte stream))
         (b1 (read-byte stream)))
    (declare (type t/uint8 b1 b2 b3)
             (optimize (speed 3) (debug 0) (safety 0) (space 0)))
    (logior (the fixnum (ash b3 16)) (the fixnum (ash b2 8)) b1)))

(define-typed-fn read-int24 (stream)
    (t/int24 t)
  "Reads three bytes from STREAM, then converts them to a signed 24-bit integer,
treating the order they're read as little endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let ((word (read-uint24 stream)))
    (declare (type t/uint24 word))
    (coerce-to-int24 (uint->int/2c word 24))))

(define-typed-fn read-int24/be (stream)
    (t/int24 t)
  "Reads three bytes from STREAM, then converts them to a signed 24-bit integer,
treating the order they're read as big endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let ((word (read-uint24/be stream)))
    (declare (type t/uint24 word))
    (coerce-to-int24 (uint->int/2c word 24))))

(define-typed-fn write-uint24 (stream (t/uint24 num))
    (null t)
  "Writes NUM to STREAM as an unsigned 24-bit little endian integer stored as three bytes."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-byte (logand num #x0000FF) stream)
  (write-byte (ash (logand num #x00FF00) -8) stream)
  (write-byte (ash (logand num #xFF0000) -16) stream)
  nil)

(define-typed-fn write-uint24/be (stream (t/uint24 num))
    (null t)
  "Writes NUM to STREAM as an unsigned 24-bit big endian integer stored as three bytes."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-byte (ash (logand num #xFF0000) -16) stream)
  (write-byte (ash (logand num #x00FF00) -8) stream)
  (write-byte (logand num #x0000FF) stream)
  nil)

(define-typed-fn write-int24 (stream (t/int24 num))
    (null t)
  "Writes NUM to STREAM as a signed 24-bit little endian integer stored as three bytes."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-uint24 stream (int->uint/2c num 24))
  nil)

(define-typed-fn write-int24/be (stream (t/int24 num))
    (null t)
  "Writes NUM to STREAM as a signed 24-bit big endian integer stored as three bytes."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-uint24/be stream (int->uint/2c num 24))
  nil)




(define-typed-fn read-uint32 (stream)
    (t/uint32 t)
  "Reads four bytes from STREAM, then converts them to an unsigned 32-bit integer,
treating the order they're read as little endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let* ((b1 (read-byte stream))
         (b2 (read-byte stream))
         (b3 (read-byte stream))
         (b4 (read-byte stream)))
    (declare (type t/uint8 b1 b2 b3 b4)
             (optimize (speed 3) (debug 0) (safety 0) (space 0)))
    (logior (the fixnum (ash b4 24))  (the fixnum (ash b3 16))  (the fixnum (ash b2 8)) b1)))

(define-typed-fn read-uint32/be (stream)
    (t/uint32 t)
  "Reads four bytes from STREAM, then converts them to an unsigned 32-bit integer,
treating the order they're read as big endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let* ((b4 (read-byte stream))
         (b3 (read-byte stream))
         (b2 (read-byte stream))
         (b1 (read-byte stream)))
    (declare (type t/uint8 b1 b2 b3 b4)
             (optimize (speed 3) (debug 0) (safety 0) (space 0)))
    (logior (the fixnum (ash b4 24))  (the fixnum (ash b3 16))  (the fixnum (ash b2 8)) b1)))

(define-typed-fn read-int32 (stream)
    (t/int32 t)
  "Reads four bytes from STREAM, then converts them to a signed 32-bit integer,
treating the order they're read as little endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let ((word (read-uint32 stream)))
    (declare (type t/uint32 word))
    (coerce-to-int32 (uint->int/2c word 32))))

(define-typed-fn read-int32/be (stream)
    (t/int32 t)
  "Reads four bytes from STREAM, then converts them to a signed 32-bit integer,
treating the order they're read as big endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (let ((word (read-uint32/be stream)))
    (declare (type t/uint32 word))
    (coerce-to-int32 (uint->int/2c word 32))))

(define-typed-fn write-uint32 (stream (t/uint32 num))
    (null t)
  "Writes NUM to STREAM as an unsigned 32-bit little endian integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-byte (logand num #x000000FF) stream)
  (write-byte (ash (logand num #x0000FF00) -8) stream)
  (write-byte (ash (logand num #x00FF0000) -16) stream)
  (write-byte (ash (logand num #xFF000000) -24) stream)
  nil)

(define-typed-fn write-uint32/be (stream (t/uint32 num))
    (null t)
  "Writes NUM to STREAM as an unsigned 32-bit big endian integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-byte (ash (logand num #xFF000000) -24) stream)
  (write-byte (ash (logand num #x00FF0000) -16) stream)
  (write-byte (ash (logand num #x0000FF00) -8) stream)
  (write-byte (logand num #x000000FF) stream)
  nil)

(define-typed-fn write-int32 (stream (t/int32 num))
    (null t)
  "Writes NUM to STREAM as a signed 32-bit little endian integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-uint32 stream (int->uint/2c num 32))
  nil)

(define-typed-fn write-int32/be (stream (t/int32 num))
    (null t)
  "Writes NUM to STREAM as a signed 32-bit big endian integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-uint32/be stream (int->uint/2c num 32))
  nil)



(define-typed-fn read-uint64 (stream)
    (t/uint64 t)
  "Reads eight bytes from STREAM, then converts them to an unsigned 64-bit integer,
treating the order they're read as little endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (let* ((b1 (read-byte stream))
         (b2 (read-byte stream))
         (b3 (read-byte stream))
         (b4 (read-byte stream))
         (b5 (read-byte stream))
         (b6 (read-byte stream))
         (b7 (read-byte stream))
         (b8 (read-byte stream)))
    (declare (type t/uint8 b1 b2 b3 b4 b5 b6 b7 b8)
             (optimize (speed 3) (debug 1) (safety 0)))
    (logand #xffffffffffffffff
            (logior (the t/uint64 (ash b8 56)) (the t/uint64 (ash b7 48))
                    (the t/uint64 (ash b6 40)) (the t/uint64 (ash b5 32))
                    (the t/uint64 (ash b4 24)) (the t/uint64 (ash b3 16))
                    (the t/uint64 (ash b2 8)) b1))))

(define-typed-fn read-uint64/be (stream)
    (t/uint64 t)
  "Reads eight bytes from STREAM, then converts them to an unsigned 64-bit integer,
treating the order they're read as big endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (let* ((b8 (read-byte stream))
         (b7 (read-byte stream))
         (b6 (read-byte stream))
         (b5 (read-byte stream))
         (b4 (read-byte stream))
         (b3 (read-byte stream))
         (b2 (read-byte stream))
         (b1 (read-byte stream)))
    (declare (type t/uint8 b1 b2 b3 b4 b5 b6 b7 b8)
             (optimize (speed 3) (debug 1) (safety 0)))
    (logand #xffffffffffffffff
            (logior (the t/uint64 (ash b8 56)) (the t/uint64 (ash b7 48))
                    (the t/uint64 (ash b6 40)) (the t/uint64 (ash b5 32))
                    (the t/uint64 (ash b4 24)) (the t/uint64 (ash b3 16))
                    (the t/uint64 (ash b2 8)) b1))))

(define-typed-fn read-int64 (stream)
    (t/int64 t)
  "Reads eight bytes from STREAM, then converts them to an signed 64-bit integer,
treating the order they're read as little endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (uint->int/2c (read-uint64 stream) 64))

(define-typed-fn read-int64/be (stream)
    (t/int64 t)
  "Reads eight bytes from STREAM, then converts them to an signed 64-bit integer,
treating the order they're read as big endian."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0))
           #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
  (uint->int/2c (read-uint64/be stream) 64))

(define-typed-fn write-uint64 (stream (t/uint64 num))
    (null t)
  "Writes NUM to STREAM as an unsigned 64-bit little endian integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-byte (logand num #x00000000000000FF) stream)
  (write-byte (ash (logand num #x000000000000FF00) -8) stream)
  (write-byte (ash (logand num #x0000000000FF0000) -16) stream)
  (write-byte (ash (logand num #x00000000FF000000) -24) stream)
  (write-byte (ash (logand num #x000000FF00000000) -32) stream)
  (write-byte (ash (logand num #x0000FF0000000000) -40) stream)
  (write-byte (ash (logand num #x00FF000000000000) -48) stream)
  (write-byte (ash (logand num #xFF00000000000000) -56) stream)
  nil)

(define-typed-fn write-uint64/be (stream (t/uint64 num))
    (null t)
  "Writes NUM to STREAM as an unsigned 64-bit big endian integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-byte (ash (logand num #xFF00000000000000) -56) stream)
  (write-byte (ash (logand num #x00FF000000000000) -48) stream)
  (write-byte (ash (logand num #x0000FF0000000000) -40) stream)
  (write-byte (ash (logand num #x000000FF00000000) -32) stream)
  (write-byte (ash (logand num #x00000000FF000000) -24) stream)
  (write-byte (ash (logand num #x0000000000FF0000) -16) stream)
  (write-byte (ash (logand num #x000000000000FF00) -8) stream)
  (write-byte (logand num #x00000000000000FF) stream)
  nil)

(define-typed-fn write-int64 (stream (t/int64 num))
    (null t)
  "Writes NUM to STREAM as a signed 64-bit little endian integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-uint64 stream (int->uint/2c num 64))
  nil)

(define-typed-fn write-int64/be (stream (t/int64 num))
    (null t)
  "Writes NUM to STREAM as a signed 64-bit big endian integer."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-uint64/be stream (int->uint/2c num 64))
  nil)



(define-typed-fn read-sfloat (stream)
    ((or single-float t/special-floats) t)
  "Reads an unsigned 32-bit integer value from STREAM, then converts it to a
floating point value, treating the integer as a 32-bit IEEE 754 float.  This
will return either a SINGLE-FLOAT or a T/SPECIAL-FLOATS."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (uint32->sfloat (read-uint32 stream)))

(define-typed-fn read-sfloat/be (stream)
    ((or single-float t/special-floats) t)
  "Reads an unsigned 32-bit integer value from STREAM, then converts it to a
floating point value, treating the integer as a 32-bit IEEE 754 float.  This
will return either a SINGLE-FLOAT or a T/SPECIAL-FLOATS."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (uint32->sfloat (read-uint32/be stream)))

(define-typed-fn read-dfloat (stream)
    ((or double-float t/special-floats) t)
  "Reads an unsigned 64-bit integer value from STREAM, then converts it to a
floating point value, treating the integer as a 64-bit IEEE 754 float.  This
will return either a DOUBLE-FLOAT or a T/SPECIAL-FLOATS."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (uint64->dfloat (read-uint64 stream)))

(define-typed-fn read-dfloat/be (stream)
    ((or double-float t/special-floats) t)
  "Reads an unsigned 64-bit integer value from STREAM, then converts it to a
floating point value, treating the integer as a 64-bit IEEE 754 float.  This
will return either a DOUBLE-FLOAT or a T/SPECIAL-FLOATS."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (uint64->dfloat (read-uint64/be stream)))

(define-typed-fn write-sfloat (stream ((or single-float t/special-floats) num))
    (null t)
  "Writes NUM to STREAM, where NUM is converted to a 32-bit IEEE 754 floating
point number before writing."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-uint32 stream (sfloat->uint32 num))
  nil)

(define-typed-fn write-sfloat/be (stream ((or single-float t/special-floats) num))
    (null t)
  "Writes NUM to STREAM, where NUM is converted to a 32-bit IEEE 754 floating
point number before writing."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-uint32/be stream (sfloat->uint32 num))
  nil)

(define-typed-fn write-dfloat (stream ((or double-float t/special-floats) num))
    (null t)
  "Writes NUM to STREAM, where NUM is converted to a 64-bit IEEE 754 floating
point number before writing."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-uint64 stream (dfloat->uint64 num))
  nil)

(define-typed-fn write-dfloat/be (stream ((or double-float t/special-floats) num))
    (null t)
  "Writes NUM to STREAM, where NUM is converted to a 64-bit IEEE 754 floating
point number before writing."
  (declare (optimize (speed 3) (debug 1) (safety 1) (compilation-speed 0)))
  (write-uint64/be stream (dfloat->uint64 num))
  nil)
