;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Extra I/O Functions
;;;;
;;;; These basically depend on other things.
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(declaim (ftype (function (T &key (:return-type (member :vector :list :string))
                             (:buffer-size fixnum)
                             (:file-length-hint (or null fixnum)))
                          (or vector list string))
                read-stream-to-end))
(defun read-stream-to-end (stream &key (return-type :vector) (buffer-size *io-buffer-size*) file-length-hint)
  "Reads (FILE-LENGTH STREAM) bytes from STREAM.  The return type depends on the
value of RETURN-TYPE:

 * :VECTOR - Returns a vector containing the bytes read.
 * :LIST - Returns a list containing the bytes read.
 * :STRING - Returns a string

FILE-LENGTH-HINT can be given and will be used as the total length of STREAM.
If this is not given, then FILE-LENGTH (or MEMORY-STREAM-LENGTH) is used
internally.

This supports both CHARACTER streams and octet streams."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (check-type file-length-hint (or null fixnum))
  (check-type buffer-size (and unsigned-byte fixnum))
  (case (stream-element-type stream)
    (character
     (let ((buf nil)
           (ret nil))
       (setf buf (make-array buffer-size :element-type 'character))
       (setf ret (with-output-to-string (out)
                   (loop for num-read fixnum = (read-sequence buf stream)
                         while (plusp num-read) do
                           (write-sequence buf out :end num-read))))

       (ecase return-type
         (:vector (string->bytes ret :as-vector t))
         (:list (string->bytes ret))
         (:string ret))))

    (otherwise
     (let* ((num-to-read (the fixnum
                              (- (if file-length-hint
                                     file-length-hint
                                     (etypecase stream
                                       (memory-stream
                                        (the fixnum (memory-stream-length stream)))
                                       (stream (the fixnum (file-length stream)))))
                                 (the fixnum (file-position stream)))))
            (bytes nil))
       (setf bytes (read-bytes stream num-to-read :buffer-size buffer-size))
       (ecase return-type
         (:vector bytes)
         (:list (coerce bytes 'list))
         (:string (%bytes->string/vector bytes)))))))

(declaim (ftype (function ((or string pathname) &key (:return-type (member :vector :list :string))
                                                (:buffer-size fixnum))
                          (or vector list string))
                read-file-to-end))
(defun read-file-to-end (filename &key (return-type :vector) (buffer-size *io-buffer-size*))
  "Reads the contents of a file.  The return type depends on the value of
  RETURN-TYPE:

 * :VECTOR - Returns a vector containing the bytes read.
 * :LIST - Returns a list containing the bytes read.
 * :STRING - Returns a string"
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-open-file (in filename :direction :input :element-type 't/uint8 :if-does-not-exist :error)
    (read-stream-to-end in :return-type return-type :buffer-size buffer-size)))

(declaim (ftype (function (T T &key (:limit-bytes fixnum) (:buffer-size fixnum)) null) copy-bytes))
(defun copy-bytes (source dest &key limit-bytes (buffer-size *io-buffer-size*))
  "Copies bytes from the SOURCE stream to the DEST stream.  If LIMIT-BYTES is
   non-nil, only that number of bytes will be copied."
  (declare (type fixnum buffer-size)
           (type (or fixnum null) limit-bytes)
           (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (when (and limit-bytes (not (plusp limit-bytes)))
    (error "LIMIT-BYTES must be a positive integer"))

  (when (and limit-bytes (<= limit-bytes buffer-size))
    (let ((buf nil)
          (expect-pos 0)
          (new-pos 0))
      (declare (type fixnum expect-pos new-pos)
               (dynamic-extent new-pos expect-pos buf))
      (setf buf (make-array limit-bytes :element-type 't/uint8))
      (setf expect-pos (+ (the fixnum (file-position source)) limit-bytes))
      (setf new-pos (read-sequence buf source))
      (unless (= new-pos expect-pos)
        (error "Failed to read enough data, only read ~:d bytes" (- expect-pos new-pos)))

      (write-sequence buf dest)
      (return-from copy-bytes)))

  (loop do
    (multiple-value-bind (buf sub-buf)
        (read-bytes source buffer-size :buffer-size buffer-size)

      (if buf
          (write-bytes dest buf)
          (progn
            (write-bytes dest sub-buf)
            (loop-finish)))))
  nil)
