;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/ini)

;;;;
;;;; INI Parser and Generator
;;;;
;;;; CL-SDM's INI parser should work for most INI file formats out there.  It
;;;; expects that INI files...
;;;;
;;;; * Have a newline after section names (so not "[foo] key = 1" on one line).
;;;; * Do not have empty section names.
;;;; * Do not have empty key names.
;;;; * Do not have empty section names with newlines in them.
;;;; * Have key names that do not span multiple lines.
;;;; * Have values that do not span multiple lines (though the sequence "\n"
;;;;   will produce a newline)
;;;; * Use a single character to indicate the start of a comment (e.g. ; or #,
;;;;   though multiple comment characters in a row are fine).
;;;; * Have no newline between the key and =, or the = and value.
;;;; * Are not whitespace sensitive except otherwise noted.
;;;;
;;;; The INI-PARSER has options for handling duplicate sections, duplicate key
;;;; names, case-insensitive on section/key names, and empty values.  It does
;;;; not recognize nested sections.
;;;;
;;;; INI files are represented as a HASH-TABLE in CL-SDM.  The T/INI type can be
;;;; used as an alias.
;;;;

(define-simple-error ini-error nil ())

(defgeneric parse-ini (source)
  (:documentation "Parses INI data from SOURCE.

SOURCE can be either a STREAM, a PATHNAME, or a STRING containing a namestring.
If you want to parse INI data that's already contained in a string, use a
STRING-INPUT-STREAM."))

(defgeneric write-ini (ini dest &key &allow-other-keys)
  (:documentation "Writes a T/INI to DEST.

DEST can be any of the following, some of which can accept additional key
arguments:

* STREAM
* NULL (writes to *STANDARD-OUTPUT*)
* PATHNAME.  Accepts they key arguments IF-EXISTS and IF-DOES-NOT-EXIST, which
  behave the same as for OPEN.
* STRING containing a namestring.  Accepts they key arguments IF-EXISTS and
  IF-DOES-NOT-EXIST, which behave the same as for OPEN."))

(deftype t/ini ()
  "An alias for HASH-TABLE.

The keys of a T/INI represent the INI sections.  These must be of type STRING.

The values of a T/INI are the key-value pairs for each section.  These must be
either a HASH-TABLE, or a (VECTOR HASH-TABLE).  When it's a (VECTOR HASH-TABLE),
then the T/INI acts as if it's an INI where duplicate section names are allowed.

Regardless of if the value is a HASH-TABLE or (VECTOR HASH-TABLE), the keys of
these hash tables must be strings and are the key names in that section.  The
values of these tables must be either STRINGs, INTEGERs, or a (VECTOR (OR STRING
INTEGER)).  When they're a (VECTOR (OR STRING INTEGER)), then the T/INI behaves
as if duplicate key names are allowed in each section."
  'hash-table)

(deftype t/parser-state ()
  "The state of the parser during parsing."
  '(member :top-level :section :key :value :key-or-section))

(deftype t/on-duplicates ()
  "When parsing INI data, the T/ON-DUPLICATES instructs the parser what to do when
it encounters a duplicate section or key.

:OVERWRITE means 'overwrite the last section/key with the same name.'

:ERROR means 'raise an INI-ERROR if a duplicate section/key is detected.'

:STORE means 'store duplicates in a VECTOR.'"
  '(member :overwrite :error :store))

(deftype t/multiple-section-vector ()
  '(vector hash-table))

(deftype t/multiple-key-vector ()
  '(vector (or null simple-string integer)))

(defstruct (ini-parser (:constructor %make-ini-parser (stream)))
  "Used to parse INI data.  You should always use MAKE-INI-PARSER to create an
instance."

  (stream nil :type stream)
  (sections (make-hash-table :test 'equal) :type hash-table)
  (line 1 :type t/ufixnum)
  (col 0 :type t/ufixnum)

  ;;
  ;; Behavior options
  ;;

  (expand-newline t :type boolean)
  (downcase-section-names nil :type boolean)
  (downcase-key-names nil :type boolean)
  (comment-style #\; :type character)
  (on-duplicate-keys :error :type t/on-duplicates)
  (on-duplicate-sections :error :type t/on-duplicates)
  (allow-empty-values nil :type boolean))

(defmethod print-object ((obj ini-parser) out)
  (print-unreadable-object (obj out :type t)
    (format out "Pos: ~a, ~a" (ini-parser-line obj) (ini-parser-col obj))))

(defun make-ini-parser (stream &key (comment-style #\;) (on-duplicate-keys :error) (on-duplicate-sections :error)
                                 allow-empty-values downcase-section-names downcase-key-names
                                 (expand-newline t))
  "Creates an INI-PARSER instance that will read from STREAM.

The COMMENT-STYLE parameter must be a CHARACTER, and indicates the character
used to start a comment.

The ON-DUPLICATE-KEYS parameter must be a T/ON-DUPLICATES, and instructs the
parser what to do when two keys are the same within a single section.

The ON-DUPLICATE-SECTIONS parameter must be a T/ON-DUPLICATES, and instructs the
parser what to do when two sections with the same name are found.

When ALLOW-EMPTY-VALUES is truthy, then a key's value can be empty, which is
stored as a NIL within the table.  If ALLOW-EMPTY-KEYS is NIL, then an INI-ERROR
will be raised when an empty value is detected.

When DOWNCASE-SECTION-NAMES is truthy, then all section names will be stored in
lower case.  This effectively makes section names case-insensitive.

When DOWNCASE-KEY-NAMES is truthy, then all key names will be stored in lower
case.  This effectively makes key names case-insensitive.

When EXPAND-NEWLINE is truthy, then the string sequence \"\\n\" will be replaced
with a #\Newline character in parsed values."
  (check-type stream stream)
  (check-type comment-style character)
  (check-type on-duplicate-keys t/on-duplicates)
  (check-type on-duplicate-sections t/on-duplicates)
  (let ((ret (%make-ini-parser stream)))
    (setf (ini-parser-comment-style ret) comment-style)
    (setf (ini-parser-on-duplicate-keys ret) on-duplicate-keys)
    (setf (ini-parser-on-duplicate-sections ret) on-duplicate-sections)
    (setf (ini-parser-allow-empty-values ret) (if allow-empty-values t nil))
    (setf (ini-parser-downcase-section-names ret) (if downcase-section-names t nil))
    (setf (ini-parser-downcase-key-names ret) (if downcase-key-names t nil))
    (setf (ini-parser-expand-newline ret) (if expand-newline t nil))
    ret))

(defmacro err! (parser msg &rest fmt-args)
  "Convenience macro for raising an INI-ERROR that includes the line/column of the
parser."
  `(ini-error () "INI line ~a column ~a: ~a"
     (ini-parser-line ,parser)
     (ini-parser-col ,parser)
     (format nil ,msg ,@fmt-args)))

(define-typed-fn parser-advance-line ((ini-parser parser))
    (null t)
  (declare (optimize speed (debug 0) (safety 1) (space 0) (compilation-speed 0)))
  (incf (ini-parser-line parser))
  (setf (ini-parser-col parser) 0)
  nil)

(define-typed-fn parser-advance-col ((ini-parser parser))
    (null t)
  (declare (optimize speed (debug 0) (safety 1) (space 0) (compilation-speed 0)))
  (incf (ini-parser-col parser))
  nil)

(define-typed-fn parser-advance-col-by ((ini-parser parser) (t/ufixnum count))
    (null t)
  (declare (optimize speed (debug 0) (safety 1) (space 0) (compilation-speed 0)))
  (incf (ini-parser-col parser) count)
  nil)

(define-typed-fn skip-whitespace ((ini-parser parser))
    (null t)
  "Reads and discards characters until a non-whitespace character is detected, or
the end of the file is encountered.  The non-whitespace character will be the
next character read from the stream.

'Whitespace' is defined as #\Newline, #\Space, #\Tab, #\Page, #\No-Break_Space,
#\Figure_Space, #\Return, and #\Zero_Width_Space.

The internal column counter of the parser is not advanced for #\Return and
#\Zero_Width_Space.  It is advanced one column for all other whitespace except
#\Newline, which advances it an entire line."
  (declare (optimize speed (debug 0) (safety 1) (space 0) (compilation-speed 0)))
  (loop for c = (peek-char nil (ini-parser-stream parser) nil nil)
        while c
        do (case c
             (#\Newline
              (read-char (ini-parser-stream parser))
              (parser-advance-line parser))

             ((#\Space #\Tab #\Page #\No-Break_Space #\Figure_Space)
              (read-char (ini-parser-stream parser))
              (parser-advance-col parser))

             ((#\Return #\Zero_Width_Space)
              (read-char (ini-parser-stream parser)))

             (t
              (loop-finish))))
  nil)

(define-typed-fn skip-whitespace! ((ini-parser parser))
    (null t)
  "The same as SKIP-WHITESPACE, except this does not consider #\Newline a whitespace character."
  (declare (optimize speed (debug 0) (safety 1) (space 0) (compilation-speed 0)))
  (loop for c = (peek-char nil (ini-parser-stream parser) nil nil)
        while c
        do (case c
             (#\Newline
              (loop-finish))

             ((#\Space #\Tab #\Page #\No-Break_Space #\Figure_Space)
              (read-char (ini-parser-stream parser))
              (parser-advance-col parser))

             ((#\Return #\Zero_Width_Space)
              (read-char (ini-parser-stream parser)))

             (t
              (loop-finish))))
  nil)

(define-typed-fn skip-to-eol ((ini-parser parser))
    (null t)
  "Reads and discards characters until the end of the line (#\Newline), or the end
of the file is encountered.

The #\Newline is discarded."
  (declare (optimize speed (debug 0) (safety 1) (space 0) (compilation-speed 0)))
  (loop for c = (peek-char nil (ini-parser-stream parser) nil nil)
        while (and c (not (char= c #\Newline)))
        do (read-char (ini-parser-stream parser) nil nil))
  (read-char (ini-parser-stream parser) nil nil) ;; Read the actual newline (if it's there)
  (parser-advance-line parser)
  nil)

(define-typed-fn read-until ((ini-parser parser) (character until) &optional
                             stop-at-newline? keep-trailing-whitespace? comment-char)
    (simple-string t)
  "Reads characters until the UNTIL character is encountered.  If STOP-AT-NEWLINE?
is truthy, then a #\Newline character will also stop the reading (the #\Newline
will not be discarded).

If KEEP-TRAILING-WHITESPACE? is truthy, then any whitespace between the last
non-whitespace character and the UNTIL character (or newline if STOP-AT-NEWLINE?
is also truthy) will be appended to the resulting string.  Otherwise it will be
discarded.

If COMMENT-CHAR is provided, it must be a CHARACTER, and READ-UNTIL will also
stop for this character.  It will not be discarded.  If COMMENT-CHAR is NIL,
then the parameter is ignored.

If the end of the stream is hit, this will return what it was able to read up
until that point.

Returns a new SIMPLE-STRING."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-output-to-string (out)
    (macrolet
        ((save-char ()
           `(write-char (read-char (ini-parser-stream parser)) out))

         (skip-char ()
           `(read-char (ini-parser-stream parser))))

      (loop with ws fixnum = 0
            for c = (peek-char nil (ini-parser-stream parser) nil nil)
            do (when (or (null c) (char= c until))
                 (loop-finish))

               (case c
                 (#\Newline
                  (cond
                    (stop-at-newline?
                     (loop-finish))

                    (t
                     (unless (zerop ws)
                       (dotimes (i ws)
                         (write-char #\Space out))
                       (setf ws 0))))

                  (parser-advance-line parser)
                  (save-char))

                 ((#\Space #\Tab #\No-Break_Space #\Figure_Space)
                  (incf ws)
                  (skip-char))

                 (otherwise
                  (unless (zerop ws)
                    (dotimes (i ws)
                      (write-char #\Space out))
                    (parser-advance-col-by parser ws)
                    (setf ws 0))

                  (when (and comment-char
                             (char= c comment-char))
                    (loop-finish))

                  (parser-advance-col parser)
                  (save-char)))

            finally
               (when (and (plusp ws)
                          keep-trailing-whitespace?)
                 (parser-advance-col-by parser ws)
                 (dotimes (i ws)
                   (write-char #\Space out)))))))

(define-typed-fn parser-read-char ((ini-parser parser))
    ((or character null) t)
  "Reads the next character from the stream, advancing the internal position.  If
there are no more characters to be read, this returns NIL."
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (let ((c (read-char (ini-parser-stream parser) nil nil)))
    (declare (type (or null character) c))
    (when c
      (if (char= c #\Newline)
          (parser-advance-line parser)
          (parser-advance-col parser)))
    c))

(defmethod parse-ini ((parser ini-parser))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))

  (labels
      ((store-section (parser section name)
         (declare (type ini-parser parser)
                  (type hash-table section)
                  (type simple-string name)
                  (optimize speed (debug 0) (safety 1) (compilation-speed 0)))

         (let ((dest (ini-parser-sections parser)))

           ;; Is the section a duplicate?
           (multiple-value-bind (old-value found?)
               (gethash name dest)
             (if found?
                 (case (ini-parser-on-duplicate-sections parser)
                   (:error
                    (err! parser "Duplicate section detected"))

                   (:store
                    (etypecase old-value
                      (hash-table
                       (setf (gethash name dest) (new-vector hash-table))

                       ;; Store both the previous section and the new section
                       (vector-push-extend old-value (the t/multiple-section-vector (gethash name dest)))
                       (vector-push-extend section (the t/multiple-section-vector (gethash name dest))))

                      (t/multiple-section-vector
                       (vector-push-extend section (the t/multiple-section-vector (gethash name dest))))))

                   (:overwrite
                    (setf (gethash name dest) section)))

                 ;; Not a duplicate
                 (setf (gethash name dest) section))))))

    (let ((section nil)
          (last-section-name nil)
          (key nil)
          (val nil)
          (state :top-level))
      (declare (type (or null hash-table) section)
               (type (or null simple-string) key last-section-name)
               (type (or null simple-string integer) val)
               (type t/parser-state state))


      ;; Skip any initial whitespace
      (skip-whitespace parser)

      ;; Start parsing
      (loop for c = (peek-char nil (ini-parser-stream parser) nil :eof) do
        (when (eq c :eof)
          ;; EOF
          (loop-finish))

        (cond
          ((char= c #\[) ;; Start of a section
           (unless (or (eq state :top-level)
                       (eq state :key-or-section))
             (err! parser "Unexpected start of section"))

           ;; Do we already have a section going?
           (when section
             (assert (stringp last-section-name))
             (store-section parser section last-section-name))

           ;; Start a new section
           (setf section (make-hash-table :test 'equal))
           (setf state :section)

           ;; Read the [, skip whitespace, then read the section name.
           (parser-read-char parser)
           (skip-whitespace! parser)
           (setf last-section-name (read-until parser #\] t nil))
           (when (ini-parser-downcase-section-names parser)
             (setf last-section-name (string-downcase last-section-name)))

           ;; There cannot be a newline in the section name.
           (when (find #\Newline last-section-name :test #'char=)
             (err! parser "Section names cannot have newlines"))

           ;; There cannot be a newline between the end of the section name and the ]
           (skip-whitespace! parser)
           (let ((next-c (read-char (ini-parser-stream parser) nil nil)))
             (parser-advance-col parser)
             (if (null next-c)
                 (err! parser "Unexpected end of file during section name")
                 (unless (char= next-c #\])
                   (err! parser "Section not terminated before the end of the line"))))

           ;; Comments are not allowed in section names.
           (when (find (ini-parser-comment-style parser) last-section-name :test #'char=)
             (err! parser "Comments in section block are not allowed"))

           ;; Empty section names are never allowed.
           (when (empty-string-p! last-section-name)
             (err! parser "Empty section name"))

           ;; The ] must be the last character on the line, excluding any
           ;; non-newline whitespace, comments, or the end-of-file.
           (skip-whitespace! parser)
           (let ((next-c (read-char (ini-parser-stream parser) nil nil)))
             (cond
               ((null next-c) nil) ;; EOF is fine
               ((char= next-c #\Newline) nil) ;; Newline is fine
               ((char= next-c (ini-parser-comment-style parser)) ;; Comments are fine
                (skip-to-eol parser))
               (t (err! parser "Junk after section name")))) ;; Everything else is junk

           ;; Check to see if this is a duplicate section name.

           ;; Skip whitespace, change state
           (skip-whitespace parser)
           (setf state :key))

          ((char= c (ini-parser-comment-style parser)) ;; Comments
           (skip-to-eol parser)
           (skip-whitespace parser))

          (t ;; All other characters
           (case state
             ;; Either of these two states means there's junk in the INI.
             (:top-level (err! parser "Unexpected character: ~a" c))
             (:section (err! parser "Section not terminated"))

             ((:key :key-or-section) ;; Parses a key
              ;; We must be within a section.
              (unless section
                (err! parser "Started a key without a section"))

              ;; Read the key
              (setf key (read-until parser #\= t nil))
              (when (ini-parser-downcase-key-names parser)
                (setf key (string-downcase key)))

              ;; The next non-whitespace character cannot be a #\Newline
              (skip-whitespace! parser)
              (let ((next-c (peek-char nil (ini-parser-stream parser) nil nil)))
                (when (eq next-c #\Newline)
                  (err! parser "A key and its value must be on the same line")))

              ;; Trim whitespace.  Check that there isn't a comment character in
              ;; the key (comments are never allowed within key names)
              (setf key (trim-whitespace! key))
              (when (find (ini-parser-comment-style parser) (the simple-string key) :test #'char=)
                (err! parser "Comments in key names are not allowed"))

              ;; The key name can never be empty.
              (when (empty-string-p! key)
                (err! parser "Empty key"))

              ;; Read the =, then advance the internal position.
              (unless (eq (read-char (ini-parser-stream parser) nil nil) #\=)
                (err! parser "Key not terminated"))
              (parser-advance-col parser)

              ;; Check for duplicate key names.
              (when (hash-table-contains-p section key)
                (ecase (ini-parser-on-duplicate-keys parser)
                  (:error
                   (err! parser "Duplicate key detected"))
                  (:store
                   (unless (typep (gethash key section) 't/multiple-key-vector)
                     ;; Setup the hash table value as a vector.
                     (multiple-value-bind (element found?)
                         (gethash key section)
                       (setf (gethash key section) (new-vector (or null simple-string integer)))
                       (when found?
                         (vector-push-extend element (the t/multiple-key-vector (gethash key section)))))))
                  (:overwrite
                   ;; Nothing to do here, it'll be overwritten.
                   nil)))

              ;; Skip whitespace, change state.
              (setf state :value)
              (skip-whitespace! parser))

             (:value ;; Parses a value
              ;; We must have a key
              (unless key
                (err! parser "Started a value without a key"))

              ;; Read the value.  If it's an empty string, figure out what we're doing with it.
              (setf val (trim-whitespace! (read-until parser #\Newline t nil (ini-parser-comment-style parser))))
              (when (empty-string-p! val)
                (unless (ini-parser-allow-empty-values parser)
                  (err! parser "Value is empty"))
                (setf val nil))

              ;; See if this is an integer, and if it is, store it as an integer.
              (when val
                (when-var (int-val (parse-integer? val))
                  (setf val int-val)))

              ;; Expand any \n instances
              (when (and (stringp val)
                         (ini-parser-expand-newline parser))
                (setf val (string-replace val "\\n" (string #\Newline))))

              ;; Store the value
              (if (typep (gethash key section nil) 't/multiple-key-vector)
                  (vector-push-extend val (the t/multiple-key-vector (gethash key section)))
                  (setf (gethash key section) val))

              ;; Reset KEY and VAL, change state, then skip whitespace.  The next
              ;; thing we read must be a section or key.
              (setf key nil)
              (setf val nil)
              (setf state :key-or-section)
              (skip-whitespace parser))))))

      ;; We must have ended while awaiting a section or a key.
      (unless (or (eq state :top-level)
                  (eq state :key)
                  (eq state :key-or-section))
        (err! parser "Unexpected end of file"))

      ;; Store the final section, if any.
      (when section
        (assert (stringp last-section-name))
        (store-section parser section last-section-name))))

  ;; Return all sections.
  (ini-parser-sections parser))

(defmethod parse-ini ((source stream))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (let ((parser (make-ini-parser source)))
    (parse-ini parser)))

(defmethod parse-ini ((source pathname))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-open-file (in source :direction :input)
    (parse-ini in)))

(defmethod parse-ini ((source string))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (parse-ini (uiop:parse-native-namestring source)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Writing
;;;

(define-typed-fn write-section ((simple-string name) (stream out))
    (null t)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (format out "[~a]~%" name)
  nil)

(define-typed-fn write-pairs ((hash-table pairs) (stream out))
    (null t)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (maphash
   #'(lambda (raw-key val)
       (unless (stringp raw-key)
         (ini-error () "Key names must be STRINGs"))

       (let ((key (trim-whitespace! raw-key)))
         (when (zerop (length key))
           (ini-error () "Key names cannot be empty strings"))
         (format out "~a = ~a~%" key val)))
   pairs)
  nil)


(defmethod write-ini ((ini hash-table) (dest stream) &key)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (let ((wrote-section? nil))
    (maphash
     #'(lambda (section-name pairs)
         (unless (stringp section-name)
           (ini-error () "Section names must be STRINGs"))

         (let ((section (trim-whitespace! section-name)))
           (when (zerop (length section))
             (ini-error () "Section names cannot be empty strings"))

           (when wrote-section? (write-char #\Newline dest))
           (write-section section dest)
           (write-pairs pairs dest)
           (setf wrote-section? t))) ;; Put a newline between sections for readability
     ini))
  nil)

(defmethod write-ini (ini (dest null) &key)
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (write-ini ini *standard-output*))

(defmethod write-ini (ini (dest pathname) &key (if-exists :error) (if-does-not-exist :create))
  (declare (optimize speed (debug 1) (safety 1) (compilation-speed 0)))
  (with-open-file (out dest :direction :output
                            :if-exists if-exists
                            :if-does-not-exist if-does-not-exist)
    (write-ini ini out)))

(defmethod write-ini (ini (dest string) &key (if-exists :error) (if-does-not-exist :create))
  (write-ini ini
             (uiop:parse-native-namestring dest)
             :if-exists if-exists
             :if-does-not-exist if-does-not-exist))
