;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;; HJSON code based on hjson-go, Copyright (C) 2016, 2017 Christian Zangl
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-hjson)

;;;;
;;;; HJSON Metaclass and Unmarshaling
;;;;
;;;; Classes that use the HJSON-CLASS metaclass can be automatically using
;;;; UNMARSHAL in a strictly typed way.  When doing this, each slot should
;;;; define what JSON type is expected for that slot.
;;;;
;;;; Scalars and arrays are specified with keywords.  There are keywords for the
;;;; non-null version, nullable version, array of non-nulls, array of nullables,
;;;; a nullable array of non-nulls, and a nullable array of nullables.  For
;;;; example, the following unsigned 8-bit integer scalar variations are
;;;; supported: :UINT8, :UINT8? (8-bit unsigned int or null), :ARRAY-UINT8,
;;;; :ARRAY-UINT8? (an array of values that can be either an 8-bit unsigned int,
;;;; or null), :ARRAY?-UINT8 (a null or an array of unsigned 8-bit ints), and
;;;; :ARRAY?-UINT8?.  The scalar types supported in this way include :UINT8,
;;;; :UINT16, :UINT24, :UINT32, :UINT64, :INT8, :INT16, :INT24, :INT32, :INT64,
;;;; :STRING, and :BOOL.
;;;;
;;;; If you need more complex type handling, you can use a custom unmarshalling
;;;; function for a slot instead.  To do this, specify it with
;;;; :HJSON-UNMARSHL-FN.  A hash table or vector will be passed to this function
;;;; containing the object that CL-SDM-HJSON produced during parsing.
;;;;
;;;; If you need a float of some sort, you should use :HJSON-UNMARSHAL-FN
;;;; together with your own type checking code.
;;;;

(deftype t/hjson-slot-types ()
  '(member
    :class :class? :class-array :class-array? :class-table :class-table?

    :string :string? :bool :bool?
    :array-string :array-string? :array?-string :array?-string??
    :array-bool :array-bool? :array?-bool? :array?-bool??

    :uint8 :uint8? :uint16 :uint16? :uint32 :uint32? :uint64 :uint64?
    :int8 :int8? :int16 :int16? :int32 :int32? :int64 :int64?

    :array-uint8 :array-uint8? :array-uint16 :array-uint16? :array-uint32 :array-uint32? :array-uint64 :array-uint64?
    :array-int8 :array-int8? :array-int16 :array-int16? :array-int32 :array-int32? :array-int64 :array-int64?

    :array?-uint8 :array?-uint8? :array?-uint16 :array?-uint16? :array?-uint32 :array?-uint32? :array?-uint64
    :array?-uint64?
    :array?-int8 :array?-int8? :array?-int16 :array?-int16? :array?-int32 :array?-int32? :array?-int64
    :array?-int64?))

;;;
;;; Scalar Utilities
;;;

(eval-when (:compile-toplevel :load-toplevel :execute)
  (defmacro define-hjson-scalar (name scalar-type condition-fn)
    (let ((thing-name (intern (format nil "T/HJSON-~A" name) :cl-sdm-hjson))
          (nillable-thing-name (intern (format nil "T/HJSON-~A?" name)  :cl-sdm-hjson))

          (thing-p (intern (format nil "T/HJSON-~A-P" name)  :cl-sdm-hjson))
          (nillable-thing-p (intern (format nil "T/HJSON-~A?-P" name)  :cl-sdm-hjson))

          (list-name (intern (format nil "T/HJSON-~A-ARRAY" name)  :cl-sdm-hjson))
          (list-p (intern (format nil "T/HJSON-~A-ARRAY-P" name)  :cl-sdm-hjson))

          (list-with-nillables-name (intern (format nil "T/HJSON-~A?-ARRAY" name)  :cl-sdm-hjson))
          (list-with-nillables-p (intern (format nil "T/HJSON-~A?-ARRAY-P" name)  :cl-sdm-hjson))

          (nillable-list-name (intern (format nil "T/HJSON-~A-ARRAY?" name)  :cl-sdm-hjson))
          (nillable-list-p (intern (format nil "T/HJSON-~A-ARRAY?-P" name)  :cl-sdm-hjson))

          (totally-nillable-list-name (intern (format nil "T/HJSON-~A?-ARRAY?" name)  :cl-sdm-hjson))
          (totally-nillable-list-p (intern (format nil "T/HJSON-~A?-ARRAY?-P" name)  :cl-sdm-hjson)))

      `(progn
         ;;
         ;; Basic type
         ;;

         (defun ,thing-p (thing)
           (typep thing ,(list 'quote scalar-type)))
         (export ,(list 'quote thing-p) :cl-sdm-hjson)

         (deftype ,thing-name ()
           ,(list 'quote scalar-type))
         (export ,(list 'quote thing-name) :cl-sdm-hjson)

         ;;
         ;; Nillable version
         ;;

         (defun ,nillable-thing-p (thing)
           (typep thing ,(list 'quote (list 'or 'null scalar-type))))
         (export ,(list 'quote nillable-thing-p) :cl-sdm-hjson)

         (deftype ,nillable-thing-name ()
           ,(list 'quote (list 'or (list 'null scalar-type))))
         (export ,(list 'quote nillable-thing-name) :cl-sdm-hjson)

         ;;
         ;; List version
         ;;

         (defun ,list-p (thing)
           (and (vectorp thing)
                (every ,condition-fn thing)))
         (export ,(list 'quote list-p) :cl-sdm-hjson)

         (deftype ,list-name ()
           ,(list 'quote (list 'and 'list (list 'satisfies list-p))))
         (export ,(list 'quote list-name) :cl-sdm-hjson)

         ;;
         ;; List of nillable version
         ;;

         (defun ,list-with-nillables-p (thing)
           (and (vectorp thing)
                (every #'(lambda (x)
                           (or (typep x 'null)
                               (funcall ,condition-fn x)))
                       thing)))
         (export ,(list 'quote list-with-nillables-p) :cl-sdm-hjson)

         (deftype ,list-with-nillables-name ()
           ,(list 'quote (list 'and 'list (list 'satisfies list-with-nillables-p))))
         (export ,(list 'quote list-with-nillables-name) :cl-sdm-hjson)

         ;;
         ;; Nillable list version
         ;;

         (defun ,nillable-list-p (thing)
           (or (typep thing 'null)
               (and (vectorp thing)
                    (every ,condition-fn thing))))
         (export ,(list 'quote nillable-list-p) :cl-sdm-hjson)

         (deftype ,nillable-list-name ()
           ,(list 'quote (list 'or 'null (list 'and 'list (list 'satisfies nillable-list-p)))))
         (export ,(list 'quote nillable-list-name) :cl-sdm-hjson)

         ;;
         ;; Nillable list with nillables version
         ;;

         (defun ,totally-nillable-list-p (thing)
           (or (typep thing 'null)
               (and (vectorp thing)
                    (every #'(lambda (x)
                               (or (typep x 'null)
                                   (funcall ,condition-fn x)))
                           thing))))
         (export ,(list 'quote totally-nillable-list-p) :cl-sdm-hjson)

         (deftype ,totally-nillable-list-name ()
           ,(list 'quote (list 'or 'null (list 'and 'list (list 'satisfies totally-nillable-list-p)))))
         (export ,(list 'quote totally-nillable-list-name) :cl-sdm-hjson)))))

(eval-when (:compile-toplevel :load-toplevel :execute)
  (define-hjson-scalar int8 (signed-byte 8) #'(lambda (x) (typep x '(signed-byte 8))))
  (define-hjson-scalar int16 (signed-byte 16) #'(lambda (x) (typep x '(signed-byte 16))))
  (define-hjson-scalar int24 (signed-byte 24) #'(lambda (x) (typep x '(signed-byte 24))))
  (define-hjson-scalar int32 (signed-byte 32) #'(lambda (x) (typep x '(signed-byte 32))))
  (define-hjson-scalar int64 (signed-byte 64) #'(lambda (x) (typep x '(signed-byte 64))))

  (define-hjson-scalar uint8 (unsigned-byte 8) #'(lambda (x) (typep x '(unsigned-byte 8))))
  (define-hjson-scalar uint16 (unsigned-byte 16) #'(lambda (x) (typep x '(unsigned-byte 16))))
  (define-hjson-scalar uint24 (unsigned-byte 24) #'(lambda (x) (typep x '(unsigned-byte 24))))
  (define-hjson-scalar uint32 (unsigned-byte 32) #'(lambda (x) (typep x '(unsigned-byte 32))))
  (define-hjson-scalar uint64 (unsigned-byte 64) #'(lambda (x) (typep x '(unsigned-byte 64))))

  (define-hjson-scalar string string #'stringp)
  (define-hjson-scalar bool (or t/hjson-true t/hjson-false) #'(lambda (x) (typep x 'boolean))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; CLOS Metaobject Protocol Stuff
;;;

(defgeneric slot-key-name (obj &optional translate-hyphens))
(defgeneric (setf slot-key-name) (value obj))

(defclass hjson-class (closer-mop:standard-class)
  ((non-strict
    :initarg :non-strict
    :initform nil
    :type boolean
    :reader hjson-class-non-strict-p)

   (translate-hyphens
    :initarg :translate-hyphems
    :initform t
    :type boolean
    :reader hjson-class-translate-hyphens-p)

   (unknown-fields-to-slot
    :initarg :unknown-to-slot
    :initform nil
    :type (or null symbol)
    :reader hjson-class-unknown-fields-to-slot)

   (non-strict-marshalling
    :initarg :non-strict-marshalling
    :initform nil
    :type boolean
    :reader hjson-class-non-strict-marshalling-p))
  (:documentation "When set as a metaclass, this is used to provide automatic
unmarshalling of HJSON data."))

(defclass hjson-slot (closer-mop:standard-direct-slot-definition)
  ((key-name
    :initarg :hjson-key-name
    :initform nil
    :type (or null string))

   (ignore?
    :initarg :hjson-ignore
    :initform nil
    :type boolean
    :accessor slot-ignore-p)

   (dont-emit-default-val
    :initarg :hjson-dont-emit-default-val
    :initform nil
    :type boolean
    :accessor slot-dont-emit-default-val-p)

   (marshal-fn
    :initarg :hjson-marshal-fn
    :initform nil
    :type (or null function)
    :accessor slot-marshal-fn)

   (unmarshal-fn
    :initarg :hjson-unmarshal-fn
    :initform nil
    :type (or null function)
    :accessor slot-unmarshal-fn)

   (unmarshal-type
    :initarg :hjson-type
    :initform nil
    :type (or null keyword symbol)
    :accessor slot-unmarshal-type))
  (:documentation "Slot metaclass for all slots in a HJSON-CLASS class.  This
type of slot has additional metadata for defining unmarshalling options."))

(defclass hjson-slot/effective (closer-mop:standard-effective-slot-definition)
  ((key-name
    :initarg :hjson-key-name
    :initform nil
    :type (or null string))

   (ignore?
    :initarg :hjson-ignore
    :initform nil
    :type boolean
    :accessor slot-ignore-p)

   (dont-emit-default-val
    :initarg :hjson-dont-emit-default-val
    :initform nil
    :type boolean
    :accessor slot-dont-emit-default-val-p)

   (marshal-fn
    :initarg :hjson-marshal-fn
    :initform nil
    :type (or null function)
    :accessor slot-marshal-fn)

   (unmarshal-fn
    :initarg :hjson-unmarshal-fn
    :initform nil
    :type (or null function)
    :accessor slot-unmarshal-fn)

   (unmarshal-type
    :initarg :hjson-type
    :initform nil
    :type (or null keyword symbol)
    :accessor slot-unmarshal-type))
  (:documentation "Slot metaclass for all slots in a HJSON-CLASS class.  This
type of slot has additional metadata for defining unmarshalling options."))

(defmethod closer-mop:validate-superclass ((class hjson-class) (superclass standard-class))
  t)

(defmethod closer-mop:direct-slot-definition-class ((class hjson-class) &rest initargs)
  (declare (ignore initargs))
  (find-class 'hjson-slot))

(defmethod closer-mop:effective-slot-definition-class ((class hjson-class) &rest initargs)
  (declare (ignore initargs))
  (find-class 'hjson-slot/effective))

(defmethod closer-mop:compute-effective-slot-definition ((class hjson-class) name direct-slots)
  (let ((slot-def (call-next-method))
        (translate? (hjson-class-translate-hyphens-p class)))

    (loop for slot in direct-slots
          for key-name = (slot-key-name slot translate?)
          for um-type = (slot-unmarshal-type slot)
          for um-fn = (slot-unmarshal-fn slot)
          for dedv = (slot-dont-emit-default-val-p slot)
          for m-fn = (slot-marshal-fn slot)
          for um-ignore = (slot-ignore-p slot)

          do (when (not (null um-type))
               (if (listp um-type)
                   (unless (typep (car um-type) 't/hjson-slot-types)
                     (error "Invalid HJSON type for slot ~a: ~a" name um-type))
                   (unless (typep um-type 't/hjson-slot-types)
                     (error "Invalid HJSON type for slot ~a: ~a" name um-type))))

             (setf
              (slot-key-name slot-def) key-name
              (slot-unmarshal-type slot-def) um-type
              (slot-ignore-p slot-def) um-ignore
              (slot-dont-emit-default-val-p slot-def) dedv
              (slot-unmarshal-fn slot-def) (if um-fn
                                               (if (listp um-fn)
                                                   (if (and (= (length um-fn) 2)
                                                            (eq (car um-fn) 'function))
                                                       (symbol-function (cadr um-fn))
                                                       (compile nil um-fn))
                                                   (symbol-function um-fn))
                                               nil)
              (slot-marshal-fn slot-def) (if m-fn
                                             (if (listp m-fn)
                                                 (if (and (= (length m-fn) 2)
                                                          (eq (car m-fn) 'function))
                                                     (symbol-function (cadr m-fn))
                                                     (compile nil m-fn))
                                                 (symbol-function m-fn))
                                             nil))
             (loop-finish))

    slot-def))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmethod slot-key-name ((slot hjson-slot) &optional translate-hyphens)
  "Returns the expected key name for the given slot.  If the slot has a KEY-NAME
assigned to it, that takes precidence.  Otherwise, it is equal to the downcased
slot name."
  (with-slots (key-name) slot
    (or key-name
        (let ((ret (string-downcase (string (closer-mop:slot-definition-name slot)))))
          (if translate-hyphens
              (sdm:string-replace ret "-" "_")
              ret)))))

(defmethod slot-key-name ((slot hjson-slot/effective) &optional translate-hyphens)
  "Returns the expected key name for the given slot.  If the slot has a KEY-NAME
assigned to it, that takes precidence.  Otherwise, it is equal to the downcased
slot name."
  (with-slots (key-name) slot
    (or key-name
        (let ((ret (string-downcase (string (closer-mop:slot-definition-name slot)))))
          (if translate-hyphens
              (sdm:string-replace ret "-" "_")
              ret)))))

(defmethod (setf slot-key-name) (value (slot hjson-slot))
  "Sets the KEY-NAME for a given slot."
  (setf (slot-value slot 'key-name) value))

(defmethod (setf slot-key-name) (value (slot hjson-slot/effective))
  "Sets the KEY-NAME for a given slot."
  (setf (slot-value slot 'key-name) value))

(defun find-applicable-slot (class key-name)
  "Check to see if a slot that has the given key name exists in CLASS.  If one
does, this returns that slot.  Otherwise this returns nil."
  (find key-name (closer-mop:class-slots class)
        :test #'(lambda (ref check)
                  (or (sdm:caseless-string= (string (closer-mop:slot-definition-name check))
                                            (string ref))
                      (when (slot-key-name check)
                        (sdm:caseless-string= (slot-key-name check) (string ref)))))))

(defun hjson-slot-needs-class (slot)
  "Returns T if the slot is expecting to be unmarshalled to another HJSON-CLASS
class, or NIL otherwise."
  (and (listp (slot-unmarshal-type slot))
       (= (length (slot-unmarshal-type slot)) 2)
       (or (eql (nth 0 (slot-unmarshal-type slot)) :class)
           (eql (nth 0 (slot-unmarshal-type slot)) :class?))))

(defun hjson-slot-needs-class-array (slot)
  "Returns T if the slot is expecting to be unmarshalled to an array of
HJSON-CLASS classs, or NIL otherwise."
  (and (listp (slot-unmarshal-type slot))
       (= (length (slot-unmarshal-type slot)) 2)
       (or (eql (nth 0 (slot-unmarshal-type slot)) :class-array)
           (eql (nth 0 (slot-unmarshal-type slot)) :class-array?))))

(defun hjson-slot-needs-class-table (slot)
  "Returns T if the slot is expecting to be unmarshalled to a hash table of
HJSON-CLASS class instances, or NIL otherwise."
  (and (listp (slot-unmarshal-type slot))
       (= (length (slot-unmarshal-type slot)) 2)
       (or (eql (nth 0 (slot-unmarshal-type slot)) :class-table)
           (eql (nth 0 (slot-unmarshal-type slot)) :class-table?))))

(defun assign-typed-value-to-slot (class instance slot value type)
  "Checks to see that VALUE is of the expected type, and if it is, assigns VALUE
to SLOT.  If it is not, a UNMARSHAL-ERROR is raised."
  (macrolet ((type-check (check-fn value-var fancy-name)
               `(if (,check-fn ,value-var)
                    ,value-var
                    (error 'unmarshal-error
                           :format-control "The slot value ~s requires ~a"
                           :format-arguments (list (slot-key-name slot (hjson-class-translate-hyphens-p class))
                                                   ,fancy-name)))))

    (setf (closer-mop:slot-value-using-class class instance slot)
          (case type
            ;; Strings
            (:string (type-check t/hjson-string-p value "a string"))
            (:string? (type-check t/hjson-string?-p value "a string or NIL"))
            (:array-string (type-check t/hjson-string-array-p value "an array of strings"))
            (:array?-string (type-check t/hjson-string-array?-p value "an array of strings, or NIL"))
            (:array-string? (type-check t/hjson-string?-array-p value "an array containing strings or NIL"))
            (:array?-string? (type-check t/hjson-string?-array?-p value "an array containing strings or NIL, or NIL"))

            ;; Bools
            (:bool (type-check t/hjson-bool-p value "a boolean"))
            (:bool? (type-check t/hjson-bool?-p value "a boolean or NIL"))
            (:array-bool (type-check t/hjson-bool-array-p value "an array of booleans"))
            (:array?-bool (type-check t/hjson-bool-array?-p value "an array of booleans, or NIL"))
            (:array-bool? (type-check t/hjson-bool?-array-p value "an array containing booleans or NIL"))
            (:array?-bool? (type-check t/hjson-bool?-array?-p value "an array containing booleans or NIL, or NIL"))

            ;; uint8
            (:uint8 (type-check t/hjson-uint8-p value "an unsigned 8-bit integer"))
            (:uint8? (type-check t/hjson-uint8?-p value "an unsigned 8-bit integer or NIL"))
            (:array-uint8 (type-check t/hjson-uint8-array-p value "an array of unsigned 8-bit integers"))
            (:array?-uint8 (type-check t/hjson-uint8-array?-p value "an array of unsigned 8-bit integers, or NIL"))
            (:array-uint8? (type-check t/hjson-uint8?-array-p value "an array containing unsigned 8-bit integers or NIL"))
            (:array?-uint8? (type-check t/hjson-uint8?-array?-p value
                                       "an array containing unsigned 8-bit integers or NIL, or NIL"))

            ;; uint16
            (:uint16 (type-check t/hjson-uint16-p value "an unsigned 16-bit integer"))
            (:uint16? (type-check t/hjson-uint16?-p value "an unsigned 16-bit integer or NIL"))
            (:array-uint16 (type-check t/hjson-uint16-array-p value "an array of unsigned 16-bit integers"))
            (:array?-uint16 (type-check t/hjson-uint16-array?-p value "an array of unsigned 16-bit integers, or NIL"))
            (:array-uint16? (type-check t/hjson-bool?-array-p value
                                       "an array containing unsigned 16-bit integers or NIL"))
            (:array?-uint16? (type-check t/hjson-uint16?-array?-p value
                                       "an array containing unsigned 16-bit integers or NIL, or NIL"))

            ;; uint24
            (:uint24 (type-check t/hjson-uint24-p value "an unsigned 24-bit integer"))
            (:uint24? (type-check t/hjson-uint24?-p value "an unsigned 24-bit integer or NIL"))
            (:array-uint24 (type-check t/hjson-uint24-array-p value "an array of unsigned 24-bit integers"))
            (:array?-uint24 (type-check t/hjson-uint24-array?-p value "an array of unsigned 24-bit integers, or NIL"))
            (:array-uint24? (type-check t/hjson-bool?-array-p value
                                       "an array containing unsigned 24-bit integers or NIL"))
            (:array?-uint24? (type-check t/hjson-uint24?-array?-p value
                                       "an array containing unsigned 24-bit integers or NIL, or NIL"))

            ;; uint32
            (:uint32 (type-check t/hjson-uint32-p value "an unsigned 32-bit integer"))
            (:uint32? (type-check t/hjson-uint32?-p value "an unsigned 32-bit integer or NIL"))
            (:array-uint32 (type-check t/hjson-uint32-array-p value "an array of unsigned 32-bit integers"))
            (:array?-uint32 (type-check t/hjson-uint32-array?-p value "an array of unsigned 32-bit integers, or NIL"))
            (:array-uint32? (type-check t/hjson-uint32?-array-p value
                                       "an array containing unsigned 32-bit integers or NIL"))
            (:array?-uint32? (type-check t/hjson-uint32?-array?-p value
                                       "an array containing unsigned 32-bit integers or NIL, or NIL"))

            ;; uint64
            (:uint64 (type-check t/hjson-uint64-p value "an unsigned 64-bit integer"))
            (:uint64? (type-check t/hjson-uint64?-p value "an unsigned 64-bit integer or NIL"))
            (:array-uint64 (type-check t/hjson-uint64-array-p value "an array of unsigned 64-bit integers"))
            (:array?-uint64 (type-check t/hjson-uint64-array?-p value "an array of unsigned 64-bit integers, or NIL"))
            (:array-uint64? (type-check t/hjson-uint64?-array-p value
                                       "an array containing unsigned 64-bit integers or NIL"))
            (:array?-uint64? (type-check t/hjson-uint64?-array?-p value
                                       "an array containing unsigned 64-bit integers or NIL, or NIL"))

            ;; int8
            (:int8 (type-check t/hjson-int8-p value "a signed 8-bit integer"))
            (:int8? (type-check t/hjson-int8?-p value "a signed 8-bit integer or NIL"))
            (:array-int8 (type-check t/hjson-int8-array-p value "an array of signed 8-bit integers"))
            (:array?-int8 (type-check t/hjson-int8-array?-p value "an array of signed 8-bit integers, or NIL"))
            (:array-int8? (type-check t/hjson-int8?-array-p value "an array containing signed 8-bit integers or NIL"))
            (:array?-int8? (type-check t/hjson-int8?-array?-p value
                                       "an array containing signed 8-bit integers or NIL, or NIL"))

            ;; int16
            (:int16 (type-check t/hjson-int16-p value "a signed 16-bit integer"))
            (:int16? (type-check t/hjson-int16?-p value "a signed 16-bit integer or NIL"))
            (:array-int16 (type-check t/hjson-int16-array-p value "an array of signed 16-bit integers"))
            (:array?-int16 (type-check t/hjson-int16-array?-p value "an array of signed 16-bit integers, or NIL"))
            (:array-int16? (type-check t/hjson-int16?-array-p value
                                       "an array containing signed 16-bit integers or NIL"))
            (:array?-int16? (type-check t/hjson-int16?-array?-p value
                                       "an array containing signed 16-bit integers or NIL, or NIL"))

            ;; int24
            (:int24 (type-check t/hjson-int24-p value "a signed 24-bit integer"))
            (:int24? (type-check t/hjson-int24?-p value "a signed 24-bit integer or NIL"))
            (:array-int24 (type-check t/hjson-int24-array-p value "an array of signed 24-bit integers"))
            (:array?-int24 (type-check t/hjson-int24-array?-p value "an array of signed 24-bit integers, or NIL"))
            (:array-int24? (type-check t/hjson-int24?-array-p value
                                       "an array containing signed 24-bit integers or NIL"))
            (:array?-int24? (type-check t/hjson-int24?-array?-p value
                                       "an array containing signed 24-bit integers or NIL, or NIL"))

            ;; int32
            (:int32 (type-check t/hjson-int32-p value "a signed 32-bit integer"))
            (:int32? (type-check t/hjson-int32?-p value "a signed 32-bit integer or NIL"))
            (:array-int32 (type-check t/hjson-int32-array-p value "an array of signed 32-bit integers"))
            (:array?-int32 (type-check t/hjson-int32-array?-p value "an array of signed 32-bit integers, or NIL"))
            (:array-int32? (type-check t/hjson-int32?-array-p value
                                       "an array containing signed 32-bit integers or NIL"))
            (:array?-int32? (type-check t/hjson-int32?-array?-p value
                                       "an array containing signed 32-bit integers or NIL, or NIL"))

            ;; int64
            (:int64 (type-check t/hjson-int64-p value "a signed 64-bit integer"))
            (:int64? (type-check t/hjson-int64?-p value "a signed 64-bit integer or NIL"))
            (:array-int64 (type-check t/hjson-int64-array-p value "an array of signed 64-bit integers"))
            (:array?-int64 (type-check t/hjson-int64-array?-p value "an array of signed 64-bit integers, or NIL"))
            (:array-int64? (type-check t/hjson-int64?-array-p value
                                       "an array containing signed 64-bit integers or NIL"))
            (:array?-int64? (type-check t/hjson-int64?-array?-p value
                                       "an array containing signed 64-bit integers or NIL, or NIL"))
            (otherwise (error "Unknown HJSON unmarshal type: ~a" type))))))

(defun assign-class-value-to-slot (class instance slot value type)
  "Checks to see that TYPE is a class who's metaclass is a subtype of
HJSON-CLASS.  If it is, it creates a new instance of that class, unmarshals
VALUE into it, and then assigns the new instance to SLOT.

This raises an UNMARSHAL-ERROR condition if anything goes wrong."
  (unless (subtypep (find-class 'hjson-class) (class-of type))
    (error "Cannot unmarshal HJSON data to a metaclass not derived from HJSON-CLASS: ~a (~a)"
           type (class-of type)))

  (if (eq (nth 0 (slot-unmarshal-type slot)) :class?)
      (if (typep value 'null)
          (setf (closer-mop:slot-value-using-class class instance slot) nil)
          (setf (closer-mop:slot-value-using-class class instance slot)
                (unmarshal value type)))
      (setf (closer-mop:slot-value-using-class class instance slot)
            (unmarshal value type))))

(defun assign-class-table-value-to-slot (class instance slot value type)
  "Checks to see that TYPE is a class who's metaclass is a subtype of
HJSON-CLASS.  If it is, it creates a hash table containing new instances of that
class, unmarshals VALUE into them, and then assigns the new hash table to SLOT.

This raises an UNMARSHAL-ERROR condition if anything goes wrong."
  (unless (subtypep (find-class 'hjson-class) (class-of type))
    (error "Cannot unmarshal HJSON data to a table of metaclasses not derived from HJSON-CLASS: ~a (~a)"
           type (class-of type)))

  (when (eq (nth 0 (slot-unmarshal-type slot)) :class-table?)
    (when (typep value 'null)
      (setf (closer-mop:slot-value-using-class class instance slot) nil)
      (return-from assign-class-table-value-to-slot)))

  (let ((new-table (make-hash-table :test 'equal)))
    (maphash #'(lambda (k v)
                 (setf (gethash k new-table) (unmarshal v type)))
             value)
    (setf (closer-mop:slot-value-using-class class instance slot) new-table)))

(defun assign-class-array-value-to-slot (class instance slot value type)
  "Checks to see that TYPE is a class who's metaclass is a subtype of
HJSON-CLASS.  If it is, it creates a vector containing new instances of that
class, unmarshals VALUE into them, and then assigns the new hash table to SLOT.

This raises an UNMARSHAL-ERROR condition if anything goes wrong."
  (unless (subtypep (find-class 'hjson-class) (class-of type))
    (error "Cannot unmarshal HJSON data to a table of metaclasses not derived from HJSON-CLASS: ~a (~a)"
           type (class-of type)))

  (when (eq (nth 0 (slot-unmarshal-type slot)) :class-array?)
    (when (typep value 'null)
      (setf (closer-mop:slot-value-using-class class instance slot) nil)
      (return-from assign-class-array-value-to-slot)))

  (let ((new-array (make-array 0 :adjustable t :fill-pointer 0)))
    (dolist (raw-val value)
      (vector-push-extend (unmarshal raw-val type) new-array))

    (setf (closer-mop:slot-value-using-class class instance slot) new-array)))

(defun assign-value-to-slot (class instance slot data)
  "Unmarshals DATA to SLOT."
  (unless (slot-ignore-p slot)

    (cond
      ((slot-unmarshal-fn slot)
       (setf (closer-mop:slot-value-using-class class instance slot)
             (funcall (slot-unmarshal-fn slot) data)))

      ((hjson-slot-needs-class slot)
       (assign-class-value-to-slot class instance slot data (find-class (nth 1 (slot-unmarshal-type slot)))))

      ((hjson-slot-needs-class-array slot)
       (assign-class-array-value-to-slot class instance slot data (find-class (nth 1 (slot-unmarshal-type slot)))))

      ((hjson-slot-needs-class-table slot)
       (assign-class-table-value-to-slot class instance slot data (find-class (nth 1 (slot-unmarshal-type slot)))))

      ((slot-unmarshal-type slot)
       (assign-typed-value-to-slot class instance slot data (slot-unmarshal-type slot)))

      (t (setf (closer-mop:slot-value-using-class class instance slot) data)))))

(defmethod unmarshal ((data hash-table) dest-class &key &allow-other-keys)
  "Unmarshals the hash table DATA that contains parsed HJSON data into a new
instance of DEST-CLASS."
  (let ((class (if (symbolp dest-class)
                   (find-class dest-class)
                   dest-class)))
    (with-slots (non-strict unknown-fields-to-slot) class
      (let ((ret (make-instance class)))
        (prog1 ret
          (maphash
           #'(lambda (key value)
               (let ((slot (find-applicable-slot class key)))
                 (if slot
                     (assign-value-to-slot class ret slot value)
                     (if non-strict
                         (when unknown-fields-to-slot
                           (unless (slot-value ret unknown-fields-to-slot)
                             ;; No unknown fields stored yet.  Create a new table.
                             (setf (slot-value ret unknown-fields-to-slot) (make-hash-table :test 'equal)))

                           (setf (gethash key (slot-value ret unknown-fields-to-slot)) value))

                         (error 'unmarshal-error
                                :format-control "Unrecognized HJSON key: ~s"
                                :format-arguments (list key))))))
           data))))))

(defmethod unmarshal ((data string) dest-class &key &allow-other-keys)
  "Parses DATA using PARSE-HJSON, then unmarshalls the result into a new
instance of DEST-CLASS."
  (unmarshal (parse-hjson data) dest-class))

(defmethod unmarshal ((data pathname) dest-class &key &allow-other-keys)
  "Opens and reads from DATA using PARSE-HJSON, then unmarshalls the result into
a new instance of DEST-CLASS."
  (unmarshal (parse-hjson data) dest-class))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun marshal-simple-hjson-slot (obj slot dest)
  (let* ((class (class-of obj))
         (key (slot-key-name slot (hjson-class-translate-hyphens-p class)))
         (value (closer-mop:slot-value-using-class class obj slot)))
    (macrolet ((type-check (check-fn value-var fancy-name)
                 `(if (,check-fn ,value-var)
                      (setf (gethash key dest) ,value-var)
                      (error 'marshal-error
                             :format-control "The slot ~s requires ~a"
                             :format-arguments (list (slot-key-name slot (hjson-class-translate-hyphens-p class))
                                                     ,fancy-name)))))

      (case (slot-unmarshal-type slot)
        ;; Strings
        (:string (type-check t/hjson-string-p value "a string"))
        (:string? (type-check t/hjson-string?-p value "a string or NIL"))
        (:array-string (type-check t/hjson-string-array-p value "an array of strings"))
        (:array?-string (type-check t/hjson-string-array?-p value "an array of strings, or NIL"))
        (:array-string? (type-check t/hjson-string?-array-p value "an array containing strings or NIL"))
        (:array?-string? (type-check t/hjson-string?-array?-p value "an array containing strings or NIL, or NIL"))

        ;; Bools
        (:bool (type-check t/hjson-bool-p value "a boolean"))
        (:bool? (type-check t/hjson-bool?-p value "a boolean or NIL"))
        (:array-bool (type-check t/hjson-bool-array-p value "an array of booleans"))
        (:array?-bool (type-check t/hjson-bool-array?-p value "an array of booleans, or NIL"))
        (:array-bool? (type-check t/hjson-bool?-array-p value "an array containing booleans or NIL"))
        (:array?-bool? (type-check t/hjson-bool?-array?-p value "an array containing booleans or NIL, or NIL"))

        ;; uint8
        (:uint8 (type-check t/hjson-uint8-p value "an unsigned 8-bit integer"))
        (:uint8? (type-check t/hjson-uint8?-p value "an unsigned 8-bit integer or NIL"))
        (:array-uint8 (type-check t/hjson-uint8-array-p value "an array of unsigned 8-bit integers"))
        (:array?-uint8 (type-check t/hjson-uint8-array?-p value "an array of unsigned 8-bit integers, or NIL"))
        (:array-uint8? (type-check t/hjson-uint8?-array-p value "an array containing unsigned 8-bit integers or NIL"))
        (:array?-uint8? (type-check t/hjson-uint8?-array?-p value
                                   "an array containing unsigned 8-bit integers or NIL, or NIL"))

        ;; uint16
        (:uint16 (type-check t/hjson-uint16-p value "an unsigned 16-bit integer"))
        (:uint16? (type-check t/hjson-uint16?-p value "an unsigned 16-bit integer or NIL"))
        (:array-uint16 (type-check t/hjson-uint16-array-p value "an array of unsigned 16-bit integers"))
        (:array?-uint16 (type-check t/hjson-uint16-array?-p value "an array of unsigned 16-bit integers, or NIL"))
        (:array-uint16? (type-check t/hjson-uint16?-array-p value
                                   "an array containing unsigned 16-bit integers or NIL"))
        (:array?-uint16? (type-check t/hjson-uint16?-array?-p value
                                    "an array containing unsigned 16-bit integers or NIL, or NIL"))

        ;; uint32
        (:uint32 (type-check t/hjson-uint32-p value "an unsigned 32-bit integer"))
        (:uint32? (type-check t/hjson-uint32?-p value "an unsigned 32-bit integer or NIL"))
        (:array-uint32 (type-check t/hjson-uint32-array-p value "an array of unsigned 32-bit integers"))
        (:array?-uint32 (type-check t/hjson-uint32-array?-p value "an array of unsigned 32-bit integers, or NIL"))
        (:array-uint32? (type-check t/hjson-uint32?-array-p value
                                   "an array containing unsigned 32-bit integers or NIL"))
        (:array?-uint32? (type-check t/hjson-uint32?-array?-p value
                                    "an array containing unsigned 32-bit integers or NIL, or NIL"))

        ;; uint64
        (:uint64 (type-check t/hjson-uint64-p value "an unsigned 64-bit integer"))
        (:uint64? (type-check t/hjson-uint64?-p value "an unsigned 64-bit integer or NIL"))
        (:array-uint64 (type-check t/hjson-uint64-array-p value "an array of unsigned 64-bit integers"))
        (:array?-uint64 (type-check t/hjson-uint64-array?-p value "an array of unsigned 64-bit integers, or NIL"))
        (:array-uint64? (type-check t/hjson-uint64?-array-p value
                                   "an array containing unsigned 64-bit integers or NIL"))
        (:array?-uint64? (type-check t/hjson-uint64?-array?-p value
                                    "an array containing unsigned 64-bit integers or NIL, or NIL"))

        ;; int8
        (:int8 (type-check t/hjson-int8-p value "a signed 8-bit integer"))
        (:int8? (type-check t/hjson-int8?-p value "a signed 8-bit integer or NIL"))
        (:array-int8 (type-check t/hjson-int8-array-p value "an array of signed 8-bit integers"))
        (:array?-int8 (type-check t/hjson-int8-array?-p value "an array of signed 8-bit integers, or NIL"))
        (:array-int8? (type-check t/hjson-int8?-array-p value "an array containing signed 8-bit integers or NIL"))
        (:array?-int8? (type-check t/hjson-int8?-array?-p value
                                  "an array containing signed 8-bit integers or NIL, or NIL"))

        ;; int16
        (:int16 (type-check t/hjson-int16-p value "a signed 16-bit integer"))
        (:int16? (type-check t/hjson-int16?-p value "a signed 16-bit integer or NIL"))
        (:array-int16 (type-check t/hjson-int16-array-p value "an array of signed 16-bit integers"))
        (:array?-int16 (type-check t/hjson-int16-array?-p value "an array of signed 16-bit integers, or NIL"))
        (:array-int16? (type-check t/hjson-int16?-array-p value
                                  "an array containing signed 16-bit integers or NIL"))
        (:array?-int16? (type-check t/hjson-int16?-array?-p value
                                   "an array containing signed 16-bit integers or NIL, or NIL"))

        ;; int32
        (:int32 (type-check t/hjson-int32-p value "a signed 32-bit integer"))
        (:int32? (type-check t/hjson-int32?-p value "a signed 32-bit integer or NIL"))
        (:array-int32 (type-check t/hjson-int32-array-p value "an array of signed 32-bit integers"))
        (:array?-int32 (type-check t/hjson-int32-array?-p value "an array of signed 32-bit integers, or NIL"))
        (:array-int32? (type-check t/hjson-int32?-array-p value
                                  "an array containing signed 32-bit integers or NIL"))
        (:array?-int32? (type-check t/hjson-int32?-array?-p value
                                   "an array containing signed 32-bit integers or NIL, or NIL"))

        ;; int64
        (:int64 (type-check t/hjson-int64-p value "a signed 64-bit integer"))
        (:int64? (type-check t/hjson-int64?-p value "a signed 64-bit integer or NIL"))
        (:array-int64 (type-check t/hjson-int64-array-p value "an array of signed 64-bit integers"))
        (:array?-int64 (type-check t/hjson-int64-array?-p value "an array of signed 64-bit integers, or NIL"))
        (:array-int64? (type-check t/hjson-int64?-array-p value
                                  "an array containing signed 64-bit integers or NIL"))
        (:array?-int64? (type-check t/hjson-int64?-array?-p value
                                   "an array containing signed 64-bit integers or NIL, or NIL"))
        (otherwise (error "Don't know how to marshal HJSON type: ~a" (slot-unmarshal-type slot)))))))

(defun marshal-hjson-slot (class obj slot non-strict dest)
  (let ((key (slot-key-name slot (hjson-class-translate-hyphens-p class)))
        (value (closer-mop:slot-value-using-class class obj slot))
        (emit-default? (not (slot-dont-emit-default-val-p slot))))
    (when (or emit-default? (not (equalp (closer-mop:slot-definition-initform slot) value)))
      (cond
        ((slot-marshal-fn slot)
         (setf (gethash key dest) (funcall (slot-marshal-fn slot) value)))

        ((or (hjson-slot-needs-class slot)
             (hjson-slot-needs-class-array slot)
             (hjson-slot-needs-class-table slot))
         (setf (gethash key dest) (marshal-to-table value)))

        ((slot-unmarshal-type slot)
         (marshal-simple-hjson-slot obj slot dest))

        (t (unless non-strict
             (error "Don't know how to marshal slot ~a to HJSON" (closer-mop:slot-definition-name slot))))))))

(defun marshal-to-table (obj)
  "Marshals OBJ to a string."
  (let ((class (class-of obj))
        (ret (make-hash-table :test 'equal)))
    (with-slots (non-strict-marshalling) class
      (dolist (slot (closer-mop:class-slots class))
        (unless (slot-ignore-p slot)
          (marshal-hjson-slot class obj slot non-strict-marshalling ret))))
    ret))

(defun marshal (obj)
  "Marshals OBJ to a HASH-TABLE containing HJSON data."
  (marshal-to-table obj))
