;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;; HJSON code based on hjson-go, Copyright (C) 2016, 2017 Christian Zangl
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm-hjson)

;;;;
;;;; HJSON Encoding
;;;;

(defstruct (encoder (:copier nil))
  (eol-str               (string #\Newline) :type simple-string)
  (indent-string         "    "             :type simple-string)
  (open-braces-same-line t                  :type boolean)
  (always-quote          nil                :type boolean)
  (unknowns-as-null      nil                :type boolean)
  (indent                0                  :type fixnum)
  (out (make-string-output-stream)          :type string-stream))

(defconst +meta+
  '((#\Bel     . "\\b")
    (#\Tab     . "\\t")
    (#\Newline . "\\n")
    (#\Page    . "\\f")
    (#\Return  . "\\r")
    (#\"       . "\\\"")
    (#\\       . "\\\\"))
  :documentation  "Reverse character->escaped string lookup")

(define-typed-fn needs-quoting-p ((simple-string str))
    (boolean t)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (loop for c character across str
        when (or (assoc c +meta+ :test #'char=)
                 (char= c #\Space)
                 (char= c #\{)
                 (char= c #\})
                 (char= c #\[)
                 (char= c #\])
                 (char= c #\,)
                 (char= c #\:))
          do (return-from needs-quoting-p t))
  nil)

(define-typed-fn quote-key ((simple-string key) &optional always?)
    (simple-string t)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (if (or (needs-quoting-p key) always?)
      (format nil "~s" key)
      key))

(define-typed-fn encoder-write-indent ((encoder enc) (fixnum count))
    (null t)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (cl:write-string (encoder-eol-str enc) (encoder-out enc))
  (dotimes (i count)
    (cl:write-string (encoder-indent-string enc) (encoder-out enc))))

(defmethod encoder-ml-string ((enc encoder) (val string) (sep string))
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))

  (let ((a (sdm:split-string (sdm:string-replace val (string #\Return) "") #\Newline)))
    (declare (type vector a))
    (cond
      ;; The string contains only a single line. We still use the
      ;; multiline format as it avoids escaping the \ character
      ;; (e.g. when used in a regex).
      ((= (length a) 1)
       (format (encoder-out enc) "~a'''~a" sep (elt a 0)))

      (t
       (encoder-write-indent enc (1+ (encoder-indent enc)))
       (cl:write-string "'''" (encoder-out enc))

       (loop for v string across a do
         (let ((indent (1+ (encoder-indent enc))))
           (when (= (length v) 0)
             (setf indent 0))

           (encoder-write-indent enc indent)
           (cl:write-string v (encoder-out enc))))

       (encoder-write-indent enc (1+ (encoder-indent enc)))))

    (cl:write-string "'''" (encoder-out enc))))

(defmethod encoder-str ((enc encoder) (data hash-table) no-indent (sep string) is-root-obj)
  (declare (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))

  (when (= (hash-table-count data) 0)
    (format (encoder-out enc) "~a{}" sep)
    (return-from encoder-str))

  (let ((indent (encoder-indent enc)))
    (incf (encoder-indent enc))

    (when is-root-obj
      (write-char #\{ (encoder-out enc)))
    (maphash #'(lambda (key val)
                 (encoder-write-indent enc (encoder-indent enc))

                 (cl:write-string (quote-key (if (stringp key)
                                                 key
                                                 (write-to-string key))
                                             (encoder-always-quote enc))
                                  (encoder-out enc))

                 (cl:write-string ": " (encoder-out enc))

                 (cond
                   ((hash-table-p val)
                    (unless (encoder-open-braces-same-line enc)
                      (write-char #\Newline (encoder-out enc))
                      (encoder-write-indent enc (1+ indent)))

                    (write-char #\{ (encoder-out enc))
                    (encoder-str enc val no-indent sep nil))

                   (t
                    (cl:write-string (cond
                                       ((stringp val) val)
                                       ((eq :false val) "false")
                                       ((eq :true val) "true")
                                       ((null val) "null")
                                       ((floatp val)
                                        (format nil "~f" val))
                                       (t (write-to-string val)))
                                     (encoder-out enc)))))
             data))

  (when is-root-obj
    (write-char #\newline (encoder-out enc))
    (write-char #\} (encoder-out enc))))

(defmethod encoder-str ((enc encoder) val no-indent (sep string) is-root-obj)
  (declare (ignore no-indent is-root-obj)
           (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))
  (encoder-error "Cannot marshal object of type ~a to HJSON" (type-of val)))

(defun write-hjson (data &key (eol #\Newline) (indent-string "  ") always-quote (open-braces-same-line t)
                           unknowns-as-null)
  "Marshals DATA into a string containing HJSON data.  Newlines in the resulting
string will be generated according to EOL.  This can be a character or string.

INDENT-STRING is the string used for indentation.

When ALWAYS-QUOTE is non-NIL, key names are always quoted.  Otherwise they are
only quoted as-needed.

When OPEN-BRACES-SAME-LINE is non-NIL, the opening brace is placed on the same
line as the key.  Otherwise it is placed on the next line.

If UNKNOWNS-AS-NULL is non-NIL, unknown data types are encoded to null.
Otherwise a restartable ENCODING-ERROR is raised."
  (declare (type (or simple-string character) eol)
           (type simple-string indent-string)
           (optimize (speed 3) (debug 1) (safety 0) (compilation-speed 0)))

  (let ((enc (make-encoder :eol-str (locally
                                        (declare #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
                                      (string eol))
                           :indent-string indent-string
                           :always-quote (and always-quote)
                           :open-braces-same-line (and open-braces-same-line)
                           :unknowns-as-null (and unknowns-as-null))))

    (encoder-str enc data t "" t)
    (get-output-stream-string (encoder-out enc))))
