;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm)

;;;;
;;;; IEEE 754 Float Conversions
;;;;
;;;; These basically convert SINGLE-FLOATS and DOUBLE-FLOATS to/from integers
;;;; using SCALE-FLOAT and DECODE-FLOAT, respectively, together with some bit
;;;; fiddling.
;;;;

(eval-when (:compile-toplevel :load-toplevel :execute)
  (defining-consts
    (+dfloat-sig-bits+ 53 :documentation "Number of significant digits for a 64-bit IEEE 754 float.")
    (+dfloat-exp-bits+ 11 :documentation "Size of the exponent in bits for a 64-bit IEEE 754 float.")
    (+sfloat-sig-bits+ 24 :documentation "Number of significant digits for a 32-bit IEEE 754 float.")
    (+sfloat-exp-bits+  8 :documentation "Size of the exponent in bits for a 32-bit IEEE 754 float."))

  (deftype t/special-floats ()
    "Equivalents for the special IEEE 754 values NaN, +Inf, and -Inf."
    '(member :not-a-number :positive-infinity :negative-infinity)))

(defmacro %define-float->int-conv-fn (name type ret-type init-literal exp-bits-const sig-bits-const
                                      sign-bit exp-start exp-size sig-size documentation)
  `(define-typed-fn ,name (((or ,type t/special-floats) value))
       (,ret-type t)
     ,documentation
     (declare (optimize speed (debug 1) (safety 1))
              #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
     (let ((%sig ,init-literal)
           (%ieee-sig 0)
           (%exp 0)
           (%sign ,init-literal)
           (%ieee-sign 0))
       ;; *sigh* ECL pisses me off
       (declare (type fixnum %exp %ieee-sign %ieee-sig)
                (type #+ecl (or double-float single-float fixnum)
                      #-ecl (or ,type fixnum) %sig %sign))

       (typecase value
         (,type
          ;; Decode the float using DECODE-FLOAT.  This gives us a floating point
          ;; significand, an exponent, and a sign.  These are different from IEEE 754
          ;; floating point format.
          (multiple-value-setq (%sig %exp %sign)
            (decode-float value))

          ;; Adjust the sign into a form used by IEEE 754.  It's essentially opposite.
          (setf %ieee-sign (if (= %sign 1) 0 1))

          ;; Adjust the exponent into a form used by IEEE 754.
          (setf %exp (if (zerop %sig)
                         %exp
                         (+ (1- %exp)
                            ,(1- (expt 2 (1- exp-bits-const))))))

          ;; Adjust the significand into a form used by IEEE 754.
          (muffling
            (setf %ieee-sig (if (plusp %exp)
                                (the t/int64 (round (* (1- (* 2 %sig))
                                                       ,(expt 2 (1- sig-bits-const)))))
                                (prog1 (ash (round (* %sig ,(expt 2 (1- sig-bits-const))))
                                            %exp)
                                  (setf %exp 0))))))
         (otherwise
          (ecase value
            (:not-a-number
             (setf %sign 0)
             (setf %sig 1)
             (setf %exp ,(1- (expt 2 exp-bits-const))))

            (:positive-infinity
             (setf %sign 0)
             (setf %sig 0)
             (setf %exp ,(1- (expt 2 exp-bits-const))))

            (:negative-infinity
             (setf %sign 1)
             (setf %sig 0)
             (setf %exp ,(1- (expt 2 exp-bits-const)))))))

       ;; Bit fiddling time.
       (let ((ret 0))
         (declare (type t/uint64 ret))
         ;; Bits indexed from 0.
         (setf ret (dpb %ieee-sign (byte 1 ,sign-bit) ret)) ;; Set the sign
         (setf ret (dpb %exp (byte ,exp-size ,exp-start) ret)) ;; Set the exponent
         (dpb %ieee-sig (byte ,sig-size 0) ret))))) ;; Set the significand, and return

(defmacro %define-int->float-conv-fn (name type ret-type sig-bits-const exp-bits-const
                                      sign-bit exp-start exp-size sig-size
                                      zero neg-zero
                                      documentation)
  `(define-typed-fn ,name ((,type value))
       ((or ,ret-type t/special-floats) t)
     ,documentation
     (declare (optimize speed (debug 1) (safety 1))
              #+sbcl (sb-ext:muffle-conditions sb-ext:compiler-note))
     ;; Retrieve the pieces from the bits of VALUE.
     (let ((sign (ldb (byte         1  ,sign-bit) value))
           (exp  (ldb (byte ,exp-size ,exp-start) value))
           (sig  (ldb (byte ,sig-size          0) value))
           (flt-sig ,(ecase ret-type
                       (double-float 0.0d0)
                       (single-float 0.0))))
       (declare (type (integer 0 ,(expt 2 (1+ exp-bits-const))) exp)
                (type (integer 0 ,(expt 2 sig-bits-const)) sig)
                (type ,ret-type flt-sig))

       ;; Check for NaN, +Inf, -Inf, 0, and -0
       (cond
         ((and (= sign 0)
               (= exp ,(1- (expt 2 exp-bits-const)))
               (= sig 0))
          (return-from ,name :positive-infinity))

         ((and (= sign 1)
               (= exp ,(1- (expt 2 exp-bits-const)))
               (= sig 0))
          (return-from ,name :negative-infinity))

         ((and (= sign 0)
               (= exp 0)
               (= sig 0))
          (return-from ,name ,zero))

         ((and (= sign 1)
               (= exp 0)
               (= sig 0))
          (return-from ,name ,neg-zero))

         ((and (= exp ,(1- (expt 2 exp-bits-const)))
               (not (zerop sig)))
          (return-from ,name :not-a-number)))

       ;; Adjust a few values.
       (if (zerop exp)
           (setf exp 1)
           (setf sig (dpb 1 (byte 1 ,(1- sig-bits-const)) sig)))

       ;; Create the float
       (setf flt-sig (float sig ,(ecase ret-type
                                   (double-float 1.0d0)
                                   (single-float 1.0))))
       (scale-float (if (zerop sign)
                        flt-sig
                        (- flt-sig))
                    (- exp ,(+ (1- (expt 2 (1- exp-bits-const)))
                               (1- sig-bits-const)))))))

(%define-float->int-conv-fn dfloat->uint64 double-float t/uint64 0.0d0
                            #.+dfloat-exp-bits+ #.+dfloat-sig-bits+
                            63
                            52 11
                            52
                            "Converts a DOUBLE-FLOAT into an IEEE 754 64-bit float, encoded as a T/UINT64.

This can also take any of the T/SPECIAL-FLOATS values as an argument.")

(%define-float->int-conv-fn sfloat->uint32 single-float t/uint32 0.0
                            #.+sfloat-exp-bits+ #.+sfloat-sig-bits+
                            31
                            23 8
                            23
                            "Converts a SINGLE-FLOAT into an IEEE 754 64-bit float, encoded as a T/UINT32.

This can also take any of the T/SPECIAL-FLOATS values as an argument.")

(%define-int->float-conv-fn uint64->dfloat t/uint64 double-float
                            #.+dfloat-sig-bits+ #.+dfloat-exp-bits+
                            63
                            52 11
                            52
                            0.0d0
                            -0.0d0
                            "Converts a 64-bit IEEE 754 float encoded as a T/UINT64 into a DOUBLE-FLOAT or a
T/SPECIAL-FLOATS.")

(%define-int->float-conv-fn uint32->sfloat t/uint32 single-float
                            #.+sfloat-sig-bits+ #.+sfloat-exp-bits+
                            31
                            23 8
                            23
                            0.0
                            -0.0
                            "Converts a 32-bit IEEE 754 float encoded as a T/UINT32 into a SINGLE-FLOAT or a
T/SPECIAL-FLOATS.")
