;;;; CL-SDM - Opinionated Extra Batteries for Common Lisp
;;;; Copyright (C) 2021-2025 Remilia Scarlet <remilia@posteo.jp>
;;;; Copyright 2012-2025 Manas Technology Solutions
;;;;
;;;; This program is free software: you can redistribute it and/or modify it
;;;; under the terms of the GNU Affero General Public License as published by
;;;; the Free Software Foundation, either version 3 of the License, or (at your
;;;; option) any later version.
;;;;
;;;; This program is distributed in the hope that it will be useful, but WITHOUT
;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Affero General Public
;;;; License for more details.
;;;;
;;;; You should have received a copy of the GNU Affero General Public License
;;;; along with this program.  If not, see <https://www.gnu.org/licenses/>.
(in-package :cl-sdm/file)

;;;;
;;;; File globbing
;;;;
;;;; Ported from the Crystal standard library v1.12.2 by Remilia Scarlet
;;;; https://crystal-lang.org/
;;;;

(defining-consts
  (+match-separators+ #+unix '(#\/)
                      #+windows '(#\\ #\/)))

(define-simple-error file-pattern-error nil ())

(defgeneric file-matches-p (pattern path))

(define-typed-fn expand-brace-pattern ((string pattern) ((or null (vector string)) expanded-patterns))
    ((vector string))
  (macrolet
      ((byte-slice (string start &optional count)
         `(make-array (or ,count (- (length ,string) ,start))
                      :element-type 'character
                      :displaced-to ,string
                      :displaced-index-offset ,start)))
    (let ((expanded (or expanded-patterns (new-vector string)))
          (lbrace nil)
          (rbrace nil)
          (alt-start nil)
          (start nil)
          (alternatives (new-vector (vector string)))
          (nest 0)
          (escaped? nil)
          (char #\Nul))
      (declare (type (or null fixnum) lbrace rbrace alt-start start)
               (type fixnum nest)
               (type boolean escaped?)
               ((vector string) expanded)
               (type character char)
               (type (vector (vector string)) alternatives))

      (loop for pos fixnum from 0
            while (< pos (length pattern))
            do (setf char (elt pattern pos))
               (cond
                 ((and (char= char #\{)
                       (not escaped?))
                  (when (zerop nest)
                    (setf lbrace pos))
                  (incf nest))

                 ((and (char= char #\})
                       (not escaped?))
                  (decf nest)
                  (when (zerop nest)
                    (setf rbrace  pos)
                    (setf start (1+ (or alt-start lbrace)))
                    (vector-push-extend (byte-slice pattern start (- pos start))
                                        alternatives)
                    (loop-finish)))

                 ((and (char= char #\,)
                       (not escaped?))
                  (when (= nest 1)
                    (setf start (1+ (or alt-start lbrace)))
                    (vector-push-extend (byte-slice pattern start (- pos start))
                                        alternatives)
                    (setf alt-start pos)))

                 ((and (char= char #\\)
                       (not escaped?))
                  (setf escaped? t))

                 (t
                  (setf escaped? nil))))

      (if (and lbrace rbrace)
          (let ((front (byte-slice pattern 0 lbrace))
                (back (byte-slice pattern (1+ rbrace))))
            (loop for alt across alternatives
                  ;; Remi: The coercsion is needed to ensure we have the proper type of CHARACTER.
                  for brace-pattern = (coerce (format nil "~a~a~a" front alt back) '(vector character))
                  do (expand-brace-pattern brace-pattern expanded)))
          (vector-push-extend pattern expanded))

      expanded)))

(defmethod file-matches-p ((pattern string) (path string))
  (labels
      ((separator-p (char)
         (when char (find char +match-separators+ :test #'char=)))

       (match-single-pattern (pattern path)
         ;; linear-time algorithm adapted from https://research.swtch.com/glob
         (macrolet
             ;;
             ;; These are all purposely unhygenic.
             ;;
             ((p-next-char ()
                `(progn
                   (incf ppos)
                   (if (< ppos patlen)
                       (elt pattern ppos)
                       nil)))

              (p-cur-char ()
                `(if (< ppos patlen)
                     (elt pattern ppos)
                     nil))

              (p-have-next? ()
                `(< ppos patlen))

              (p-peek ()
                `(if (< (1+ ppos) patlen)
                     (elt pattern (1+ ppos))
                     nil))

              (s-have-next? ()
                `(< spos strlen))

              (s-cur-char ()
                `(if (< spos strlen)
                     (elt path spos)
                     nil))

              (s-next-char ()
                `(progn
                   (incf spos)
                   (if (< spos strlen)
                       (elt path spos)
                       nil))))

           (let ((ppos 0)
                 (spos 0)
                 (pnext nil)
                 (snext nil)
                 (pchar nil)
                 (char nil)
                 (next-ppos 0)
                 (next-spos 0)
                 (patlen (length pattern))
                 (strlen (length path))
                 (escaped? nil))
             (declare (type fixnum next-ppos next-spos strlen)
                      (type boolean pnext snext escaped?)
                      (type (or null character) pchar char))

             (loop do
               (tagbody
                  (setf pnext (p-have-next?))
                  (setf snext (s-have-next?))

                  (unless (or pnext snext)
                    (return-from match-single-pattern t))

                  (when pnext
                    (setf pchar (p-cur-char))
                    (setf char (s-cur-char))

                    (cond
                      ((and (char= pchar #\\)
                            (not escaped?))
                       (setf escaped? t)
                       (incf ppos)
                       (go ==next==))

                      ((and (char= pchar #\?)
                            (not escaped?))
                       (when (and snext (not (separator-p char)))
                         (incf ppos)
                         (incf spos)
                         (go ==next==)))

                      ((and (char= pchar #\*)
                            (not escaped?))
                       (let ((double-star? (and (< (1+ ppos) patlen)
                                                (char= (elt pattern (1+ ppos)) #\*))))
                         (cond
                           ((and (separator-p char)
                                 (not double-star?))
                            (incf ppos)
                            (setf next-spos 0)
                            (go ==next==))

                           (t
                            (setf next-ppos ppos)
                            (setf next-spos (1+ spos))
                            (incf ppos)
                            (when double-star?
                              (incf ppos))
                            (go ==next==)))))

                      ((and (char= pchar #\[)
                            (not escaped?))
                       (setf pnext (p-have-next?)) ;; Remi: needless?

                       (let ((character-matched? nil)
                             (character-set-open? t)
                             (inverted? nil))
                         (declare (type boolean character-matched? character-set-open? inverted?))
                         (setf escaped? nil)

                         (case (p-peek)
                           (#\^
                            (setf inverted? t)
                            (incf ppos))
                           (#\]
                            (file-pattern-error () "Invalid character set: empty character set"))
                           (otherwise
                            ;; Nothing
                            ;; TODO: check if this branch is fine
                            ))

                         (loop while pnext do
                           (setf pchar (p-next-char))
                           (cond
                             ((and (equal pchar #\\)
                                   (not escaped?))
                              (setf escaped? t))
                             ((and (equal pchar #\])
                                   (not escaped?))
                              (setf character-set-open? nil)
                              (loop-finish))
                             ((and (equal pchar #\-)
                                   (not escaped?))
                              (file-pattern-error () "Invalid character set: missing range start"))

                             (t
                              (setf escaped? nil)
                              (cond
                                ((and (p-have-next?) (equal (p-peek) #\-))
                                 (incf ppos)
                                 (let ((range-end (p-next-char))
                                       (range-low -1)
                                       (range-high -1))
                                   (case range-end
                                     (#\]
                                      (file-pattern-error () "Invalid character set: missing end range"))
                                     (#\\
                                      (setf range-end (p-next-char)))
                                     (otherwise
                                      ;; Nothing
                                      ;; TODO: check if this branch is fine
                                      ))
                                   (setf range-low (char-code pchar))
                                   (setf range-high (char-code range-end))
                                   (when (and char
                                              (>= (char-code char) range-low)
                                              (<= (char-code char) range-high))
                                     (setf character-matched? t))))
                                ((equal char pchar)
                                 (setf character-matched? t)))))

                           (setf pnext (p-have-next?)))

                         (when character-set-open?
                           (file-pattern-error () "Invalid character set: unterminated character set"))

                         (when (and (not (eq character-matched? inverted?))
                                    snext)
                           (incf ppos)
                           (incf spos)
                           (go ==next==))))

                      (t
                       (setf escaped? nil)
                       (when (and snext (equal (s-cur-char) pchar))
                         (incf ppos)
                         (incf spos)
                         (go ==next==)))))


                  (when (and (< 0 next-spos)
                             (<= next-spos strlen))
                    (setf ppos next-ppos)
                    (setf spos next-spos)
                    (go ==next==))

                  (when escaped?
                    (file-pattern-error () "Empty escape character"))

                  (return-from match-single-pattern)

                ==next==))))))
    (declare (ftype (function (string string) boolean) match-single-pattern)
             (ftype (function ((or null character)) (or null character)) separator-p)
             (inline match-single-pattern))

    (let ((expanded-patterns (expand-brace-pattern (coerce pattern '(vector character)) nil)))
      (loop for pattern across expanded-patterns
            when (match-single-pattern pattern path)
              do (return-from file-matches-p t))))
  nil)

(defmethod file-matches-p ((pattern string) (path pathname))
  (file-matches-p pattern (uiop:native-namestring path)))
